<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/PngHandler.php';

// Require login
Auth::requireLogin();

// Get pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 10;

// Get paginated images
$handler = new PngHandler();
$pagination = $handler->getPaginatedImages($page, $perPage);
$images = $pagination['images'];

// Get all images for stats
$allImages = $handler->getAllImages();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PNG Gallery - Private</title>
    
    <!-- SweetAlert2 -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --bg-primary: #f5f5f5;
            --bg-secondary: #ffffff;
            --bg-card: #ffffff;
            --text-primary: #333;
            --text-secondary: #666;
            --text-tertiary: #999;
            --border-color: #e0e0e0;
            --shadow: rgba(0,0,0,0.1);
            --shadow-hover: rgba(0,0,0,0.15);
            --gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            --checkerboard-light: #f0f0f0;
            --checkerboard-dark: transparent;
        }
        
        [data-theme="dark"] {
            --bg-primary: #1a1a1a;
            --bg-secondary: #2d2d2d;
            --bg-card: #2d2d2d;
            --text-primary: #e0e0e0;
            --text-secondary: #b0b0b0;
            --text-tertiary: #808080;
            --border-color: #404040;
            --shadow: rgba(0,0,0,0.3);
            --shadow-hover: rgba(0,0,0,0.5);
            --checkerboard-light: #808080;
            --checkerboard-dark: #6a6a6a;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background: var(--bg-primary);
            color: var(--text-primary);
            transition: background 0.3s, color 0.3s;
        }
        .header {
            background: var(--bg-secondary);
            box-shadow: 0 2px 4px var(--shadow);
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
            transition: background 0.3s;
        }
        .header h1 {
            font-size: 24px;
            background: var(--gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        .dark-mode-toggle {
            background: var(--bg-card);
            border: 2px solid var(--border-color);
            border-radius: 20px;
            padding: 5px 10px;
            cursor: pointer;
            font-size: 20px;
            transition: all 0.3s;
        }
        .dark-mode-toggle:hover {
            transform: scale(1.1);
        }
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            transition: all 0.3s;
            cursor: pointer;
            border: none;
            font-size: 14px;
        }
        .btn-primary {
            background: var(--gradient);
            color: white;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        .btn-secondary {
            background: var(--bg-card);
            color: var(--text-secondary);
            border: 2px solid var(--border-color);
        }
        .btn-secondary:hover {
            background: var(--bg-primary);
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 30px;
        }
        .stats {
            background: var(--bg-card);
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
            display: flex;
            gap: 30px;
            box-shadow: 0 2px 4px var(--shadow);
            transition: background 0.3s;
        }
        .stat-item {
            flex: 1;
        }
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            background: var(--gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 5px;
        }
        .stat-label {
            color: var(--text-secondary);
            font-size: 14px;
        }
        .gallery-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }
        .gallery-item {
            background: var(--bg-card);
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 8px var(--shadow);
            transition: all 0.3s;
            cursor: pointer;
        }
        .gallery-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px var(--shadow-hover);
        }
        .image-container {
            width: 100%;
            height: 250px;
            background: 
                linear-gradient(45deg, var(--checkerboard-light) 25%, var(--checkerboard-dark) 25%),
                linear-gradient(-45deg, var(--checkerboard-light) 25%, var(--checkerboard-dark) 25%),
                linear-gradient(45deg, var(--checkerboard-dark) 75%, var(--checkerboard-light) 75%),
                linear-gradient(-45deg, var(--checkerboard-dark) 75%, var(--checkerboard-light) 75%);
            background-size: 20px 20px;
            background-position: 0 0, 0 10px, 10px -10px, -10px 0px;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
        }
        .image-container img {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            transition: transform 0.3s;
        }
        .gallery-item:hover .image-container img {
            transform: scale(1.05);
        }
        .image-info {
            padding: 15px;
        }
        .image-filename {
            font-weight: 600;
            margin-bottom: 8px;
            font-size: 14px;
            word-break: break-all;
            color: var(--text-primary);
        }
        .image-meta {
            display: flex;
            gap: 15px;
            font-size: 12px;
            color: var(--text-tertiary);
            margin-bottom: 10px;
        }
        .image-actions {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
        .btn-small {
            padding: 6px 12px;
            font-size: 12px;
            border-radius: 6px;
            text-decoration: none;
            display: inline-block;
            border: none;
            cursor: pointer;
            transition: all 0.3s;
        }
        .btn-view {
            background: #667eea;
            color: white;
        }
        .btn-download {
            background: #48c774;
            color: white;
        }
        .btn-rename {
            background: #ffa500;
            color: white;
        }
        .btn-delete {
            background: #f14668;
            color: white;
        }
        .btn-small:hover {
            opacity: 0.8;
            transform: translateY(-1px);
        }
        .empty-state {
            text-align: center;
            padding: 80px 20px;
            background: var(--bg-card);
            border-radius: 12px;
        }
        .empty-state svg {
            width: 120px;
            height: 120px;
            margin-bottom: 20px;
            opacity: 0.3;
            stroke: var(--text-tertiary);
        }
        .empty-state h2 {
            margin-bottom: 10px;
            color: var(--text-secondary);
        }
        .empty-state p {
            color: var(--text-tertiary);
            margin-bottom: 20px;
        }
        .badge {
            display: inline-block;
            padding: 3px 8px;
            background: #48c774;
            color: white;
            border-radius: 4px;
            font-size: 10px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
            margin-top: 30px;
            padding: 20px;
            background: var(--bg-card);
            border-radius: 12px;
        }
        .pagination a,
        .pagination span {
            padding: 8px 15px;
            border-radius: 6px;
            text-decoration: none;
            color: var(--text-primary);
            background: var(--bg-primary);
            border: 2px solid var(--border-color);
            transition: all 0.3s;
        }
        .pagination a:hover {
            background: var(--gradient);
            color: white;
            border-color: transparent;
        }
        .pagination .active {
            background: var(--gradient);
            color: white;
            border-color: transparent;
        }
        .pagination .disabled {
            opacity: 0.5;
            cursor: not-allowed;
            pointer-events: none;
        }
        
        /* Lightbox */
        .lightbox {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.95);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }
        .lightbox.active {
            display: flex;
        }
        .lightbox-content {
            max-width: 90%;
            max-height: 90%;
            position: relative;
        }
        .lightbox-image {
            max-width: 100%;
            max-height: 90vh;
            object-fit: contain;
        }
        .lightbox-close {
            position: absolute;
            top: -40px;
            right: 0;
            color: white;
            font-size: 30px;
            cursor: pointer;
            background: none;
            border: none;
            padding: 0;
            width: 40px;
            height: 40px;
        }
        .lightbox-close:hover {
            opacity: 0.7;
        }
        .transparent-bg {
            background: 
                linear-gradient(45deg, #999999 25%, #7a7a7a 25%),
                linear-gradient(-45deg, #999999 25%, #7a7a7a 25%),
                linear-gradient(45deg, #7a7a7a 75%, #999999 75%),
                linear-gradient(-45deg, #7a7a7a 75%, #999999 75%);
            background-size: 30px 30px;
            background-position: 0 0, 0 15px, 15px -15px, -15px 0px;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🖼️ PNG Gallery</h1>
        <div class="header-actions">
            <button class="dark-mode-toggle" onclick="toggleDarkMode()" title="Toggle Dark Mode">
                <span id="theme-icon">🌙</span>
            </button>
            <a href="admin/upload.php" class="btn btn-primary">📤 Upload PNG</a>
            <a href="logout.php" class="btn btn-secondary">Logout</a>
        </div>
    </div>

    <div class="container">
        <?php if (!empty($allImages)): ?>
            <div class="stats">
                <div class="stat-item">
                    <div class="stat-value"><?php echo count($allImages); ?></div>
                    <div class="stat-label">Total Images</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">
                        <?php 
                        $totalSize = array_sum(array_column($allImages, 'size'));
                        echo number_format($totalSize / 1024 / 1024, 2);
                        ?>
                    </div>
                    <div class="stat-label">MB Storage</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">
                        <?php 
                        $withAlpha = count(array_filter($allImages, function($img) {
                            return $img['has_alpha'];
                        }));
                        echo $withAlpha;
                        ?>
                    </div>
                    <div class="stat-label">With Transparency</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?php echo $pagination['current_page']; ?>/<?php echo $pagination['pages']; ?></div>
                    <div class="stat-label">Current Page</div>
                </div>
            </div>

            <?php if (!empty($images)): ?>
                <div class="gallery-grid">
                    <?php foreach ($images as $img): ?>
                        <div class="gallery-item" onclick="openLightbox('<?php echo htmlspecialchars($img['filename']); ?>')">
                            <div class="image-container">
                                <img src="image.php?type=thumb&file=<?php echo urlencode($img['filename']); ?>" 
                                     alt="<?php echo htmlspecialchars($img['filename']); ?>"
                                     loading="lazy">
                            </div>
                            <div class="image-info">
                                <div class="image-filename">
                                    <?php echo htmlspecialchars(substr($img['filename'], 0, 30)); ?>
                                    <?php if ($img['has_alpha']): ?>
                                        <span class="badge">Alpha</span>
                                    <?php endif; ?>
                                </div>
                                <div class="image-meta">
                                    <span><?php echo number_format($img['size'] / 1024, 1); ?> KB</span>
                                    <span><?php echo date('d M Y', $img['uploaded']); ?></span>
                                </div>
                                <div class="image-actions" onclick="event.stopPropagation()">
                                    <a href="image.php?type=optimized&file=<?php echo urlencode($img['filename']); ?>" 
                                       target="_blank" class="btn-small btn-view">View</a>
                                    <a href="image.php?type=original&file=<?php echo urlencode($img['filename']); ?>&download=1" 
                                       class="btn-small btn-download">Download</a>
                                    <button onclick="renameImage('<?php echo htmlspecialchars($img['filename']); ?>')" 
                                            class="btn-small btn-rename">Rename</button>
                                    <button onclick="deleteImage('<?php echo htmlspecialchars($img['filename']); ?>')" 
                                            class="btn-small btn-delete">Delete</button>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Pagination -->
                <?php if ($pagination['pages'] > 1): ?>
                    <div class="pagination">
                        <?php if ($pagination['has_prev']): ?>
                            <a href="?page=1">« First</a>
                            <a href="?page=<?php echo $pagination['current_page'] - 1; ?>">‹ Prev</a>
                        <?php else: ?>
                            <span class="disabled">« First</span>
                            <span class="disabled">‹ Prev</span>
                        <?php endif; ?>

                        <?php
                        $start = max(1, $pagination['current_page'] - 2);
                        $end = min($pagination['pages'], $pagination['current_page'] + 2);
                        
                        for ($i = $start; $i <= $end; $i++):
                            if ($i == $pagination['current_page']):
                        ?>
                            <span class="active"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                        <?php 
                            endif;
                        endfor; 
                        ?>

                        <?php if ($pagination['has_next']): ?>
                            <a href="?page=<?php echo $pagination['current_page'] + 1; ?>">Next ›</a>
                            <a href="?page=<?php echo $pagination['pages']; ?>">Last »</a>
                        <?php else: ?>
                            <span class="disabled">Next ›</span>
                            <span class="disabled">Last »</span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        <?php else: ?>
            <div class="empty-state">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                </svg>
                <h2>Galeri Masih Kosong</h2>
                <p>Upload PNG transparan pertama Anda untuk memulai</p>
                <a href="admin/upload.php" class="btn btn-primary">Upload Sekarang</a>
            </div>
        <?php endif; ?>
    </div>

    <!-- Lightbox -->
    <div class="lightbox" id="lightbox" onclick="closeLightbox()">
        <div class="lightbox-content transparent-bg" onclick="event.stopPropagation()">
            <button class="lightbox-close" onclick="closeLightbox()">×</button>
            <img class="lightbox-image" id="lightbox-image" src="" alt="">
        </div>
    </div>

    <script>
        // Dark Mode
        function toggleDarkMode() {
            const html = document.documentElement;
            const currentTheme = html.getAttribute('data-theme');
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            
            html.setAttribute('data-theme', newTheme);
            localStorage.setItem('theme', newTheme);
            
            document.getElementById('theme-icon').textContent = newTheme === 'dark' ? '☀️' : '🌙';
        }

        // Load saved theme
        (function() {
            const savedTheme = localStorage.getItem('theme') || 'light';
            document.documentElement.setAttribute('data-theme', savedTheme);
            document.getElementById('theme-icon').textContent = savedTheme === 'dark' ? '☀️' : '🌙';
        })();

        // Lightbox
        function openLightbox(filename) {
            const lightbox = document.getElementById('lightbox');
            const image = document.getElementById('lightbox-image');
            image.src = 'image.php?type=optimized&file=' + encodeURIComponent(filename);
            lightbox.classList.add('active');
        }

        function closeLightbox() {
            document.getElementById('lightbox').classList.remove('active');
        }

        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') closeLightbox();
        });

        // Rename Image
        function renameImage(filename) {
            const currentName = filename.replace('.png', '');
            
            Swal.fire({
                title: 'Rename File',
                input: 'text',
                inputLabel: 'Enter new filename (without .png)',
                inputValue: currentName,
                showCancelButton: true,
                confirmButtonText: 'Rename',
                cancelButtonText: 'Cancel',
                confirmButtonColor: '#667eea',
                inputValidator: (value) => {
                    if (!value) {
                        return 'Filename cannot be empty!';
                    }
                    if (!/^[a-zA-Z0-9_-]+$/.test(value)) {
                        return 'Only letters, numbers, underscore and dash allowed!';
                    }
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const newFilename = result.value;
                    
                    // Show loading
                    Swal.fire({
                        title: 'Renaming...',
                        allowOutsideClick: false,
                        didOpen: () => {
                            Swal.showLoading();
                        }
                    });
                    
                    // Send AJAX request
                    const formData = new FormData();
                    formData.append('old_filename', filename);
                    formData.append('new_filename', newFilename);
                    formData.append('<?php echo CSRF_TOKEN_NAME; ?>', '<?php echo CSRF::getToken(); ?>');
                    
                    fetch('admin/rename.php', {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Success!',
                                text: data.message,
                                confirmButtonColor: '#667eea'
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: data.message,
                                confirmButtonColor: '#667eea'
                            });
                        }
                    })
                    .catch(error => {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: 'An error occurred: ' + error,
                            confirmButtonColor: '#667eea'
                        });
                    });
                }
            });
        }

        // Delete Image
        function deleteImage(filename) {
            Swal.fire({
                title: 'Delete this image?',
                text: "This action cannot be undone!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#f14668',
                cancelButtonColor: '#999',
                confirmButtonText: 'Yes, delete it!',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = 'admin/delete.php?file=' + encodeURIComponent(filename) + '&confirm=1';
                }
            });
        }
    </script>
</body>
</html>
