<?php
/**
 * PNG Handler Class - UPDATED VERSION
 * 
 * Tambahan method:
 * - rename() untuk rename file
 * - getAllImages() dengan pagination support
 */

// TAMBAHKAN METHOD BARU INI KE FILE PngHandler.php YANG SUDAH ADA
// Letakkan setelah method delete()

/**
 * Rename PNG and all its versions
 * 
 * @param string $oldFilename - Current filename
 * @param string $newFilename - New filename (without extension)
 * @return bool|array - True on success, array with error on failure
 */
public function rename($oldFilename, $newFilename)
{
    // Validate old filename
    $oldFilename = basename($oldFilename);
    if (empty($oldFilename) || !preg_match('/^[a-zA-Z0-9_-]+\.png$/', $oldFilename)) {
        return ['error' => 'Invalid old filename'];
    }

    // Sanitize new filename
    $newFilename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $newFilename);
    $newFilename = substr($newFilename, 0, 50);
    $newFilename = trim($newFilename, '_-');
    
    if (empty($newFilename)) {
        return ['error' => 'New filename cannot be empty'];
    }

    // Add .png extension
    $newFilename = $newFilename . '.png';

    // Check if new filename already exists
    if (file_exists(ORIGINALS_PATH . '/' . $newFilename)) {
        return ['error' => 'File with this name already exists'];
    }

    $oldPaths = [
        'original' => ORIGINALS_PATH . '/' . $oldFilename,
        'optimized' => OPTIMIZED_PATH . '/' . $oldFilename,
        'thumb' => THUMBS_PATH . '/' . $oldFilename
    ];

    $newPaths = [
        'original' => ORIGINALS_PATH . '/' . $newFilename,
        'optimized' => OPTIMIZED_PATH . '/' . $newFilename,
        'thumb' => THUMBS_PATH . '/' . $newFilename
    ];

    // Check if all old files exist
    foreach ($oldPaths as $key => $path) {
        if (!file_exists($path)) {
            return ['error' => ucfirst($key) . ' file not found'];
        }
    }

    // Rename all files
    $renamed = 0;
    $errors = [];
    
    foreach ($oldPaths as $key => $oldPath) {
        if (rename($oldPath, $newPaths[$key])) {
            $renamed++;
        } else {
            $errors[] = "Failed to rename $key";
        }
    }

    // Rollback if not all files renamed
    if ($renamed < count($oldPaths)) {
        // Rollback successfully renamed files
        foreach ($newPaths as $key => $newPath) {
            if (file_exists($newPath)) {
                rename($newPath, $oldPaths[$key]);
            }
        }
        return ['error' => 'Rename failed: ' . implode(', ', $errors)];
    }

    return true;
}

/**
 * Get paginated images
 * 
 * @param int $page - Current page (1-based)
 * @param int $perPage - Items per page
 * @return array - ['images' => [], 'total' => int, 'pages' => int, 'current_page' => int]
 */
public function getPaginatedImages($page = 1, $perPage = 10)
{
    $allImages = $this->getAllImages();
    $total = count($allImages);
    $pages = ceil($total / $perPage);
    
    // Validate page
    $page = max(1, min($page, $pages ?: 1));
    
    // Get slice
    $offset = ($page - 1) * $perPage;
    $images = array_slice($allImages, $offset, $perPage);
    
    return [
        'images' => $images,
        'total' => $total,
        'pages' => $pages,
        'current_page' => $page,
        'per_page' => $perPage,
        'has_prev' => $page > 1,
        'has_next' => $page < $pages
    ];
}
