# Testing Guide - PNG Transparency

## Cara Test Transparansi PNG

### 1. Persiapan Test Image

Buat atau download PNG transparan untuk testing:

**Cara membuat PNG transparan (Photoshop/GIMP):**
1. Buat dokumen baru dengan background transparent
2. Tambahkan shape/text dengan berbagai level opacity (0-255)
3. Save as PNG (bukan PNG-8)
4. Pastikan "Transparency" dicentang

**Download Sample PNG:**
- https://www.transparenttextures.com/
- https://www.pngwing.com/ (pilih yang ada alpha channel)

### 2. Test Upload

1. Login ke sistem
2. Upload PNG transparan
3. **VERIFIKASI:** Lihat background checkerboard (kotak abu-abu) di thumbnail
   - Jika checkerboard terlihat = transparansi BERHASIL
   - Jika background putih/hitam solid = transparansi HILANG ❌

### 3. Test Thumbnail

```bash
# Check thumbnail file dengan tool image
file storage/thumbs/yourfile.png

# Output yang benar:
# PNG image data, 200 x 200, 8-bit/color RGBA, non-interlaced
#                                      ^^^^
#                                      RGBA = Ada alpha channel!
```

**Manual Check:**
- Buka thumbnail di image editor (Photoshop/GIMP)
- Check channels panel
- Harus ada channel "Alpha" atau "Transparency"

### 4. Test Download

1. Klik "Download" di galeri
2. Buka file yang di-download di image editor
3. **VERIFIKASI:** Check alpha channel masih ada
4. **VERIFIKASI:** Test di background berbeda (hitam & putih):
   - Drag PNG ke background hitam → harus transparan
   - Drag PNG ke background putih → harus transparan

### 5. Test dengan PHP Script

Buat file `test_transparency.php`:

```php
<?php
// Test jika PNG memiliki alpha channel

$imagePath = 'path/to/your/image.png';

$img = imagecreatefrompng($imagePath);

// Check if image has alpha
$width = imagesx($img);
$height = imagesy($img);

$hasAlpha = false;
for ($x = 0; $x < $width; $x += 10) {
    for ($y = 0; $y < $height; $y += 10) {
        $rgba = imagecolorat($img, $x, $y);
        $alpha = ($rgba & 0x7F000000) >> 24;
        
        if ($alpha > 0) {
            $hasAlpha = true;
            break 2;
        }
    }
}

echo $hasAlpha ? "✅ Image HAS alpha channel\n" : "❌ Image DOES NOT have alpha\n";

imagedestroy($img);
?>
```

### 6. Visual Test Cases

Test dengan berbagai kondisi:

**Test Case 1: Full Transparent**
- Upload PNG dengan area 100% transparent
- Harus terlihat checkerboard di area tersebut

**Test Case 2: Partial Opacity**
- Upload PNG dengan opacity 50% (alpha = 127)
- Harus terlihat semi-transparent

**Test Case 3: Gradient Alpha**
- Upload PNG dengan gradient transparency (0-255)
- Semua level transparency harus terlihat smooth

**Test Case 4: Complex Alpha**
- Upload logo/icon dengan anti-aliasing edge
- Edge harus tetap smooth, tidak jagged

### 7. Common Issues & Fixes

**Issue 1: Transparansi hilang**
```php
// ❌ SALAH - Ini menghilangkan alpha:
$img = imagecreatefrompng($source);
imagepng($img, $dest);

// ✅ BENAR - Ini mempertahankan alpha:
$img = imagecreatefrompng($source);
imagealphablending($img, false);
imagesavealpha($img, true);
imagepng($img, $dest);
```

**Issue 2: Background putih/hitam muncul**
```php
// Selalu set alpha channel untuk canvas baru:
$thumb = imagecreatetruecolor($w, $h);
imagealphablending($thumb, false);
$transparent = imagecolorallocatealpha($thumb, 0, 0, 0, 127);
imagefill($thumb, 0, 0, $transparent);
imagesavealpha($thumb, true);
```

**Issue 3: Compression menghilangkan alpha**
```php
// Alpha tetap terjaga dengan compression level 0-9
imagepng($img, $dest, 9); // Level 9 = max compression
// Alpha channel TIDAK akan hilang selama imagealphablending(false)
// dan imagesavealpha(true) sudah diset
```

### 8. Browser Testing

Test di berbagai browser:

1. **Chrome/Edge:** 
   - Inspect element
   - Check computed background
   - Harus `transparent` bukan `white` atau `black`

2. **Firefox:**
   - Right-click image → View Image Info
   - Check color depth (harus 32-bit untuk RGBA)

3. **Safari:**
   - Web Inspector → Resources → Images
   - Check format (PNG-32 with alpha)

### 9. Command Line Testing

```bash
# Check PNG info dengan ImageMagick
identify -verbose storage/uploads/originals/yourfile.png | grep -i alpha

# Output yang benar:
# Alpha: srgba

# Check dengan exiftool
exiftool storage/uploads/originals/yourfile.png | grep -i color

# Output yang benar:
# Color Type: RGB with Alpha
```

### 10. Success Criteria

PNG transparency dianggap BERHASIL jika:

✅ Upload PNG → Transparansi tetap ada
✅ Thumbnail → Transparansi tetap ada  
✅ Optimized → Transparansi tetap ada
✅ Download → Transparansi tetap ada
✅ View di browser → Background checkerboard terlihat
✅ Open di image editor → Alpha channel terdeteksi
✅ File size berkurang → Tapi alpha tidak hilang

## Automated Test Script

Buat `test_all.php`:

```php
<?php
require_once 'includes/config.php';
require_once 'includes/PngHandler.php';

$handler = new PngHandler();
$images = $handler->getAllImages();

echo "=== Testing PNG Transparency ===\n\n";

foreach ($images as $img) {
    echo "Testing: {$img['filename']}\n";
    
    $tests = [
        'Original' => $img['original'],
        'Optimized' => $img['optimized'],
        'Thumbnail' => $img['thumb']
    ];
    
    foreach ($tests as $name => $path) {
        $hasAlpha = $handler->hasAlpha($path);
        $status = $hasAlpha ? '✅' : '❌';
        echo "  $status $name: " . ($hasAlpha ? 'HAS' : 'NO') . " alpha\n";
    }
    echo "\n";
}
?>
```

Run: `php test_all.php`

Semua file harus menunjukkan ✅ (HAS alpha).
