<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/PngHandler.php';
require_once __DIR__ . '/../includes/UserManager.php';

// Require login
Auth::requireLogin();

$currentUser = Auth::getCurrentUser();
$pngHandler = new PngHandler();
$userManager = new UserManager();

// Handle file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['png_file'])) {
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || !CSRF::validate($_POST['csrf_token'])) {
        $error = 'Invalid request';
    } else {
        // BUG #4 FIX: Pass user ID and name to track uploader
        $result = $pngHandler->upload(
            $_FILES['png_file'],
            $currentUser['id'],
            $currentUser['display_name']
        );
        
        if ($result['success']) {
            // BUG #3 FIX: Increment upload count for the user
            $userManager->incrementUploadCount($currentUser['id']);
            
            // BUG #7 FIX: Refresh session to update stats
            $userManager->refreshSession($currentUser['id']);
            
            $success = $result['message'];
            $uploadedFile = $result['filename'];
        } else {
            $error = $result['message'];
        }
    }
}

$pageTitle = 'Upload PNG';
?>
<!DOCTYPE html>
<html lang="en" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle) ?> - PNG Gallery Ultimate</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #9333ea;
            --primary-dark: #7e22ce;
            --primary-light: #a855f7;
            --bg-dark: #0a0a0a;
            --bg-card: #1a1a1a;
            --bg-hover: #2a2a2a;
            --text-primary: #ffffff;
            --text-secondary: #a3a3a3;
            --border: #333333;
            --success: #10b981;
            --error: #ef4444;
        }

        [data-theme="light"] {
            --bg-dark: #f5f5f5;
            --bg-card: #ffffff;
            --bg-hover: #e5e5e5;
            --text-primary: #0a0a0a;
            --text-secondary: #525252;
            --border: #d4d4d4;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--bg-dark);
            color: var(--text-primary);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .logo {
            font-size: 1.5rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .nav-buttons {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .btn {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            border: none;
            background: var(--bg-hover);
            color: var(--text-primary);
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .btn:hover {
            background: var(--primary);
            transform: translateY(-2px);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
        }

        /* Main Content */
        .container {
            max-width: 800px;
            margin: 3rem auto;
            padding: 0 2rem;
        }

        .upload-card {
            background: var(--bg-card);
            border-radius: 16px;
            padding: 3rem;
            border: 1px solid var(--border);
        }

        .upload-card h1 {
            font-size: 2rem;
            margin-bottom: 0.5rem;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .upload-card p {
            color: var(--text-secondary);
            margin-bottom: 2rem;
        }

        /* Upload Zone */
        .upload-zone {
            border: 3px dashed var(--border);
            border-radius: 12px;
            padding: 3rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
            margin-bottom: 1.5rem;
        }

        .upload-zone:hover {
            border-color: var(--primary);
            background: var(--bg-hover);
        }

        .upload-zone.dragover {
            border-color: var(--primary);
            background: var(--bg-hover);
            transform: scale(1.02);
        }

        .upload-icon {
            font-size: 4rem;
            margin-bottom: 1rem;
            opacity: 0.6;
        }

        .upload-text {
            font-size: 1.2rem;
            margin-bottom: 0.5rem;
        }

        .upload-hint {
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        input[type="file"] {
            display: none;
        }

        /* Preview */
        .preview-container {
            margin-top: 2rem;
            display: none;
        }

        .preview-container.active {
            display: block;
        }

        .preview-image {
            max-width: 100%;
            border-radius: 12px;
            margin-bottom: 1rem;
        }

        /* Messages */
        .message {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .message.success {
            background: rgba(16, 185, 129, 0.1);
            border: 1px solid var(--success);
            color: var(--success);
        }

        .message.error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--error);
            color: var(--error);
        }

        /* Submit Button */
        .submit-btn {
            width: 100%;
            padding: 1rem;
            font-size: 1.1rem;
            margin-top: 1rem;
        }

        .submit-btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        @media (max-width: 768px) {
            .container {
                padding: 0 1rem;
            }

            .upload-card {
                padding: 2rem 1.5rem;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-content">
            <a href="/" class="logo">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                    <circle cx="8.5" cy="8.5" r="1.5"/>
                    <polyline points="21 15 16 10 5 21"/>
                </svg>
                PNG Gallery Ultimate
            </a>
            
            <div class="nav-buttons">
                <a href="/" class="btn">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                    </svg>
                    Back to Gallery
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="container">
        <div class="upload-card">
            <h1>Upload PNG</h1>
            <p>Upload transparent PNG images (max 10MB)</p>

            <?php if (isset($success)): ?>
                <div class="message success">
                    ✓ <?= htmlspecialchars($success) ?>
                    <br><small>Your upload count has been updated!</small>
                </div>
            <?php endif; ?>

            <?php if (isset($error)): ?>
                <div class="message error">
                    ✗ <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="POST" enctype="multipart/form-data" id="uploadForm">
                <input type="hidden" name="csrf_token" value="<?= CSRF::generate() ?>">
                
                <div class="upload-zone" id="uploadZone">
                    <div class="upload-icon">📤</div>
                    <div class="upload-text">Click to upload or drag & drop</div>
                    <div class="upload-hint">PNG files only, up to 10MB</div>
                </div>
                
                <input type="file" name="png_file" id="fileInput" accept=".png,image/png" required>
                
                <div class="preview-container" id="previewContainer">
                    <img id="previewImage" class="preview-image" alt="Preview">
                </div>
                
                <button type="submit" class="btn btn-primary submit-btn" id="submitBtn" disabled>
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="17 8 12 3 7 8"/>
                        <line x1="12" y1="3" x2="12" y2="15"/>
                    </svg>
                    Upload PNG
                </button>
            </form>
        </div>
    </div>

    <script>
        const uploadZone = document.getElementById('uploadZone');
        const fileInput = document.getElementById('fileInput');
        const previewContainer = document.getElementById('previewContainer');
        const previewImage = document.getElementById('previewImage');
        const submitBtn = document.getElementById('submitBtn');

        // Click to upload
        uploadZone.addEventListener('click', () => {
            fileInput.click();
        });

        // File selected
        fileInput.addEventListener('change', handleFileSelect);

        // Drag and drop
        uploadZone.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadZone.classList.add('dragover');
        });

        uploadZone.addEventListener('dragleave', () => {
            uploadZone.classList.remove('dragover');
        });

        uploadZone.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadZone.classList.remove('dragover');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                fileInput.files = files;
                handleFileSelect();
            }
        });

        function handleFileSelect() {
            const file = fileInput.files[0];
            if (file) {
                // Validate PNG
                if (!file.type.match('image/png')) {
                    alert('Only PNG files are allowed!');
                    fileInput.value = '';
                    return;
                }

                // Validate size (10MB)
                if (file.size > 10485760) {
                    alert('File too large! Maximum size is 10MB.');
                    fileInput.value = '';
                    return;
                }

                // Show preview
                const reader = new FileReader();
                reader.onload = (e) => {
                    previewImage.src = e.target.result;
                    previewContainer.classList.add('active');
                    submitBtn.disabled = false;
                };
                reader.readAsDataURL(file);
            }
        }
    </script>
</body>
</html>
