<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/UserManager.php';

// Require login
Auth::requireLogin();

$currentUser = Auth::getCurrentUser();
$userManager = new UserManager();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!CSRF::validate($_POST['csrf_token'])) {
        $error = 'Invalid request';
    } else {
        $updateData = [];
        
        // Check what fields are being updated
        if (!empty($_POST['username']) && $_POST['username'] !== $currentUser['username']) {
            $updateData['username'] = trim($_POST['username']);
        }
        
        if (!empty($_POST['display_name']) && $_POST['display_name'] !== $currentUser['display_name']) {
            $updateData['display_name'] = trim($_POST['display_name']);
        }
        
        if (!empty($_POST['email'])) {
            $updateData['email'] = trim($_POST['email']);
        }
        
        if (!empty($_POST['new_password'])) {
            if (empty($_POST['current_password'])) {
                $error = 'Current password is required to change password';
            } else {
                // Verify current password
                $user = $userManager->getUserById($currentUser['id']);
                if (!password_verify($_POST['current_password'], $user['password'])) {
                    $error = 'Current password is incorrect';
                } else {
                    $updateData['password'] = $_POST['new_password'];
                }
            }
        }
        
        if (!isset($error) && !empty($updateData)) {
            $result = $userManager->updateUser($currentUser['id'], $updateData);
            
            if ($result['success']) {
                // BUG #7 FIX: Refresh session after profile update
                $userManager->refreshSession($currentUser['id']);
                
                // Refresh current user variable
                $currentUser = Auth::getCurrentUser();
                
                $success = 'Profile updated successfully!';
            } else {
                $error = $result['message'];
            }
        } elseif (!isset($error) && empty($updateData)) {
            $info = 'No changes to save';
        }
    }
}

$pageTitle = 'Profile Settings';
?>
<!DOCTYPE html>
<html lang="en" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle) ?> - PNG Gallery Ultimate</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #9333ea;
            --primary-dark: #7e22ce;
            --primary-light: #a855f7;
            --bg-dark: #0a0a0a;
            --bg-card: #1a1a1a;
            --bg-hover: #2a2a2a;
            --text-primary: #ffffff;
            --text-secondary: #a3a3a3;
            --border: #333333;
            --success: #10b981;
            --error: #ef4444;
            --info: #3b82f6;
        }

        [data-theme="light"] {
            --bg-dark: #f5f5f5;
            --bg-card: #ffffff;
            --bg-hover: #e5e5e5;
            --text-primary: #0a0a0a;
            --text-secondary: #525252;
            --border: #d4d4d4;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--bg-dark);
            color: var(--text-primary);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .logo {
            font-size: 1.5rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .nav-buttons {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .btn {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            border: none;
            background: var(--bg-hover);
            color: var(--text-primary);
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .btn:hover {
            background: var(--primary);
            transform: translateY(-2px);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
        }

        /* Main Content */
        .container {
            max-width: 900px;
            margin: 3rem auto;
            padding: 0 2rem;
        }

        .card {
            background: var(--bg-card);
            border-radius: 16px;
            padding: 3rem;
            border: 1px solid var(--border);
            margin-bottom: 2rem;
        }

        .card h1 {
            font-size: 2rem;
            margin-bottom: 0.5rem;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .card h2 {
            font-size: 1.5rem;
            margin-bottom: 1rem;
            color: var(--text-primary);
        }

        .card p {
            color: var(--text-secondary);
            margin-bottom: 2rem;
        }

        /* Profile Header */
        .profile-header {
            display: flex;
            align-items: center;
            gap: 2rem;
            margin-bottom: 2rem;
            padding-bottom: 2rem;
            border-bottom: 1px solid var(--border);
        }

        .avatar-large {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2.5rem;
            font-weight: 700;
            color: white;
        }

        .profile-info h2 {
            margin-bottom: 0.25rem;
        }

        .profile-badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            background: rgba(147, 51, 234, 0.2);
            color: var(--primary-light);
            font-size: 0.85rem;
            font-weight: 600;
            margin-top: 0.5rem;
        }

        /* Form */
        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: var(--text-secondary);
        }

        .form-group input {
            width: 100%;
            padding: 0.75rem 1rem;
            border-radius: 8px;
            border: 1px solid var(--border);
            background: var(--bg-hover);
            color: var(--text-primary);
            font-size: 1rem;
            transition: all 0.2s;
        }

        .form-group input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(147, 51, 234, 0.1);
        }

        .form-hint {
            color: var(--text-secondary);
            font-size: 0.85rem;
            margin-top: 0.25rem;
        }

        /* Messages */
        .message {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .message.success {
            background: rgba(16, 185, 129, 0.1);
            border: 1px solid var(--success);
            color: var(--success);
        }

        .message.error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--error);
            color: var(--error);
        }

        .message.info {
            background: rgba(59, 130, 246, 0.1);
            border: 1px solid var(--info);
            color: var(--info);
        }

        /* Section */
        .section {
            margin-bottom: 2rem;
            padding-bottom: 2rem;
            border-bottom: 1px solid var(--border);
        }

        .section:last-child {
            border-bottom: none;
            margin-bottom: 0;
            padding-bottom: 0;
        }

        @media (max-width: 768px) {
            .container {
                padding: 0 1rem;
            }

            .card {
                padding: 2rem 1.5rem;
            }

            .profile-header {
                flex-direction: column;
                text-align: center;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-content">
            <a href="/" class="logo">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                    <circle cx="8.5" cy="8.5" r="1.5"/>
                    <polyline points="21 15 16 10 5 21"/>
                </svg>
                PNG Gallery Ultimate
            </a>
            
            <div class="nav-buttons">
                <a href="/" class="btn">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M15 19l-7-7 7-7"/>
                    </svg>
                    Back to Gallery
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="container">
        <div class="card">
            <h1>Profile Settings</h1>
            <p>Manage your account information</p>

            <?php if (isset($success)): ?>
                <div class="message success">
                    ✓ <?= htmlspecialchars($success) ?>
                </div>
            <?php endif; ?>

            <?php if (isset($error)): ?>
                <div class="message error">
                    ✗ <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <?php if (isset($info)): ?>
                <div class="message info">
                    ℹ <?= htmlspecialchars($info) ?>
                </div>
            <?php endif; ?>

            <!-- Profile Header -->
            <div class="profile-header">
                <div class="avatar-large">
                    <?php
                    $initials = '';
                    $nameParts = explode(' ', $currentUser['display_name']);
                    foreach ($nameParts as $part) {
                        $initials .= strtoupper(substr($part, 0, 1));
                    }
                    echo htmlspecialchars(substr($initials, 0, 2));
                    ?>
                </div>
                <div class="profile-info">
                    <h2><?= htmlspecialchars($currentUser['display_name']) ?></h2>
                    <div style="color: var(--text-secondary);">@<?= htmlspecialchars($currentUser['username']) ?></div>
                    <span class="profile-badge"><?= htmlspecialchars(ucfirst($currentUser['role'])) ?></span>
                </div>
            </div>

            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?= CSRF::generate() ?>">
                
                <!-- Basic Info Section -->
                <div class="section">
                    <h2>Basic Information</h2>
                    
                    <div class="form-group">
                        <label for="username">Username *</label>
                        <input 
                            type="text" 
                            id="username" 
                            name="username" 
                            value="<?= htmlspecialchars($currentUser['username']) ?>" 
                            required
                            pattern="[a-zA-Z0-9_]{3,20}"
                            title="3-20 characters, alphanumeric and underscore only"
                        >
                        <div class="form-hint">Used for login. 3-20 characters, alphanumeric and underscore only.</div>
                    </div>

                    <div class="form-group">
                        <label for="display_name">Display Name *</label>
                        <input 
                            type="text" 
                            id="display_name" 
                            name="display_name" 
                            value="<?= htmlspecialchars($currentUser['display_name']) ?>" 
                            required
                        >
                        <div class="form-hint">Your name as it appears throughout the site.</div>
                    </div>

                    <div class="form-group">
                        <label for="email">Email</label>
                        <input 
                            type="email" 
                            id="email" 
                            name="email" 
                            value="<?= htmlspecialchars($currentUser['email'] ?? '') ?>"
                        >
                        <div class="form-hint">Optional. Used for notifications.</div>
                    </div>
                </div>

                <!-- Password Section -->
                <div class="section">
                    <h2>Change Password</h2>
                    
                    <div class="form-group">
                        <label for="current_password">Current Password</label>
                        <input 
                            type="password" 
                            id="current_password" 
                            name="current_password"
                        >
                        <div class="form-hint">Required to change password.</div>
                    </div>

                    <div class="form-group">
                        <label for="new_password">New Password</label>
                        <input 
                            type="password" 
                            id="new_password" 
                            name="new_password"
                            minlength="6"
                        >
                        <div class="form-hint">Leave empty to keep current password. Minimum 6 characters.</div>
                    </div>
                </div>

                <button type="submit" class="btn btn-primary" style="width: 100%; padding: 1rem; font-size: 1.1rem;">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/>
                        <polyline points="17 21 17 13 7 13 7 21"/>
                        <polyline points="7 3 7 8 15 8"/>
                    </svg>
                    Save Changes
                </button>
            </form>
        </div>

        <!-- Additional Info Card -->
        <div class="card">
            <h2>Account Statistics</h2>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1.5rem; margin-top: 1.5rem;">
                <div style="text-align: center;">
                    <div style="font-size: 2rem; font-weight: 700; color: var(--primary);">
                        <?= $currentUser['uploads'] ?? 0 ?>
                    </div>
                    <div style="color: var(--text-secondary); margin-top: 0.5rem;">Total Uploads</div>
                </div>
                <div style="text-align: center;">
                    <div style="font-size: 2rem; font-weight: 700; color: var(--primary);">
                        <?= date('M d, Y', strtotime($currentUser['created_at'])) ?>
                    </div>
                    <div style="color: var(--text-secondary); margin-top: 0.5rem;">Member Since</div>
                </div>
                <div style="text-align: center;">
                    <div style="font-size: 2rem; font-weight: 700; color: var(--primary);">
                        <?= $currentUser['last_login'] ? date('M d, Y', strtotime($currentUser['last_login'])) : 'Never' ?>
                    </div>
                    <div style="color: var(--text-secondary); margin-top: 0.5rem;">Last Login</div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
