<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/UserManager.php';

// Redirect if already logged in
if (Auth::isLoggedIn()) {
    header('Location: /');
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!CSRF::validate($_POST['csrf_token'])) {
        $error = 'Invalid request';
    } else {
        $username = trim($_POST['username']);
        $password = $_POST['password'];
        
        $userManager = new UserManager();
        $user = $userManager->getUserByUsername($username);
        
        if ($user && password_verify($password, $user['password'])) {
            $userManager->updateLastLogin($user['id']);
            Auth::login($user);
            header('Location: /');
            exit;
        } else {
            $error = 'Invalid username or password';
            // Log failed attempt
            if ($user) {
                Auth::logLoginActivity($user['id'], $username, 'failed');
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - PNG Gallery Ultimate</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #9333ea;
            --primary-dark: #7e22ce;
            --primary-light: #a855f7;
            --bg-dark: #0a0a0a;
            --bg-card: #1a1a1a;
            --text-primary: #ffffff;
            --text-secondary: #a3a3a3;
            --border: #333333;
            --error: #ef4444;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--bg-dark);
            color: var(--text-primary);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }

        .login-container {
            width: 100%;
            max-width: 420px;
        }

        .logo {
            text-align: center;
            margin-bottom: 3rem;
        }

        .logo h1 {
            font-size: 2.5rem;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 0.5rem;
        }

        .logo p {
            color: var(--text-secondary);
        }

        .login-card {
            background: var(--bg-card);
            border-radius: 16px;
            padding: 3rem 2.5rem;
            border: 1px solid var(--border);
        }

        .login-card h2 {
            font-size: 1.75rem;
            margin-bottom: 0.5rem;
        }

        .login-card p {
            color: var(--text-secondary);
            margin-bottom: 2rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: var(--text-secondary);
        }

        .form-group input {
            width: 100%;
            padding: 0.75rem 1rem;
            border-radius: 8px;
            border: 1px solid var(--border);
            background: var(--bg-dark);
            color: var(--text-primary);
            font-size: 1rem;
            transition: all 0.2s;
        }

        .form-group input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(147, 51, 234, 0.1);
        }

        .error-message {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--error);
            color: var(--error);
            padding: 0.75rem 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            font-size: 0.9rem;
        }

        .btn-login {
            width: 100%;
            padding: 0.875rem;
            border-radius: 8px;
            border: none;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
        }

        .btn-login:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 16px rgba(147, 51, 234, 0.3);
        }

        .default-creds {
            margin-top: 2rem;
            padding-top: 2rem;
            border-top: 1px solid var(--border);
            text-align: center;
            font-size: 0.85rem;
            color: var(--text-secondary);
        }

        .default-creds code {
            background: var(--bg-dark);
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-family: monospace;
            color: var(--primary-light);
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="logo">
            <h1>🖼️ PNG Gallery</h1>
            <p>Ultimate Edition</p>
        </div>

        <div class="login-card">
            <h2>Welcome Back</h2>
            <p>Sign in to your account</p>

            <?php if ($error): ?>
                <div class="error-message">
                    ✗ <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?= CSRF::generate() ?>">
                
                <div class="form-group">
                    <label for="username">Username</label>
                    <input 
                        type="text" 
                        id="username" 
                        name="username" 
                        required 
                        autofocus
                        autocomplete="username"
                    >
                </div>

                <div class="form-group">
                    <label for="password">Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        required
                        autocomplete="current-password"
                    >
                </div>

                <button type="submit" class="btn-login">
                    Sign In
                </button>
            </form>

            <div class="default-creds">
                Default: <code>admin</code> / <code>password</code><br>
                <small>⚠️ Change password after first login!</small>
            </div>
        </div>
    </div>
</body>
</html>
