<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';

// Require login
Auth::requireLogin();

$currentUser = Auth::getCurrentUser();
$isAdmin = Auth::isAdmin();

// Get login history
$historyFile = __DIR__ . '/../storage/session/login_history.json';
$loginHistory = [];

if (file_exists($historyFile)) {
    $json = file_get_contents($historyFile);
    $allHistory = json_decode($json, true) ?: [];
    
    // Filter by user if not admin
    if ($isAdmin) {
        $loginHistory = $allHistory;
    } else {
        $loginHistory = array_filter($allHistory, function($entry) use ($currentUser) {
            return $entry['user_id'] === $currentUser['id'];
        });
    }
    
    // Sort by date descending
    usort($loginHistory, function($a, $b) {
        return strtotime($b['timestamp']) - strtotime($a['timestamp']);
    });
}

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 20;
$total = count($loginHistory);
$pages = ceil($total / $perPage);
$offset = ($page - 1) * $perPage;
$pagedHistory = array_slice($loginHistory, $offset, $perPage);

$pageTitle = 'Login History';
?>
<!DOCTYPE html>
<html lang="en" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle) ?> - PNG Gallery Ultimate</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #9333ea;
            --primary-dark: #7e22ce;
            --primary-light: #a855f7;
            --bg-dark: #0a0a0a;
            --bg-card: #1a1a1a;
            --bg-hover: #2a2a2a;
            --text-primary: #ffffff;
            --text-secondary: #a3a3a3;
            --border: #333333;
            --success: #10b981;
            --error: #ef4444;
        }

        [data-theme="light"] {
            --bg-dark: #f5f5f5;
            --bg-card: #ffffff;
            --bg-hover: #e5e5e5;
            --text-primary: #0a0a0a;
            --text-secondary: #525252;
            --border: #d4d4d4;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--bg-dark);
            color: var(--text-primary);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .logo {
            font-size: 1.5rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .nav-buttons {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .btn {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            border: none;
            background: var(--bg-hover);
            color: var(--text-primary);
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .btn:hover {
            background: var(--primary);
            transform: translateY(-2px);
        }

        /* Container */
        .container {
            max-width: 1200px;
            margin: 3rem auto;
            padding: 0 2rem;
        }

        .page-header {
            margin-bottom: 2rem;
        }

        .page-header h1 {
            font-size: 2.5rem;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 0.5rem;
        }

        .page-header p {
            color: var(--text-secondary);
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: var(--bg-card);
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid var(--border);
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--primary);
            margin-bottom: 0.5rem;
        }

        .stat-label {
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        /* History Table */
        .history-table {
            background: var(--bg-card);
            border-radius: 12px;
            border: 1px solid var(--border);
            overflow: hidden;
        }

        .table-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .table-header h2 {
            font-size: 1.5rem;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        thead {
            background: var(--bg-hover);
        }

        th {
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: var(--text-secondary);
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        td {
            padding: 1rem;
            border-top: 1px solid var(--border);
        }

        tbody tr {
            transition: background 0.2s;
        }

        tbody tr:hover {
            background: var(--bg-hover);
        }

        .status-badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.85rem;
            font-weight: 600;
        }

        .status-success {
            background: rgba(16, 185, 129, 0.2);
            color: var(--success);
        }

        .status-failed {
            background: rgba(239, 68, 68, 0.2);
            color: var(--error);
        }

        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: var(--text-secondary);
        }

        .empty-state svg {
            width: 64px;
            height: 64px;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 0.5rem;
            margin-top: 2rem;
        }

        .pagination a {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            background: var(--bg-card);
            color: var(--text-primary);
            text-decoration: none;
            border: 1px solid var(--border);
            transition: all 0.2s;
        }

        .pagination a:hover {
            background: var(--primary);
            border-color: var(--primary);
        }

        .pagination a.active {
            background: var(--primary);
            border-color: var(--primary);
        }

        @media (max-width: 768px) {
            .container {
                padding: 0 1rem;
            }

            .history-table {
                overflow-x: auto;
            }

            table {
                min-width: 600px;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-content">
            <a href="/" class="logo">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                    <circle cx="8.5" cy="8.5" r="1.5"/>
                    <polyline points="21 15 16 10 5 21"/>
                </svg>
                PNG Gallery Ultimate
            </a>
            
            <div class="nav-buttons">
                <a href="/" class="btn">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M15 19l-7-7 7-7"/>
                    </svg>
                    Back to Gallery
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="container">
        <div class="page-header">
            <h1>Login History</h1>
            <p><?= $isAdmin ? 'All user login activities' : 'Your login activity history' ?></p>
        </div>

        <!-- Stats -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-value"><?= count($loginHistory) ?></div>
                <div class="stat-label">Total Logins</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">
                    <?php
                    $successCount = count(array_filter($loginHistory, function($e) { return $e['status'] === 'success'; }));
                    echo $successCount;
                    ?>
                </div>
                <div class="stat-label">Successful Logins</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">
                    <?php
                    $failedCount = count(array_filter($loginHistory, function($e) { return $e['status'] === 'failed'; }));
                    echo $failedCount;
                    ?>
                </div>
                <div class="stat-label">Failed Attempts</div>
            </div>
        </div>

        <!-- History Table -->
        <div class="history-table">
            <div class="table-header">
                <h2>Login Activity</h2>
                <?php if (count($loginHistory) > 0): ?>
                <a href="#" class="btn" onclick="exportCSV(); return false;">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="7 10 12 15 17 10"/>
                        <line x1="12" y1="15" x2="12" y2="3"/>
                    </svg>
                    Export CSV
                </a>
                <?php endif; ?>
            </div>

            <?php if (count($pagedHistory) > 0): ?>
            <table>
                <thead>
                    <tr>
                        <th>Date & Time</th>
                        <?php if ($isAdmin): ?><th>User</th><?php endif; ?>
                        <th>IP Address</th>
                        <th>Browser</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($pagedHistory as $entry): ?>
                    <tr>
                        <td><?= date('M d, Y H:i:s', strtotime($entry['timestamp'])) ?></td>
                        <?php if ($isAdmin): ?>
                        <td><?= htmlspecialchars($entry['username'] ?? 'Unknown') ?></td>
                        <?php endif; ?>
                        <td><?= htmlspecialchars($entry['ip'] ?? 'Unknown') ?></td>
                        <td><?= htmlspecialchars($entry['user_agent'] ?? 'Unknown') ?></td>
                        <td>
                            <span class="status-badge status-<?= $entry['status'] ?>">
                                <?= ucfirst($entry['status']) ?>
                            </span>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <!-- Pagination -->
            <?php if ($pages > 1): ?>
            <div class="pagination">
                <?php for ($i = 1; $i <= $pages; $i++): ?>
                <a href="?page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>">
                    <?= $i ?>
                </a>
                <?php endfor; ?>
            </div>
            <?php endif; ?>

            <?php else: ?>
            <div class="empty-state">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="10"/>
                    <line x1="12" y1="8" x2="12" y2="12"/>
                    <line x1="12" y1="16" x2="12.01" y2="16"/>
                </svg>
                <h3>No Login History</h3>
                <p>Login history will appear here once you start logging in.</p>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        function exportCSV() {
            const data = <?= json_encode($loginHistory) ?>;
            
            let csv = 'Date,Time' + (<?= $isAdmin ? 'true' : 'false' ?> ? ',User' : '') + ',IP Address,Browser,Status\n';
            
            data.forEach(entry => {
                const date = new Date(entry.timestamp);
                const dateStr = date.toLocaleDateString();
                const timeStr = date.toLocaleTimeString();
                
                csv += `${dateStr},${timeStr}`;
                if (<?= $isAdmin ? 'true' : 'false' ?>) {
                    csv += `,${entry.username || 'Unknown'}`;
                }
                csv += `,${entry.ip || 'Unknown'}`;
                csv += `,"${entry.user_agent || 'Unknown'}"`;
                csv += `,${entry.status}\n`;
            });
            
            const blob = new Blob([csv], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'login_history_' + new Date().toISOString().split('T')[0] + '.csv';
            a.click();
            window.URL.revokeObjectURL(url);
        }
    </script>
</body>
</html>
