<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/PngHandler.php';
require_once __DIR__ . '/../includes/UserManager.php';

Auth::requireLogin();

$currentUser = Auth::getCurrentUser();
$isAdmin = Auth::isAdmin();
$pngHandler = new PngHandler();
$userManager = new UserManager();

// Get files with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$filesData = $pngHandler->getFiles($page, 12);
$storageInfo = $pngHandler->getStorageInfo();
$userStats = $userManager->getStats();

$pageTitle = 'Gallery';
?>
<!DOCTYPE html>
<html lang="en" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle) ?> - PNG Gallery Ultimate</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #9333ea;
            --primary-dark: #7e22ce;
            --primary-light: #a855f7;
            --bg-dark: #0a0a0a;
            --bg-card: #1a1a1a;
            --bg-hover: #2a2a2a;
            --text-primary: #ffffff;
            --text-secondary: #a3a3a3;
            --border: #333333;
            --success: #10b981;
        }

        [data-theme="light"] {
            --bg-dark: #f5f5f5;
            --bg-card: #ffffff;
            --bg-hover: #e5e5e5;
            --text-primary: #0a0a0a;
            --text-secondary: #525252;
            --border: #d4d4d4;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--bg-dark);
            color: var(--text-primary);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .logo {
            font-size: 1.5rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .nav-buttons {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .btn {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            border: none;
            background: var(--bg-hover);
            color: var(--text-primary);
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .btn:hover {
            background: var(--primary);
            transform: translateY(-2px);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
        }

        /* User Dropdown */
        .user-dropdown {
            position: relative;
        }

        .user-button {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .user-avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            font-size: 0.9rem;
        }

        .dropdown-menu {
            position: absolute;
            right: 0;
            top: 100%;
            margin-top: 0.5rem;
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 8px;
            min-width: 200px;
            display: none;
            box-shadow: 0 8px 16px rgba(0,0,0,0.3);
        }

        .dropdown-menu.active {
            display: block;
        }

        .dropdown-menu a {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem 1rem;
            color: var(--text-primary);
            text-decoration: none;
            transition: background 0.2s;
        }

        .dropdown-menu a:hover {
            background: var(--bg-hover);
        }

        /* Stats */
        .stats-grid {
            max-width: 1400px;
            margin: 2rem auto;
            padding: 0 2rem;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
        }

        .stat-card {
            background: var(--bg-card);
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid var(--border);
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--primary);
            margin-bottom: 0.5rem;
        }

        .stat-label {
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        /* Gallery */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }

        .gallery-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 1.5rem;
        }

        .gallery-card {
            background: var(--bg-card);
            border-radius: 12px;
            border: 1px solid var(--border);
            overflow: hidden;
            transition: transform 0.2s, box-shadow 0.2s;
            cursor: pointer;
        }

        .gallery-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 24px rgba(147, 51, 234, 0.2);
        }

        .image-container {
            position: relative;
            width: 100%;
            height: 200px;
            background: repeating-conic-gradient(#404040 0% 25%, transparent 0% 50%) 50% / 20px 20px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .gallery-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
        }

        .card-info {
            padding: 1rem;
        }

        .card-title {
            font-weight: 600;
            margin-bottom: 0.5rem;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .card-meta {
            display: flex;
            justify-content: space-between;
            color: var(--text-secondary);
            font-size: 0.85rem;
        }

        /* BUG #4 FIX: Uploader info display */
        .uploader-info {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--primary-light);
            font-size: 0.85rem;
            margin-top: 0.5rem;
        }

        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: var(--text-secondary);
        }

        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 0.5rem;
            margin-top: 2rem;
        }

        .pagination a {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            background: var(--bg-card);
            color: var(--text-primary);
            text-decoration: none;
            border: 1px solid var(--border);
        }

        .pagination a:hover {
            background: var(--primary);
        }

        @media (max-width: 768px) {
            .header-content {
                flex-wrap: wrap;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .gallery-grid {
                grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <a href="/" class="logo">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                    <circle cx="8.5" cy="8.5" r="1.5"/>
                    <polyline points="21 15 16 10 5 21"/>
                </svg>
                PNG Gallery Ultimate
            </a>
            
            <div class="nav-buttons">
                <?php if ($isAdmin): ?>
                <a href="/users" class="btn">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                        <circle cx="9" cy="7" r="4"/>
                        <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                        <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                    </svg>
                    Users
                </a>
                <?php endif; ?>
                
                <a href="/upload" class="btn btn-primary">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="17 8 12 3 7 8"/>
                        <line x1="12" y1="3" x2="12" y2="15"/>
                    </svg>
                    Upload
                </a>
                
                <div class="user-dropdown">
                    <button class="btn user-button" onclick="toggleDropdown()">
                        <div class="user-avatar">
                            <?php
                            $initials = '';
                            $nameParts = explode(' ', $currentUser['display_name']);
                            foreach ($nameParts as $part) {
                                $initials .= strtoupper(substr($part, 0, 1));
                            }
                            echo htmlspecialchars(substr($initials, 0, 2));
                            ?>
                        </div>
                        <span><?= htmlspecialchars($currentUser['display_name']) ?></span>
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <polyline points="6 9 12 15 18 9"/>
                        </svg>
                    </button>
                    
                    <div class="dropdown-menu" id="dropdownMenu">
                        <a href="/profile-settings">
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="3"/>
                                <path d="M12 1v6m0 6v6"/>
                            </svg>
                            Settings
                        </a>
                        <a href="/login-history">
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <polyline points="12 6 12 12 16 14"/>
                            </svg>
                            Login History
                        </a>
                        <a href="/logout">
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"/>
                                <polyline points="16 17 21 12 16 7"/>
                                <line x1="21" y1="12" x2="9" y2="12"/>
                            </svg>
                            Logout
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-value"><?= $storageInfo['total_files'] ?></div>
            <div class="stat-label">Total Images</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?= $storageInfo['total_size_mb'] ?> MB</div>
            <div class="stat-label">Storage Used</div>
        </div>
        <?php if ($isAdmin): ?>
        <div class="stat-card">
            <div class="stat-value"><?= $userStats['total'] ?></div>
            <div class="stat-label">Total Users</div>
        </div>
        <?php endif; ?>
        <div class="stat-card">
            <div class="stat-value"><?= $currentUser['uploads'] ?? 0 ?></div>
            <div class="stat-label">Your Uploads</div>
        </div>
    </div>

    <div class="container">
        <?php if (count($filesData['files']) > 0): ?>
        <div class="gallery-grid">
            <?php foreach ($filesData['files'] as $file): ?>
            <div class="gallery-card">
                <div class="image-container">
                    <img src="<?= htmlspecialchars($file['url']) ?>" alt="<?= htmlspecialchars($file['original_name']) ?>" class="gallery-image">
                </div>
                <div class="card-info">
                    <div class="card-title"><?= htmlspecialchars($file['original_name']) ?></div>
                    <div class="card-meta">
                        <span><?= round($file['size'] / 1024, 1) ?> KB</span>
                        <?php if (!empty($file['width'])): ?>
                        <span><?= $file['width'] ?>×<?= $file['height'] ?></span>
                        <?php endif; ?>
                    </div>
                    <!-- BUG #4 FIX: Display uploader info -->
                    <div class="uploader-info">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                            <circle cx="12" cy="7" r="4"/>
                        </svg>
                        <span>By <?= htmlspecialchars($file['uploader_name']) ?></span>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <?php if ($filesData['pages'] > 1): ?>
        <div class="pagination">
            <?php for ($i = 1; $i <= $filesData['pages']; $i++): ?>
            <a href="?page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>">
                <?= $i ?>
            </a>
            <?php endfor; ?>
        </div>
        <?php endif; ?>

        <?php else: ?>
        <div class="empty-state">
            <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                <circle cx="8.5" cy="8.5" r="1.5"/>
                <polyline points="21 15 16 10 5 21"/>
            </svg>
            <h2>No images yet</h2>
            <p>Upload your first transparent PNG to get started!</p>
            <a href="/upload" class="btn btn-primary" style="margin-top: 1.5rem; display: inline-flex;">
                Upload PNG
            </a>
        </div>
        <?php endif; ?>
    </div>

    <script>
        function toggleDropdown() {
            document.getElementById('dropdownMenu').classList.toggle('active');
        }

        // Close dropdown when clicking outside
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-dropdown')) {
                document.getElementById('dropdownMenu').classList.remove('active');
            }
        });
    </script>
</body>
</html>
