<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/UserManager.php';

// Require login
Auth::requireLogin();

$currentUser = Auth::getCurrentUser();
$userManager = new UserManager();

// Get user ID from query
$userId = $_GET['id'] ?? null;

if (!$userId) {
    header('Location: /users');
    exit;
}

// Get user to edit
$userToEdit = $userManager->getUserById($userId);

if (!$userToEdit) {
    header('Location: /users?error=user_not_found');
    exit;
}

// Check permissions: Admin can edit anyone, user can only edit themselves
$isAdmin = Auth::isAdmin();
if (!$isAdmin && $userToEdit['id'] !== $currentUser['id']) {
    header('Location: /?error=permission_denied');
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!CSRF::validate($_POST['csrf_token'])) {
        $error = 'Invalid request';
    } else {
        $updateData = [
            'username' => trim($_POST['username']),
            'display_name' => trim($_POST['display_name']),
            'email' => trim($_POST['email'])
        ];
        
        // Only admin can change role
        if ($isAdmin && isset($_POST['role'])) {
            $updateData['role'] = $_POST['role'];
        }
        
        // Update password only if provided
        if (!empty($_POST['password'])) {
            $updateData['password'] = $_POST['password'];
        }
        
        $result = $userManager->updateUser($userId, $updateData);
        
        if ($result['success']) {
            // BUG #7 FIX: Refresh session if editing own profile
            if ($userId === $currentUser['id']) {
                $userManager->refreshSession($userId);
            }
            
            header('Location: /users?success=user_updated');
            exit;
        } else {
            $error = $result['message'];
        }
    }
}

$pageTitle = 'Edit User';
?>
<!DOCTYPE html>
<html lang="en" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle) ?> - PNG Gallery Ultimate</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #9333ea;
            --primary-dark: #7e22ce;
            --primary-light: #a855f7;
            --bg-dark: #0a0a0a;
            --bg-card: #1a1a1a;
            --bg-hover: #2a2a2a;
            --text-primary: #ffffff;
            --text-secondary: #a3a3a3;
            --border: #333333;
            --success: #10b981;
            --error: #ef4444;
        }

        [data-theme="light"] {
            --bg-dark: #f5f5f5;
            --bg-card: #ffffff;
            --bg-hover: #e5e5e5;
            --text-primary: #0a0a0a;
            --text-secondary: #525252;
            --border: #d4d4d4;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--bg-dark);
            color: var(--text-primary);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .logo {
            font-size: 1.5rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .nav-buttons {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .btn {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            border: none;
            background: var(--bg-hover);
            color: var(--text-primary);
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .btn:hover {
            background: var(--primary);
            transform: translateY(-2px);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
        }

        /* Main Content */
        .container {
            max-width: 800px;
            margin: 3rem auto;
            padding: 0 2rem;
        }

        .card {
            background: var(--bg-card);
            border-radius: 16px;
            padding: 3rem;
            border: 1px solid var(--border);
        }

        .card h1 {
            font-size: 2rem;
            margin-bottom: 0.5rem;
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .card p {
            color: var(--text-secondary);
            margin-bottom: 2rem;
        }

        /* Form */
        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: var(--text-secondary);
        }

        .form-group input,
        .form-group select {
            width: 100%;
            padding: 0.75rem 1rem;
            border-radius: 8px;
            border: 1px solid var(--border);
            background: var(--bg-hover);
            color: var(--text-primary);
            font-size: 1rem;
            transition: all 0.2s;
        }

        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(147, 51, 234, 0.1);
        }

        .form-hint {
            color: var(--text-secondary);
            font-size: 0.85rem;
            margin-top: 0.25rem;
        }

        /* Messages */
        .message {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .message.error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--error);
            color: var(--error);
        }

        /* Buttons */
        .form-actions {
            display: flex;
            gap: 1rem;
            margin-top: 2rem;
        }

        .btn-submit {
            flex: 1;
            padding: 1rem;
            font-size: 1.1rem;
        }

        @media (max-width: 768px) {
            .container {
                padding: 0 1rem;
            }

            .card {
                padding: 2rem 1.5rem;
            }

            .form-actions {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-content">
            <a href="/" class="logo">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                    <circle cx="8.5" cy="8.5" r="1.5"/>
                    <polyline points="21 15 16 10 5 21"/>
                </svg>
                PNG Gallery Ultimate
            </a>
            
            <div class="nav-buttons">
                <a href="/users" class="btn">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M15 19l-7-7 7-7"/>
                    </svg>
                    Back to Users
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="container">
        <div class="card">
            <h1>Edit User</h1>
            <p>Update user information</p>

            <?php if (isset($error)): ?>
                <div class="message error">
                    ✗ <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?= CSRF::generate() ?>">
                
                <div class="form-group">
                    <label for="username">Username *</label>
                    <input 
                        type="text" 
                        id="username" 
                        name="username" 
                        value="<?= htmlspecialchars($userToEdit['username']) ?>" 
                        required
                        pattern="[a-zA-Z0-9_]{3,20}"
                        title="3-20 characters, alphanumeric and underscore only"
                    >
                    <div class="form-hint">3-20 characters, alphanumeric and underscore only</div>
                </div>

                <div class="form-group">
                    <label for="display_name">Display Name *</label>
                    <input 
                        type="text" 
                        id="display_name" 
                        name="display_name" 
                        value="<?= htmlspecialchars($userToEdit['display_name']) ?>" 
                        required
                    >
                </div>

                <div class="form-group">
                    <label for="email">Email</label>
                    <input 
                        type="email" 
                        id="email" 
                        name="email" 
                        value="<?= htmlspecialchars($userToEdit['email']) ?>"
                    >
                </div>

                <?php if ($isAdmin): ?>
                <div class="form-group">
                    <label for="role">Role</label>
                    <select id="role" name="role">
                        <option value="user" <?= $userToEdit['role'] === 'user' ? 'selected' : '' ?>>User</option>
                        <option value="admin" <?= $userToEdit['role'] === 'admin' ? 'selected' : '' ?>>Admin</option>
                    </select>
                </div>
                <?php endif; ?>

                <div class="form-group">
                    <label for="password">New Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        minlength="6"
                    >
                    <div class="form-hint">Leave empty to keep current password. Minimum 6 characters.</div>
                </div>

                <div class="form-actions">
                    <a href="/users" class="btn">Cancel</a>
                    <button type="submit" class="btn btn-primary btn-submit">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/>
                            <polyline points="17 21 17 13 7 13 7 21"/>
                            <polyline points="7 3 7 8 15 8"/>
                        </svg>
                        Save Changes
                    </button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>
