<?php

class UserManager {
    private $usersFile;
    
    public function __construct() {
        $this->usersFile = __DIR__ . '/../storage/users/users.json';
        $this->ensureUsersFile();
    }
    
    private function ensureUsersFile() {
        if (!file_exists($this->usersFile)) {
            $defaultUser = [
                'id' => $this->generateId(),
                'username' => 'admin',
                'password' => password_hash('password', PASSWORD_DEFAULT),
                'display_name' => 'Administrator',
                'email' => 'admin@example.com',
                'role' => 'admin',
                'avatar' => null,
                'created_at' => date('Y-m-d H:i:s'),
                'uploads' => 0,  // BUG #3 FIX: Upload tracking
                'last_login' => null
            ];
            
            $this->saveUsers([$defaultUser]);
        }
    }
    
    private function generateId() {
        return bin2hex(random_bytes(16));
    }
    
    public function getUsers() {
        if (!file_exists($this->usersFile)) {
            return [];
        }
        
        $json = file_get_contents($this->usersFile);
        return json_decode($json, true) ?: [];
    }
    
    private function saveUsers($users) {
        $dir = dirname($this->usersFile);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        
        file_put_contents($this->usersFile, json_encode($users, JSON_PRETTY_PRINT));
    }
    
    public function getUserById($id) {
        $users = $this->getUsers();
        foreach ($users as $user) {
            if ($user['id'] === $id) {
                return $user;
            }
        }
        return null;
    }
    
    public function getUserByUsername($username) {
        $users = $this->getUsers();
        foreach ($users as $user) {
            if ($user['username'] === $username) {
                return $user;
            }
        }
        return null;
    }
    
    public function createUser($data) {
        // Validate required fields
        if (empty($data['username']) || empty($data['password'])) {
            return ['success' => false, 'message' => 'Username and password are required'];
        }
        
        // Check if username already exists
        if ($this->getUserByUsername($data['username'])) {
            return ['success' => false, 'message' => 'Username already exists'];
        }
        
        $users = $this->getUsers();
        
        $newUser = [
            'id' => $this->generateId(),
            'username' => $data['username'],
            'password' => password_hash($data['password'], PASSWORD_DEFAULT),
            'display_name' => $data['display_name'] ?? $data['username'],
            'email' => $data['email'] ?? '',
            'role' => $data['role'] ?? 'user',
            'avatar' => null,
            'created_at' => date('Y-m-d H:i:s'),
            'uploads' => 0,  // BUG #3 FIX: Initialize upload count
            'last_login' => null
        ];
        
        $users[] = $newUser;
        $this->saveUsers($users);
        
        return ['success' => true, 'message' => 'User created successfully', 'user' => $newUser];
    }
    
    public function updateUser($id, $data) {
        $users = $this->getUsers();
        $updated = false;
        
        foreach ($users as &$user) {
            if ($user['id'] === $id) {
                // Update fields if provided
                if (isset($data['username'])) {
                    // Check if new username already exists (excluding current user)
                    $existingUser = $this->getUserByUsername($data['username']);
                    if ($existingUser && $existingUser['id'] !== $id) {
                        return ['success' => false, 'message' => 'Username already exists'];
                    }
                    $user['username'] = $data['username'];
                }
                
                if (isset($data['display_name'])) $user['display_name'] = $data['display_name'];
                if (isset($data['email'])) $user['email'] = $data['email'];
                if (isset($data['role'])) $user['role'] = $data['role'];
                if (isset($data['avatar'])) $user['avatar'] = $data['avatar'];
                if (isset($data['uploads'])) $user['uploads'] = $data['uploads'];
                
                // Update password if provided and not empty
                if (!empty($data['password'])) {
                    $user['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
                }
                
                $updated = true;
                break;
            }
        }
        
        if ($updated) {
            $this->saveUsers($users);
            return ['success' => true, 'message' => 'User updated successfully'];
        }
        
        return ['success' => false, 'message' => 'User not found'];
    }
    
    public function deleteUser($id) {
        $users = $this->getUsers();
        $filtered = array_filter($users, function($user) use ($id) {
            return $user['id'] !== $id;
        });
        
        if (count($filtered) < count($users)) {
            $this->saveUsers(array_values($filtered));
            return ['success' => true, 'message' => 'User deleted successfully'];
        }
        
        return ['success' => false, 'message' => 'User not found'];
    }
    
    public function updateLastLogin($userId) {
        $user = $this->getUserById($userId);
        if ($user) {
            $user['last_login'] = date('Y-m-d H:i:s');
            $this->updateUser($userId, $user);
        }
    }
    
    // BUG #3 FIX: Increment upload count when user uploads PNG
    public function incrementUploadCount($userId) {
        $user = $this->getUserById($userId);
        if ($user) {
            $user['uploads'] = ($user['uploads'] ?? 0) + 1;
            $this->updateUser($userId, $user);
            return true;
        }
        return false;
    }
    
    // BUG #7 FIX: Refresh session after profile update
    public function refreshSession($userId) {
        $user = $this->getUserById($userId);
        if ($user) {
            unset($user['password']); // Don't store password in session
            $_SESSION['user'] = $user;
            return true;
        }
        return false;
    }
    
    public function getStats() {
        $users = $this->getUsers();
        
        $stats = [
            'total' => count($users),
            'admins' => 0,
            'users' => 0,
            'active_today' => 0,
            'total_uploads' => 0
        ];
        
        $today = date('Y-m-d');
        
        foreach ($users as $user) {
            if ($user['role'] === 'admin') {
                $stats['admins']++;
            } else {
                $stats['users']++;
            }
            
            if ($user['last_login'] && strpos($user['last_login'], $today) === 0) {
                $stats['active_today']++;
            }
            
            $stats['total_uploads'] += ($user['uploads'] ?? 0);
        }
        
        return $stats;
    }
    
    // Get user's upload count
    public function getUserUploadCount($userId) {
        $user = $this->getUserById($userId);
        return $user ? ($user['uploads'] ?? 0) : 0;
    }
}
