<?php

class PngHandler {
    private $uploadsDir;
    private $maxFileSize = 10485760; // 10MB
    
    public function __construct() {
        $this->uploadsDir = __DIR__ . '/../uploads';
        $this->ensureDirectories();
    }
    
    private function ensureDirectories() {
        if (!is_dir($this->uploadsDir)) {
            mkdir($this->uploadsDir, 0755, true);
        }
    }
    
    public function upload($file, $userId, $userName) {
        // Validate file
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            return ['success' => false, 'message' => 'No file uploaded'];
        }
        
        // Check file size
        if ($file['size'] > $this->maxFileSize) {
            return ['success' => false, 'message' => 'File too large (max 10MB)'];
        }
        
        // Check if it's a PNG
        $imageInfo = getimagesize($file['tmp_name']);
        if ($imageInfo === false || $imageInfo['mime'] !== 'image/png') {
            return ['success' => false, 'message' => 'Only PNG files are allowed'];
        }
        
        // Generate unique filename
        $originalName = pathinfo($file['name'], PATHINFO_FILENAME);
        $timestamp = time();
        $uniqueId = substr(md5(uniqid()), 0, 8);
        $newFilename = $this->sanitizeFilename($originalName) . '_' . $timestamp . '_' . $uniqueId . '.png';
        
        $targetPath = $this->uploadsDir . '/' . $newFilename;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
            return ['success' => false, 'message' => 'Failed to save file'];
        }
        
        // BUG #4 FIX: Save uploader information in metadata
        $metadata = [
            'original_name' => $originalName,
            'filename' => $newFilename,
            'size' => filesize($targetPath),
            'width' => $imageInfo[0],
            'height' => $imageInfo[1],
            'uploaded_at' => date('Y-m-d H:i:s'),
            'uploaded_by' => $userId,        // User ID who uploaded
            'uploader_name' => $userName,    // Display name for showing in UI
            'mime_type' => 'image/png'
        ];
        
        // Save metadata
        $metadataPath = $this->uploadsDir . '/' . $newFilename . '.json';
        file_put_contents($metadataPath, json_encode($metadata, JSON_PRETTY_PRINT));
        
        return [
            'success' => true,
            'message' => 'File uploaded successfully',
            'filename' => $newFilename,
            'metadata' => $metadata
        ];
    }
    
    private function sanitizeFilename($filename) {
        // Remove special characters
        $filename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $filename);
        // Limit length
        return substr($filename, 0, 50);
    }
    
    public function getFiles($page = 1, $perPage = 12) {
        $files = glob($this->uploadsDir . '/*.png');
        
        if ($files === false) {
            return ['files' => [], 'total' => 0, 'pages' => 0];
        }
        
        // Sort by modified time (newest first)
        usort($files, function($a, $b) {
            return filemtime($b) - filemtime($a);
        });
        
        $total = count($files);
        $pages = ceil($total / $perPage);
        
        // Paginate
        $offset = ($page - 1) * $perPage;
        $pagedFiles = array_slice($files, $offset, $perPage);
        
        // Get file info with uploader data
        $filesData = [];
        foreach ($pagedFiles as $file) {
            $filename = basename($file);
            $metadata = $this->getMetadata($filename);
            
            $filesData[] = [
                'filename' => $filename,
                'url' => '/uploads/' . $filename,
                'size' => filesize($file),
                'uploaded_at' => date('Y-m-d H:i:s', filemtime($file)),
                'uploader_name' => $metadata['uploader_name'] ?? 'Unknown',  // BUG #4 FIX
                'uploader_id' => $metadata['uploaded_by'] ?? null,
                'original_name' => $metadata['original_name'] ?? $filename,
                'width' => $metadata['width'] ?? null,
                'height' => $metadata['height'] ?? null
            ];
        }
        
        return [
            'files' => $filesData,
            'total' => $total,
            'pages' => $pages,
            'current_page' => $page
        ];
    }
    
    public function getMetadata($filename) {
        $metadataPath = $this->uploadsDir . '/' . $filename . '.json';
        
        if (file_exists($metadataPath)) {
            $json = file_get_contents($metadataPath);
            return json_decode($json, true) ?: [];
        }
        
        return [];
    }
    
    public function deleteFile($filename) {
        $filePath = $this->uploadsDir . '/' . $filename;
        $metadataPath = $filePath . '.json';
        
        if (!file_exists($filePath)) {
            return ['success' => false, 'message' => 'File not found'];
        }
        
        // Delete file
        if (!unlink($filePath)) {
            return ['success' => false, 'message' => 'Failed to delete file'];
        }
        
        // Delete metadata if exists
        if (file_exists($metadataPath)) {
            unlink($metadataPath);
        }
        
        return ['success' => true, 'message' => 'File deleted successfully'];
    }
    
    public function renameFile($oldFilename, $newName) {
        $oldPath = $this->uploadsDir . '/' . $oldFilename;
        
        if (!file_exists($oldPath)) {
            return ['success' => false, 'message' => 'File not found'];
        }
        
        // Sanitize new name and add timestamp
        $newName = $this->sanitizeFilename($newName);
        $timestamp = time();
        $uniqueId = substr(md5(uniqid()), 0, 8);
        $newFilename = $newName . '_' . $timestamp . '_' . $uniqueId . '.png';
        
        $newPath = $this->uploadsDir . '/' . $newFilename;
        
        // Rename file
        if (!rename($oldPath, $newPath)) {
            return ['success' => false, 'message' => 'Failed to rename file'];
        }
        
        // Update metadata filename
        $oldMetadataPath = $oldPath . '.json';
        $newMetadataPath = $newPath . '.json';
        
        if (file_exists($oldMetadataPath)) {
            $metadata = json_decode(file_get_contents($oldMetadataPath), true);
            $metadata['filename'] = $newFilename;
            $metadata['original_name'] = $newName;
            file_put_contents($newMetadataPath, json_encode($metadata, JSON_PRETTY_PRINT));
            unlink($oldMetadataPath);
        }
        
        return [
            'success' => true,
            'message' => 'File renamed successfully',
            'new_filename' => $newFilename
        ];
    }
    
    public function getStorageInfo() {
        $files = glob($this->uploadsDir . '/*.png');
        $totalSize = 0;
        
        if ($files) {
            foreach ($files as $file) {
                $totalSize += filesize($file);
            }
        }
        
        return [
            'total_files' => count($files ?: []),
            'total_size' => $totalSize,
            'total_size_mb' => round($totalSize / 1048576, 2)
        ];
    }
    
    // Get files uploaded by specific user
    public function getFilesByUser($userId) {
        $files = glob($this->uploadsDir . '/*.png');
        $userFiles = [];
        
        if ($files) {
            foreach ($files as $file) {
                $filename = basename($file);
                $metadata = $this->getMetadata($filename);
                
                if (isset($metadata['uploaded_by']) && $metadata['uploaded_by'] === $userId) {
                    $userFiles[] = [
                        'filename' => $filename,
                        'url' => '/uploads/' . $filename,
                        'uploaded_at' => $metadata['uploaded_at'] ?? date('Y-m-d H:i:s', filemtime($file)),
                        'original_name' => $metadata['original_name'] ?? $filename
                    ];
                }
            }
        }
        
        return $userFiles;
    }
}
