<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/UserManager.php';

Auth::requireLogin();

$userManager = new UserManager();
$username = Auth::getUsername();
$currentUser = $userManager->getUser($username);

// Fallback if session user not found for safety
if (!$currentUser) {
    Auth::logout();
    header('Location: login.php');
    exit;
}

$profileData = $currentUser; // use single source of truth (UserManager)
$loginHistory = $userManager->getLoginHistory($username, 10);
$activityLog = $userManager->getActivityLog($username, 15);

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    CSRF::verifyOrDie();

    $action = $_POST['action'] ?? '';

    switch ($action) {
        case 'update_profile':
            // Only allow updating user's own profile
            $updateData = [
                'display_name' => trim($_POST['display_name'] ?? ''),
                'bio' => trim($_POST['bio'] ?? ''),
                'email' => trim($_POST['email'] ?? ''),
            ];

            // Basic validation (you can expand as needed)
            if (empty($updateData['display_name'])) {
                $error = 'Display name cannot be empty';
            } elseif (!empty($updateData['email']) && !filter_var($updateData['email'], FILTER_VALIDATE_EMAIL)) {
                $error = 'Invalid email address';
            } else {
                $res = $userManager->updateUser($username, $updateData);
                if (isset($res['success']) && $res['success'] === true) {
                    $userManager->logActivity($username, 'profile_updated', 'Updated profile information');
                    $message = 'Profile updated successfully!';
                } elseif ($res === true) {
                    // backward compatibility if updateUser returns true
                    $userManager->logActivity($username, 'profile_updated', 'Updated profile information');
                    $message = 'Profile updated successfully!';
                } else {
                    $error = $res['error'] ?? 'Failed to update profile';
                }
            }

            // refresh current user data
            $currentUser = $userManager->getUser($username);
            $profileData = $currentUser;
            break;

        case 'change_password':
            $old = $_POST['old_password'] ?? '';
            $new = $_POST['new_password'] ?? '';

            if (empty($old) || empty($new)) {
                $error = 'Please provide both current and new password';
            } elseif (strlen($new) < 6) {
                $error = 'New password must be at least 6 characters';
            } else {
                if (!$userManager->verifyCredentials($username, $old)) {
                    $error = 'Current password is incorrect';
                } else {
                    $res = $userManager->changePassword($username, $new);
                    if (isset($res['success']) && $res['success'] === true) {
                        $userManager->logActivity($username, 'password_changed', 'Password changed successfully');
                        $message = 'Password changed successfully!';
                    } else {
                        $error = $res['error'] ?? 'Failed to change password';
                    }
                }
            }
            break;

        case 'change_username':
            $newUsername = $_POST['new_username'] ?? '';
            $password = $_POST['password'] ?? '';

            if (!$newUsername) {
                $error = 'New username cannot be empty';
            } elseif (!preg_match('/^[a-zA-Z0-9_-]{3,20}$/', $newUsername)) {
                $error = 'Username: 3-20 chars, letters/numbers/underscore/dash only';
            } else {
                if (!$userManager->verifyCredentials($username, $password)) {
                    $error = 'Current password is incorrect';
                } else {
                    $result = $userManager->changeUsername($username, $newUsername);
                    if (isset($result['success'])) {
                        // Update session values
                        $_SESSION['username'] = $newUsername;
                        $_SESSION['display_name'] = $currentUser['display_name'] ?? $newUsername;

                        // Refresh local variables
                        $username = $newUsername;
                        $currentUser = $userManager->getUser($username);
                        $profileData = $currentUser;

                        $message = 'Username changed successfully! You are now @' . htmlspecialchars($newUsername);

                        // Redirect to avoid form resubmit and to refresh URLs
                        header('Location: settings.php?success=username_changed');
                        exit;
                    } else {
                        $error = $result['error'] ?? 'Failed to change username';
                    }
                }
            }
            break;

        case 'upload_avatar':
            if (isset($_FILES['avatar'])) {
                $result = $userManager->uploadAvatar($username, $_FILES['avatar']);
                if (isset($result['success'])) {
                    $userManager->logActivity($username, 'avatar_uploaded', 'Profile picture updated');
                    $message = 'Avatar uploaded successfully!';
                } else {
                    $error = $result['error'] ?? 'Failed to upload avatar';
                }
            } else {
                $error = 'No file uploaded';
            }

            // refresh
            $currentUser = $userManager->getUser($username);
            $profileData = $currentUser;
            break;

        case 'delete_avatar':
            $ok = $userManager->deleteAvatar($username);
            if ($ok) {
                $userManager->logActivity($username, 'avatar_deleted', 'Profile picture deleted');
                $message = 'Avatar deleted successfully!';
            } else {
                $error = 'Failed to delete avatar';
            }

            $currentUser = $userManager->getUser($username);
            $profileData = $currentUser;
            break;
    }
}

// Check for success message from redirect
if (isset($_GET['success']) && $_GET['success'] === 'username_changed') {
    $message = 'Username changed successfully!';
}
?>
<!DOCTYPE html>
<html lang="id" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - Galeri PNG PRO</title>
    <meta name="robots" content="noindex">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --primary-darker: #6D28D9;
            --secondary: #C4B5FD;
            --accent: #A78BFA;
            
            --bg-primary: #FAF5FF;
            --bg-secondary: #F5F3FF;
            --bg-card: #FFFFFF;
            --bg-hover: #F3F4F6;
            
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --text-tertiary: #9CA3AF;
            
            --border-color: #E5E7EB;
            --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            --shadow-xl: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
            
            --success: #10B981;
            --error: #EF4444;
            --warning: #F59E0B;
            --info: #3B82F6;
        }
        
        [data-theme="dark"] {
            --primary: #A78BFA;
            --primary-dark: #8B5CF6;
            --primary-darker: #7C3AED;
            
            --bg-primary: #0F0B1F;
            --bg-secondary: #1E1B4B;
            --bg-card: #312E81;
            --bg-hover: #3730A3;
            
            --text-primary: #F9FAFB;
            --text-secondary: #D1D5DB;
            --text-tertiary: #9CA3AF;
            
            --border-color: #4C1D95;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(180deg, var(--bg-primary) 0%, var(--bg-secondary) 100%);
            color: var(--text-primary);
            min-height: 100vh;
            transition: all 0.3s ease;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            flex-wrap: wrap;
            gap: 1rem;
        }
        
        .header-left {
            flex: 1;
        }
        
        h1 {
            font-size: 2.5rem;
            font-weight: 800;
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-darker) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 0.5rem;
        }
        
        .subtitle {
            color: var(--text-secondary);
            font-size: 1rem;
        }
        
        .header-actions {
            display: flex;
            gap: 0.75rem;
            align-items: center;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            border: none;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            text-decoration: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-darker) 100%);
            color: white;
        }
        
        .btn-secondary {
            background: var(--bg-card);
            color: var(--text-primary);
            border: 1px solid var(--border-color);
        }
        
        .btn-danger {
            background: var(--error);
            color: white;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }
        
        .theme-toggle {
            width: 44px;
            height: 44px;
            border-radius: 12px;
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .theme-toggle:hover {
            transform: translateY(-2px);
            border-color: var(--primary);
        }
        
        .grid {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 2rem;
        }
        
        @media (max-width: 968px) {
            .grid {
                grid-template-columns: 1fr;
            }
        }
        
        /* SIDEBAR */
        .sidebar {
            background: var(--bg-card);
            border-radius: 20px;
            padding: 1.5rem;
            border: 1px solid var(--border-color);
            height: fit-content;
            position: sticky;
            top: 2rem;
        }
        
        .nav-item {
            padding: 1rem;
            border-radius: 12px;
            margin-bottom: 0.5rem;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-weight: 500;
            color: var(--text-secondary);
        }
        
        .nav-item:hover {
            background: var(--bg-hover);
            color: var(--text-primary);
        }
        
        .nav-item.active {
            background: var(--primary);
            color: white;
        }
        
        .nav-item svg {
            width: 20px;
            height: 20px;
        }
        
        /* MAIN CONTENT */
        .main-content {
            display: none;
        }
        
        .main-content.active {
            display: block;
        }
        
        .card {
            background: var(--bg-card);
            border-radius: 20px;
            padding: 2rem;
            margin-bottom: 1.5rem;
            border: 1px solid var(--border-color);
            box-shadow: var(--shadow);
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--border-color);
        }
        
        .card-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-primary);
        }
        
        .card-description {
            color: var(--text-secondary);
            margin-top: 0.5rem;
            font-size: 0.875rem;
        }
        
        /* PROFILE SECTION */
        .profile-header {
            display: flex;
            align-items: center;
            gap: 2rem;
            margin-bottom: 2rem;
        }
        
        .avatar-container {
            position: relative;
        }
        
        .avatar-large {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            object-fit: cover;
            border: 4px solid var(--primary);
            box-shadow: 0 8px 16px rgba(139, 92, 246, 0.3);
        }
        
        .avatar-badge {
            position: absolute;
            bottom: 5px;
            right: 5px;
            background: var(--success);
            width: 20px;
            height: 20px;
            border-radius: 50%;
            border: 3px solid var(--bg-card);
        }
        
        .profile-info h2 {
            font-size: 1.75rem;
            font-weight: 700;
            margin-bottom: 0.25rem;
        }
        
        .profile-username {
            color: var(--text-secondary);
            font-size: 1rem;
            margin-bottom: 0.5rem;
        }
        
        .profile-meta {
            display: flex;
            gap: 1.5rem;
            margin-top: 1rem;
        }
        
        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--text-secondary);
            font-size: 0.875rem;
        }
        
        .meta-item svg {
            width: 16px;
            height: 16px;
        }
        
        /* FORM STYLES */
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--text-primary);
            font-size: 0.875rem;
        }
        
        input[type="text"],
        input[type="email"],
        input[type="password"],
        textarea {
            width: 100%;
            padding: 0.875rem 1rem;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: var(--bg-card);
            color: var(--text-primary);
            font-family: 'Inter', sans-serif;
        }
        
        input:focus,
        textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 4px rgba(139, 92, 246, 0.1);
        }
        
        textarea {
            resize: vertical;
            min-height: 100px;
        }
        
        .input-group {
            position: relative;
        }
        
        .input-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-tertiary);
        }
        
        .input-icon + input {
            padding-left: 2.75rem;
        }
        
        /* ALERT */
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-weight: 500;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
            border: 2px solid var(--success);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--error);
            border: 2px solid var(--error);
        }
        
        .alert svg {
            width: 20px;
            height: 20px;
        }
        
        /* LOGIN HISTORY */
        .history-item {
            padding: 1rem;
            background: var(--bg-hover);
            border-radius: 12px;
            margin-bottom: 0.75rem;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .history-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            background: var(--primary);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .history-icon svg {
            width: 20px;
            height: 20px;
        }
        
        .history-details {
            flex: 1;
        }
        
        .history-title {
            font-weight: 600;
            margin-bottom: 0.25rem;
        }
        
        .history-meta {
            font-size: 0.8rem;
            color: var(--text-secondary);
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }
        
        .history-time {
            color: var(--text-tertiary);
            font-size: 0.75rem;
        }
        
        /* ACTIVITY LOG */
        .activity-item {
            display: flex;
            gap: 1rem;
            margin-bottom: 1.25rem;
            padding-bottom: 1.25rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .activity-item:last-child {
            border-bottom: none;
            padding-bottom: 0;
            margin-bottom: 0;
        }
        
        .activity-icon {
            width: 36px;
            height: 36px;
            border-radius: 8px;
            background: var(--bg-hover);
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--primary);
            flex-shrink: 0;
        }
        
        .activity-content {
            flex: 1;
        }
        
        .activity-title {
            font-weight: 600;
            margin-bottom: 0.25rem;
        }
        
        .activity-description {
            font-size: 0.875rem;
            color: var(--text-secondary);
            margin-bottom: 0.25rem;
        }
        
        .activity-time {
            font-size: 0.75rem;
            color: var(--text-tertiary);
        }
        
        /* STATS */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1.5rem;
        }
        
        .stat-card {
            background: var(--bg-hover);
            padding: 1.5rem;
            border-radius: 16px;
            text-align: center;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 800;
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-darker) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .stat-label {
            color: var(--text-secondary);
            font-size: 0.875rem;
            margin-top: 0.5rem;
        }
        
        .badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .badge-primary {
            background: rgba(139, 92, 246, 0.1);
            color: var(--primary);
        }
        
        .badge-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }
        
        .empty-state {
            text-align: center;
            padding: 3rem 1rem;
            color: var(--text-secondary);
        }
        
        .empty-icon {
            width: 64px;
            height: 64px;
            margin: 0 auto 1rem;
            opacity: 0.5;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="header-left">
                <h1>⚙️ Settings</h1>
                <p class="subtitle">Manage your account preferences and security</p>
            </div>
            <div class="header-actions">
                <button class="theme-toggle" onclick="toggleDarkMode()" title="Toggle theme">
                    <svg id="theme-icon-sun" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="5"/><line x1="12" y1="1" x2="12" y2="3"/>
                        <line x1="12" y1="21" x2="12" y2="23"/><line x1="4.22" y1="4.22" x2="5.64" y2="5.64"/>
                        <line x1="18.36" y1="18.36" x2="19.78" y2="19.78"/><line x1="1" y1="12" x2="3" y2="12"/>
                        <line x1="21" y1="12" x2="23" y2="12"/><line x1="4.22" y1="19.78" x2="5.64" y2="18.36"/>
                        <line x1="18.36" y1="5.64" x2="19.78" y2="4.22"/>
                    </svg>
                    <svg id="theme-icon-moon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display: none;">
                        <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"/>
                    </svg>
                </button>
                <a href="index.php" class="btn btn-secondary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                        <path d="M19 12H5M12 19l-7-7 7-7"/>
                    </svg>
                    Back to Gallery
                </a>
            </div>
        </div>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                    <polyline points="22 4 12 14.01 9 11.01"/>
                </svg>
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="10"/>
                    <line x1="15" y1="9" x2="9" y2="15"/>
                    <line x1="9" y1="9" x2="15" y2="15"/>
                </svg>
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <div class="grid">
            <!-- SIDEBAR NAVIGATION -->
            <div class="sidebar">
                <div class="nav-item active" onclick="showTab('profile')">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                        <circle cx="12" cy="7" r="4"/>
                    </svg>
                    Profile
                </div>
                <div class="nav-item" onclick="showTab('account')">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="3"/>
                        <path d="M12 1v6m0 6v6M5.64 5.64l4.24 4.24m4.24 4.24l4.24 4.24M1 12h6m6 0h6M5.64 18.36l4.24-4.24m4.24-4.24l4.24-4.24"/>
                    </svg>
                    Account Security
                </div>
                <div class="nav-item" onclick="showTab('sessions')">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                        <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                    </svg>
                    Login History
                </div>
                <div class="nav-item" onclick="showTab('activity')">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/>
                    </svg>
                    Activity Log
                </div>
            </div>

            <!-- MAIN CONTENT -->
            <div>
                <!-- PROFILE TAB -->
                <div id="tab-profile" class="main-content active">
                    <div class="card">
                        <div class="profile-header">
                            <div class="avatar-container">
                                <img src="profile-avatar.php?t=<?php echo time(); ?>" alt="Avatar" class="avatar-large" id="avatar-preview">
                                <div class="avatar-badge"></div>
                            </div>
                            <div class="profile-info">
                                <h2><?php echo htmlspecialchars($currentUser['display_name']); ?></h2>
                                <p class="profile-username">@<?php echo htmlspecialchars($username); ?></p>
                                <div class="profile-meta">
                                    <div class="meta-item">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                                            <line x1="16" y1="2" x2="16" y2="6"/>
                                            <line x1="8" y1="2" x2="8" y2="6"/>
                                            <line x1="3" y1="10" x2="21" y2="10"/>
                                        </svg>
                                        Joined <?php echo date('M Y', $currentUser['created_at']); ?>
                                    </div>
                                    <div class="meta-item">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                            <polyline points="17 8 12 3 7 8"/>
                                            <line x1="12" y1="3" x2="12" y2="15"/>
                                        </svg>
                                        <?php echo $currentUser['total_uploads']; ?> uploads
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="card-header">
                            <div>
                                <h3 class="card-title">Profile Picture</h3>
                                <p class="card-description">Update your profile photo (PNG or JPG, max 2MB)</p>
                            </div>
                        </div>

                        <form method="POST" enctype="multipart/form-data" id="avatar-form">
                            <?php echo CSRF::field(); ?>
                            <input type="hidden" name="action" value="upload_avatar">
                            <input type="file" name="avatar" id="avatar-input" accept="image/png,image/jpeg" style="display: none;">
                            <div style="display: flex; gap: 0.75rem;">
                                <button type="button" onclick="document.getElementById('avatar-input').click()" class="btn btn-primary">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                        <polyline points="17 8 12 3 7 8"/>
                                        <line x1="12" y1="3" x2="12" y2="15"/>
                                    </svg>
                                    Upload Photo
                                </button>
                                <?php if ($profileData['has_avatar']): ?>
                                    <button type="button" onclick="deleteAvatar()" class="btn btn-danger">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                            <polyline points="3 6 5 6 21 6"/>
                                            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
                                        </svg>
                                        Delete
                                    </button>
                                <?php endif; ?>
                            </div>
                        </form>
                    </div>

                    <div class="card">
                        <div class="card-header">
                            <div>
                                <h3 class="card-title">Profile Information</h3>
                                <p class="card-description">Update your personal details</p>
                            </div>
                        </div>

                        <form method="POST">
                            <?php echo CSRF::field(); ?>
                            <input type="hidden" name="action" value="update_profile">

                            <div class="form-group">
                                <label>Display Name</label>
                                <input type="text" name="display_name" value="<?php echo htmlspecialchars($profileData['display_name']); ?>" required>
                            </div>

                            <div class="form-group">
                                <label>Bio</label>
                                <textarea name="bio" rows="3" placeholder="Tell us about yourself..."><?php echo htmlspecialchars($profileData['bio']); ?></textarea>
                            </div>

                            <button type="submit" class="btn btn-primary">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/>
                                    <polyline points="17 21 17 13 7 13 7 21"/>
                                    <polyline points="7 3 7 8 15 8"/>
                                </svg>
                                Save Changes
                            </button>
                        </form>
                    </div>
                </div>

                <!-- ACCOUNT SECURITY TAB -->
                <div id="tab-account" class="main-content">
                    <div class="card">
                        <div class="card-header">
                            <div>
                                <h3 class="card-title">Change Username</h3>
                                <p class="card-description">Update your unique username</p>
                            </div>
                        </div>

                        <button type="button" onclick="changeUsername()" class="btn btn-primary">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
                                <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
                            </svg>
                            Change Username
                        </button>
                    </div>

                    <div class="card">
                        <div class="card-header">
                            <div>
                                <h3 class="card-title">Change Password</h3>
                                <p class="card-description">Keep your account secure with a strong password</p>
                            </div>
                        </div>

                        <form method="POST">
                            <?php echo CSRF::field(); ?>
                            <input type="hidden" name="action" value="change_password">

                            <div class="form-group">
                                <label>Current Password</label>
                                <input type="password" name="old_password" required autocomplete="current-password">
                            </div>

                            <div class="form-group">
                                <label>New Password</label>
                                <input type="password" name="new_password" minlength="6" required autocomplete="new-password">
                                <small style="color: var(--text-secondary); font-size: 0.8rem;">Minimum 6 characters</small>
                            </div>

                            <button type="submit" class="btn btn-primary">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                                    <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                                </svg>
                                Update Password
                            </button>
                        </form>
                    </div>

                    <div class="card">
                        <div class="card-header">
                            <div>
                                <h3 class="card-title">Account Status</h3>
                                <p class="card-description">Current account information</p>
                            </div>
                        </div>

                        <div class="stats-grid">
                            <div class="stat-card">
                                <div class="stat-value"><?php echo $currentUser['total_logins'] ?? 0; ?></div>
                                <div class="stat-label">Total Logins</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-value"><?php echo $currentUser['total_uploads']; ?></div>
                                <div class="stat-label">Total Uploads</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-value">
                                    <span class="badge badge-success"><?php echo ucfirst($currentUser['status']); ?></span>
                                </div>
                                <div class="stat-label">Status</div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- LOGIN HISTORY TAB -->
                <div id="tab-sessions" class="main-content">
                    <div class="card">
                        <div class="card-header">
                            <div>
                                <h3 class="card-title">Login History</h3>
                                <p class="card-description">Recent login activity and devices</p>
                            </div>
                        </div>

                        <?php if (!empty($loginHistory)): ?>
                            <?php foreach ($loginHistory as $index => $login): ?>
                                <div class="history-item">
                                    <div class="history-icon">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <rect x="2" y="7" width="20" height="14" rx="2" ry="2"/>
                                            <path d="M16 21V5a2 2 0 0 0-2-2h-4a2 2 0 0 0-2 2v16"/>
                                        </svg>
                                    </div>
                                    <div class="history-details">
                                        <div class="history-title">
                                            <?php echo htmlspecialchars($login['browser']); ?> on <?php echo htmlspecialchars($login['os']); ?>
                                            <?php if ($index === 0): ?>
                                                <span class="badge badge-success">Current</span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="history-meta">
                                            <span>📍 <?php echo htmlspecialchars($login['ip']); ?></span>
                                            <span>🕒 <?php echo date('M d, Y H:i', $login['timestamp']); ?></span>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="empty-state">
                                <svg class="empty-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                                </svg>
                                <p>No login history available</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- ACTIVITY LOG TAB -->
                <div id="tab-activity" class="main-content">
                    <div class="card">
                        <div class="card-header">
                            <div>
                                <h3 class="card-title">Activity Log</h3>
                                <p class="card-description">Recent account activities and changes</p>
                            </div>
                        </div>

                        <?php if (!empty($activityLog)): ?>
                            <?php foreach ($activityLog as $activity): ?>
                                <div class="activity-item">
                                    <div class="activity-icon">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <circle cx="12" cy="12" r="10"/>
                                            <polyline points="12 6 12 12 16 14"/>
                                        </svg>
                                    </div>
                                    <div class="activity-content">
                                        <div class="activity-title"><?php echo ucwords(str_replace('_', ' ', $activity['action'])); ?></div>
                                        <div class="activity-description">
                                            <?php echo htmlspecialchars($activity['description']); ?>
                                            <?php if (!empty($activity['ip']) && $activity['ip'] !== 'Unknown'): ?>
                                                · IP: <?php echo htmlspecialchars($activity['ip']); ?>
                                            <?php endif; ?>
                                        </div>
                                        <div class="activity-time"><?php echo date('M d, Y H:i', $activity['timestamp']); ?></div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="empty-state">
                                <svg class="empty-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/>
                                </svg>
                                <p>No activity recorded yet</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Tab Navigation
        function showTab(tabName) {
            // Hide all tabs
            document.querySelectorAll('.main-content').forEach(tab => {
                tab.classList.remove('active');
            });

            // Remove active from all nav items
            document.querySelectorAll('.nav-item').forEach(item => {
                item.classList.remove('active');
            });

            // Show selected tab
            document.getElementById('tab-' + tabName).classList.add('active');

            // Highlight nav item
            event.currentTarget.classList.add('active');
        }

        // Theme Toggle
        function toggleDarkMode() {
            const html = document.documentElement;
            const currentTheme = html.getAttribute('data-theme');
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';

            html.setAttribute('data-theme', newTheme);
            localStorage.setItem('theme', newTheme);

            const sunIcon = document.getElementById('theme-icon-sun');
            const moonIcon = document.getElementById('theme-icon-moon');

            if (newTheme === 'dark') {
                sunIcon.style.display = 'none';
                moonIcon.style.display = 'block';
            } else {
                sunIcon.style.display = 'block';
                moonIcon.style.display = 'none';
            }
        }

        // Initialize theme
        (function() {
            const savedTheme = localStorage.getItem('theme') || 'light';
            document.documentElement.setAttribute('data-theme', savedTheme);

            const sunIcon = document.getElementById('theme-icon-sun');
            const moonIcon = document.getElementById('theme-icon-moon');

            if (savedTheme === 'dark') {
                sunIcon.style.display = 'none';
                moonIcon.style.display = 'block';
            }
        })();

        // Avatar Upload
        document.getElementById('avatar-input').addEventListener('change', function() {
            if (this.files.length > 0) {
                document.getElementById('avatar-form').submit();
            }
        });

        // Delete Avatar
        function deleteAvatar() {
            Swal.fire({
                title: '<strong>Delete Profile Picture?</strong>',
                text: 'Your avatar will be removed from your profile',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#EF4444',
                confirmButtonText: 'Delete',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = '<?php echo CSRF::field(); ?><input type="hidden" name="action" value="delete_avatar">';
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }

        // Change Username
        function changeUsername() {
            Swal.fire({
                title: '<strong>Change Username</strong>',
                html: `
                    <div style="text-align: left; margin: 1rem 0;">
                        <p style="color: #6B7280; margin-bottom: 1rem;">Current username: <strong>@<?php echo htmlspecialchars($username); ?></strong></p>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">New Username</label>
                            <input id="new-username" class="swal2-input" placeholder="username" style="width: 100%; margin: 0;">
                            <small style="color: #6B7280; font-size: 0.75rem;">3-20 characters, letters, numbers, underscore, dash only</small>
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Confirm Password</label>
                            <input id="confirm-password" type="password" class="swal2-input" placeholder="Your password" style="width: 100%; margin: 0;">
                        </div>
                        <div style="background: #FEF3C7; border: 2px solid #F59E0B; border-radius: 8px; padding: 0.75rem; margin-top: 1rem;">
                            <strong style="color: #92400E;">⚠️ Important:</strong>
                            <p style="color: #92400E; font-size: 0.875rem; margin-top: 0.25rem;">Changing your username will update your profile URL and all references. This cannot be undone!</p>
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: 'Change Username',
                confirmButtonColor: '#8B5CF6',
                cancelButtonText: 'Cancel',
                preConfirm: () => {
                    const newUsername = document.getElementById('new-username').value;
                    const password = document.getElementById('confirm-password').value;

                    if (!newUsername) {
                        Swal.showValidationMessage('Please enter a new username');
                        return false;
                    }

                    if (!/^[a-zA-Z0-9_-]{3,20}$/.test(newUsername)) {
                        Swal.showValidationMessage('Invalid username format');
                        return false;
                    }

                    if (!password) {
                        Swal.showValidationMessage('Please enter your password');
                        return false;
                    }

                    return { newUsername, password };
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="change_username">
                        <input type="hidden" name="new_username" value="${result.value.newUsername}">
                        <input type="hidden" name="password" value="${result.value.password}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }
    </script>
</body>
</html>
