<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/PngHandler.php';
require_once __DIR__ . '/../includes/UserManager.php';

// Require login
Auth::requireLogin();

// Get current user
$currentUser = Auth::getCurrentUser();
$isAdmin = Auth::isAdmin();
$userManager = new UserManager();

// Get pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 12;

// Get paginated images
$handler = new PngHandler();
$pagination = $handler->getPaginatedImages($page, $perPage);
$images = $pagination['images'];

// Get all images for stats
$allImages = $handler->getAllImages();

// Get user stats if admin
$userStats = $isAdmin ? $userManager->getUserStats() : null;
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Galeri Yesus V1 Created By Immortaldev</title>
    <meta name="robots" content="noindex">    
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link href="https://fonts.googleapis.com/css2?family=Questrial&display=swap" rel="stylesheet">

    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --primary-darker: #6D28D9;
            --secondary: #C4B5FD;
            --accent: #A78BFA;
            
            --bg-primary: #FAF5FF;
            --bg-secondary: #F5F3FF;
            --bg-card: #FFFFFF;
            --bg-hover: #F3F4F6;
            
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --text-tertiary: #9CA3AF;
            
            --border-color: #E5E7EB;
            --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            --shadow-xl: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
            
            --gradient-primary: linear-gradient(135deg, #8B5CF6 0%, #7C3AED 50%, #6D28D9 100%);
            --gradient-bg: linear-gradient(180deg, #FAF5FF 0%, #F5F3FF 100%);
            
            --checkerboard-light: #F3F4F6;
            --checkerboard-dark: #E5E7EB;
        }
        
        [data-theme="dark"] {
            --primary: #A78BFA;
            --primary-dark: #8B5CF6;
            --primary-darker: #7C3AED;
            --secondary: #6D28D9;
            --accent: #8B5CF6;
            
            --bg-primary: #0F0B1F;
            --bg-secondary: #1E1B4B;
            --bg-card: #312E81;
            --bg-hover: #3730A3;
            
            --text-primary: #F9FAFB;
            --text-secondary: #D1D5DB;
            --text-tertiary: #9CA3AF;
            
            --border-color: #4C1D95;
            --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.4);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.5);
            --shadow-xl: 0 20px 25px -5px rgba(0, 0, 0, 0.6);
            
            --gradient-primary: linear-gradient(135deg, #A78BFA 0%, #8B5CF6 50%, #7C3AED 100%);
            --gradient-bg: linear-gradient(180deg, #0F0B1F 0%, #1E1B4B 100%);
            
            --checkerboard-light: #4C1D95;
            --checkerboard-dark: #3730A3;
        }
        
        body {
            font-family: 'Questrial', sans-serif;
            background: var(--gradient-bg);
            color: var(--text-primary);
            min-height: 100vh;
            transition: all 0.3s ease;
        }
        
        /* HEADER WITH MULTI-USER */
        .header {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border-bottom: 1px solid var(--border-color);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        [data-theme="dark"] .header {
            background: rgba(30, 27, 75, 0.8);
        }
        
        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 1rem;
            flex-wrap: wrap;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1.5rem;
            font-weight: 700;
            background: var(--gradient-primary);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .logo svg {
            width: 32px;
            height: 32px;
            filter: drop-shadow(0 2px 4px rgba(139, 92, 246, 0.3));
        }
        
        .header-actions {
            display: flex;
            gap: 0.75rem;
            align-items: center;
        }
        
        .theme-toggle {
            width: 44px;
            height: 44px;
            border-radius: 12px;
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .theme-toggle:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
            border-color: var(--primary);
        }
        
        .theme-toggle svg {
            width: 20px;
            height: 20px;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.875rem;
            transition: all 0.3s ease;
            cursor: pointer;
            border: none;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            white-space: nowrap;
        }
        
        .btn-primary {
            background: var(--gradient-primary);
            color: white;
            box-shadow: var(--shadow);
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-xl);
        }
        
        .btn-secondary {
            background: var(--bg-card);
            color: var(--text-primary);
            border: 1px solid var(--border-color);
        }
        
        .btn-secondary:hover {
            background: var(--bg-hover);
            border-color: var(--primary);
        }
        
        .btn svg {
            width: 18px;
            height: 18px;
        }
        
        /* USER PROFILE DROPDOWN */
        .user-dropdown {
            position: relative;
        }
        
        .user-trigger {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.5rem 1rem;
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .user-trigger:hover {
            border-color: var(--primary);
            box-shadow: var(--shadow);
        }
        
        .user-avatar {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--primary);
        }
        
        .user-info {
            display: flex;
            flex-direction: column;
            align-items: flex-start;
        }
        
        .user-name {
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-primary);
        }
        
        .user-role {
            font-size: 0.75rem;
            color: var(--text-tertiary);
        }
        
        .dropdown-menu {
            position: absolute;
            top: calc(100% + 0.5rem);
            right: 0;
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            box-shadow: var(--shadow-xl);
            min-width: 200px;
            display: none;
            z-index: 1000;
        }
        
        .dropdown-menu.active {
            display: block;
            animation: slideDown 0.3s ease;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: var(--text-primary);
            text-decoration: none;
            transition: all 0.2s ease;
            border-bottom: 1px solid var(--border-color);
        }
        
        .dropdown-item:last-child {
            border-bottom: none;
        }
        
        .dropdown-item:hover {
            background: var(--bg-hover);
        }
        
        .dropdown-item svg {
            width: 18px;
            height: 18px;
        }
        
        /* Container */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        /* Stats Cards */
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: rgba(255, 255, 255, 0.7);
            backdrop-filter: blur(20px);
            border: 1px solid var(--border-color);
            border-radius: 20px;
            padding: 1.5rem;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        [data-theme="dark"] .stat-card {
            background: rgba(49, 46, 129, 0.5);
        }
        
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: var(--gradient-primary);
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-xl);
            border-color: var(--primary);
        }
        
        .stat-card:hover::before {
            opacity: 1;
        }
        
        .stat-header {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1rem;
        }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            background: var(--gradient-primary);
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: var(--shadow);
        }
        
        .stat-icon svg {
            width: 24px;
            height: 24px;
            color: white;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            background: var(--gradient-primary);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            line-height: 1;
        }
        
        .stat-label {
            color: var(--text-secondary);
            font-size: 0.875rem;
            font-weight: 500;
            margin-top: 0.5rem;
        }
        
        /* Gallery Grid */
        .gallery-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .gallery-item {
            background: var(--bg-card);
            border-radius: 20px;
            overflow: hidden;
            border: 1px solid var(--border-color);
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .gallery-item:hover {
            transform: translateY(-8px);
            box-shadow: var(--shadow-xl);
            border-color: var(--primary);
        }
        
        .image-container {
            width: 100%;
            height: 280px;
            background: 
                linear-gradient(45deg, var(--checkerboard-light) 25%, var(--checkerboard-dark) 25%),
                linear-gradient(-45deg, var(--checkerboard-light) 25%, var(--checkerboard-dark) 25%),
                linear-gradient(45deg, var(--checkerboard-dark) 75%, var(--checkerboard-light) 75%),
                linear-gradient(-45deg, var(--checkerboard-dark) 75%, var(--checkerboard-light) 75%);
            background-size: 20px 20px;
            background-position: 0 0, 0 10px, 10px -10px, -10px 0px;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
        }
        
        .image-container img {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            transition: transform 0.3s ease;
        }
        
        .gallery-item:hover .image-container img {
            transform: scale(1.05);
        }
        
        .image-info {
            padding: 1.25rem;
        }
        
        .image-filename {
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-primary);
            margin-bottom: 0.75rem;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        .image-meta {
            display: flex;
            gap: 1rem;
            margin-bottom: 1rem;
            font-size: 0.75rem;
            color: var(--text-tertiary);
        }
        
        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .meta-item svg {
            width: 14px;
            height: 14px;
        }
        
        .image-actions {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 0.5rem;
        }
        
        .btn-small {
            padding: 0.5rem;
            font-size: 0.75rem;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.375rem;
            text-decoration: none;
        }
        
        .btn-small svg {
            width: 14px;
            height: 14px;
        }
        
        .btn-view {
            background: var(--gradient-primary);
            color: white;
        }
        
        .btn-download {
            background: linear-gradient(135deg, #10B981 0%, #059669 100%);
            color: white;
        }
        
        .btn-rename {
            background: linear-gradient(135deg, #F59E0B 0%, #D97706 100%);
            color: white;
        }
        
        .btn-delete {
            background: linear-gradient(135deg, #EF4444 0%, #DC2626 100%);
            color: white;
        }
        
        .btn-small:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            background: var(--bg-card);
            border-radius: 24px;
            border: 1px solid var(--border-color);
        }
        
        .empty-icon {
            width: 120px;
            height: 120px;
            margin: 0 auto 2rem;
            opacity: 0.3;
        }
        
        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 0.5rem;
            margin-top: 2rem;
            flex-wrap: wrap;
        }
        
        .pagination a,
        .pagination span {
            min-width: 44px;
            height: 44px;
            padding: 0 0.75rem;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            text-decoration: none;
            color: var(--text-primary);
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .pagination a:hover {
            background: var(--gradient-primary);
            color: white;
            border-color: transparent;
            transform: translateY(-2px);
        }
        
        .pagination .active {
            background: var(--gradient-primary);
            color: white;
            border-color: transparent;
        }
        
        .pagination .disabled {
            opacity: 0.4;
            cursor: not-allowed;
            pointer-events: none;
        }
        
        /* Lightbox */
        .lightbox {
            display: none;
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0.95);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }
        
        .lightbox.active {
            display: flex;
        }
        
        .lightbox-content {
            max-width: 90%;
            max-height: 90%;
            position: relative;
        }
        
        .lightbox-image {
            max-width: 100%;
            max-height: 90vh;
            border-radius: 12px;
        }
        
        .lightbox-close {
            position: absolute;
            top: -50px;
            right: 0;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.1);
            color: white;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
        }
        
        @media (max-width: 768px) {
            .container { padding: 1rem; }
            .header { padding: 1rem; }
            .header-content { gap: 0.5rem; }
            .logo { font-size: 1.25rem; }
            .stats { grid-template-columns: 1fr; gap: 1rem; }
            .gallery-grid { grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); }
            .image-container { height: 180px; }
            .header-actions { flex-wrap: wrap; width: 100%; }
            .user-dropdown { width: 100%; }
            .user-trigger { width: 100%; justify-content: space-between; }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M4 16L8.586 11.414C9.367 10.633 10.633 10.633 11.414 11.414L16 16M14 14L15.586 12.414C16.367 11.633 17.633 11.633 18.414 12.414L20 14M14 8H14.01M6 20H18C19.105 20 20 19.105 20 18V6C20 4.895 19.105 4 18 4H6C4.895 4 4 4.895 4 6V18C4 19.105 4.895 20 6 20Z" stroke="url(#gradient1)" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    <defs>
                        <linearGradient id="gradient1" x1="4" y1="4" x2="20" y2="20">
                            <stop offset="0%" stop-color="#8B5CF6"/>
                            <stop offset="100%" stop-color="#6D28D9"/>
                        </linearGradient>
                    </defs>
                </svg>
                <span>GALERI YESUS V3.1.0</span>
            </div>
            <div class="header-actions">
                <button class="theme-toggle" onclick="toggleDarkMode()">
                    <svg id="theme-icon-sun" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="5"/>
                        <line x1="12" y1="1" x2="12" y2="3"/>
                        <line x1="12" y1="21" x2="12" y2="23"/>
                        <line x1="4.22" y1="4.22" x2="5.64" y2="5.64"/>
                        <line x1="18.36" y1="18.36" x2="19.78" y2="19.78"/>
                        <line x1="1" y1="12" x2="3" y2="12"/>
                        <line x1="21" y1="12" x2="23" y2="12"/>
                        <line x1="4.22" y1="19.78" x2="5.64" y2="18.36"/>
                        <line x1="18.36" y1="5.64" x2="19.78" y2="4.22"/>
                    </svg>
                    <svg id="theme-icon-moon" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2" style="display: none;">
                        <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"/>
                    </svg>
                </button>
                
                <?php if ($isAdmin): ?>
                <a href="admin/users.php" class="btn btn-secondary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                        <circle cx="9" cy="7" r="4"/>
                        <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                        <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                    </svg>
                    Users
                </a>
                <?php endif; ?>
                
                <a href="admin/upload.php" class="btn btn-primary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="17 8 12 3 7 8"/>
                        <line x1="12" y1="3" x2="12" y2="15"/>
                    </svg>
                    Upload
                </a>
                
                <div class="user-dropdown">
                    <div class="user-trigger" onclick="toggleDropdown()">
                        <img src="profile-avatar.php?t=<?php echo time(); ?>" alt="Avatar" class="user-avatar">
                        <div class="user-info">
                            <div class="user-name"><?php echo htmlspecialchars($currentUser['display_name']); ?></div>
                            <div class="user-role"><?php echo ucfirst($currentUser['role']); ?></div>
                        </div>
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <polyline points="6 9 12 15 18 9"/>
                        </svg>
                    </div>
                    <div class="dropdown-menu" id="dropdown-menu">
                        <a href="settings_pro.php" class="dropdown-item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="3"/>
                                <path d="M12 1v6m0 6v6m5.2-13.2l-4.2 4.2m0 6l4.2 4.2M23 12h-6m-6 0H1m18.2 5.2l-4.2-4.2m-6 0l-4.2 4.2"/>
                            </svg>
                            Settings
                        </a>
                        <a href="logout.php" class="dropdown-item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"/>
                                <polyline points="16 17 21 12 16 7"/>
                                <line x1="21" y1="12" x2="9" y2="12"/>
                            </svg>
                            Logout
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <div class="container">
        <?php if (!empty($allImages)): ?>
            <div class="stats">
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="3" y="3" width="7" height="7"/>
                                <rect x="14" y="3" width="7" height="7"/>
                                <rect x="14" y="14" width="7" height="7"/>
                                <rect x="3" y="14" width="7" height="7"/>
                            </svg>
                        </div>
                        <div class="stat-label">Total Images</div>
                    </div>
                    <div class="stat-value"><?php echo count($allImages); ?></div>
                </div>

                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                            </svg>
                        </div>
                        <div class="stat-label">Storage Used</div>
                    </div>
                    <div class="stat-value">
                        <?php 
                        $totalSize = array_sum(array_column($allImages, 'size'));
                        echo number_format($totalSize / 1024 / 1024, 2);
                        ?> <span style="font-size: 1rem;  ">MB</span>


                    </div>
                </div>

                <?php if ($isAdmin && $userStats): ?>
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                                <circle cx="9" cy="7" r="4"/>
                                <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                                <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                            </svg>
                        </div>
                        <div class="stat-label">Total Users</div>
                    </div>
                    <div class="stat-value"><?php echo $userStats['total']; ?></div>
                </div>
                <?php else: ?>
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="3" y="3" width="18" height="18" rx="2"/>
                                <circle cx="8.5" cy="8.5" r="1.5"/>
                                <polyline points="21 15 16 10 5 21"/>
                            </svg>
                        </div>
                        <div class="stat-label">With Alpha</div>
                    </div>
                    <div class="stat-value">
                        <?php 
                       $withAlpha = count(array_filter($allImages, function($img) {
    return $img['has_alpha'];
}));

                        echo $withAlpha;
                        ?>
                    </div>
                </div>
                <?php endif; ?>

                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="3" y="4" width="18" height="18" rx="2"/>
                                <line x1="16" y1="2" x2="16" y2="6"/>
                                <line x1="8" y1="2" x2="8" y2="6"/>
                                <line x1="3" y1="10" x2="21" y2="10"/>
                            </svg>
                        </div>
                        <div class="stat-label">Page</div>
                    </div>
                    <div class="stat-value"><?php echo $pagination['current_page']; ?>/<span style="font-size: 1.5rem; "><?php echo $pagination['pages']; ?></span></div>
                </div>
            </div>

            <?php if (!empty($images)): ?>
                <div class="gallery-grid">
                    <?php foreach ($images as $img): ?>
                        <div class="gallery-item" onclick="openLightbox('<?php echo htmlspecialchars($img['filename']); ?>')">
                            <div class="image-container">
                                <img src="image.php?type=thumb&file=<?php echo urlencode($img['filename']); ?>" 
                                     alt="<?php echo htmlspecialchars($img['filename']); ?>"
                                     loading="lazy">
                            </div>
                            <div class="image-info">
                                <div class="image-filename" title="<?php echo htmlspecialchars($img['filename']); ?>">
                                    <?php echo htmlspecialchars(substr($img['filename'], 0, 30)); ?>
                                </div>
                                <div class="image-meta">
                                    <div class="meta-item">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"/>
                                            <polyline points="13 2 13 9 20 9"/>
                                        </svg>
                                        <?php echo number_format($img['size'] / 1024, 1); ?> KB
                                    </div>
                                    <div class="meta-item">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <circle cx="12" cy="12" r="10"/>
                                            <polyline points="12 6 12 12 16 14"/>
                                        </svg>
                                        <?php echo date('d M', $img['uploaded']); ?>
                                    </div>
                                </div>
                                <div class="image-actions" onclick="event.stopPropagation()">
                                    <a href="image.php?type=optimized&file=<?php echo urlencode($img['filename']); ?>" 
                                       target="_blank" class="btn-small btn-view">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                                            <circle cx="12" cy="12" r="3"/>
                                        </svg>
                                        View
                                    </a>
                                    <a href="image.php?type=original&file=<?php echo urlencode($img['filename']); ?>&download=1" 
                                       class="btn-small btn-download">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                            <polyline points="7 10 12 15 17 10"/>
                                            <line x1="12" y1="15" x2="12" y2="3"/>
                                        </svg>
                                        Download
                                    </a>
                                    <button onclick="renameImage('<?php echo htmlspecialchars($img['filename']); ?>')" 
                                            class="btn-small btn-rename">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
                                            <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
                                        </svg>
                                        Rename
                                    </button>
                                    <button onclick="deleteImage('<?php echo htmlspecialchars($img['filename']); ?>')" 
                                            class="btn-small btn-delete">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <polyline points="3 6 5 6 21 6"/>
                                            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
                                        </svg>
                                        Delete
                                    </button>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <?php if ($pagination['pages'] > 1): ?>
                    <div class="pagination">
                        <?php if ($pagination['has_prev']): ?>
                            <a href="?page=1">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="11 17 6 12 11 7"/>
                                    <polyline points="18 17 13 12 18 7"/>
                                </svg>
                            </a>
                            <a href="?page=<?php echo $pagination['current_page'] - 1; ?>">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="15 18 9 12 15 6"/>
                                </svg>
                            </a>
                        <?php else: ?>
                            <span class="disabled">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="11 17 6 12 11 7"/>
                                    <polyline points="18 17 13 12 18 7"/>
                                </svg>
                            </span>
                            <span class="disabled">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="15 18 9 12 15 6"/>
                                </svg>
                            </span>
                        <?php endif; ?>

                        <?php
                        $start = max(1, $pagination['current_page'] - 2);
                        $end = min($pagination['pages'], $pagination['current_page'] + 2);
                        
                        for ($i = $start; $i <= $end; $i++):
                            if ($i == $pagination['current_page']):
                        ?>
                            <span class="active"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                        <?php 
                            endif;
                        endfor; 
                        ?>

                        <?php if ($pagination['has_next']): ?>
                            <a href="?page=<?php echo $pagination['current_page'] + 1; ?>">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="9 18 15 12 9 6"/>
                                </svg>
                            </a>
                            <a href="?page=<?php echo $pagination['pages']; ?>">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="13 17 18 12 13 7"/>
                                    <polyline points="6 17 11 12 6 7"/>
                                </svg>
                            </a>
                        <?php else: ?>
                            <span class="disabled">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="9 18 15 12 9 6"/>
                                </svg>
                            </span>
                            <span class="disabled">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width: 16px; height: 16px;">
                                    <polyline points="13 17 18 12 13 7"/>
                                    <polyline points="6 17 11 12 6 7"/>
                                </svg>
                            </span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        <?php else: ?>
            <div class="empty-state">
                <svg class="empty-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="3" y="3" width="18" height="18" rx="2"/>
                    <circle cx="8.5" cy="8.5" r="1.5"/>
                    <polyline points="21 15 16 10 5 21"/>
                </svg>
                <h2>Gallery is Empty</h2>
                <p>Upload your first transparent PNG to get started</p>
                <a href="admin/upload.php" class="btn btn-primary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="17 8 12 3 7 8"/>
                        <line x1="12" y1="3" x2="12" y2="15"/>
                    </svg>
                    Upload Now
                </a>
            </div>
        <?php endif; ?>
    </div>

    <div class="lightbox" id="lightbox" onclick="closeLightbox()">
        <div class="lightbox-content" onclick="event.stopPropagation()">
            <button class="lightbox-close" onclick="closeLightbox()">×</button>
            <img class="lightbox-image" id="lightbox-image" src="" alt="">
        </div>
    </div>

    <script>
        function toggleDarkMode() {
            const html = document.documentElement;
            const currentTheme = html.getAttribute('data-theme');
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            
            html.setAttribute('data-theme', newTheme);
            localStorage.setItem('theme', newTheme);
            
            const sunIcon = document.getElementById('theme-icon-sun');
            const moonIcon = document.getElementById('theme-icon-moon');
            
            if (newTheme === 'dark') {
                sunIcon.style.display = 'none';
                moonIcon.style.display = 'block';
            } else {
                sunIcon.style.display = 'block';
                moonIcon.style.display = 'none';
            }
        }

        (function() {
            const savedTheme = localStorage.getItem('theme') || 'light';
            document.documentElement.setAttribute('data-theme', savedTheme);
            
            const sunIcon = document.getElementById('theme-icon-sun');
            const moonIcon = document.getElementById('theme-icon-moon');
            
            if (savedTheme === 'dark') {
                sunIcon.style.display = 'none';
                moonIcon.style.display = 'block';
            }
        })();

        function toggleDropdown() {
            document.getElementById('dropdown-menu').classList.toggle('active');
        }

        document.addEventListener('click', function(e) {
            const dropdown = document.querySelector('.user-dropdown');
            if (!dropdown.contains(e.target)) {
                document.getElementById('dropdown-menu').classList.remove('active');
            }
        });

        function openLightbox(filename) {
            const lightbox = document.getElementById('lightbox');
            const image = document.getElementById('lightbox-image');
            image.src = 'image.php?type=optimized&file=' + encodeURIComponent(filename);
            lightbox.classList.add('active');
        }

        function closeLightbox() {
            document.getElementById('lightbox').classList.remove('active');
        }

        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') closeLightbox();
        });

        function renameImage(filename) {
            const currentName = filename.replace('.png', '');
            
            Swal.fire({
                title: '<strong>Rename File</strong>',
                input: 'text',
                inputValue: currentName,
                showCancelButton: true,
                confirmButtonText: 'Rename',
                confirmButtonColor: '#8B5CF6',
                inputValidator: (value) => {
                    if (!value) return 'Gabisa Kosong Master J Minimal Isi!';
                    if (!/^[a-zA-Z0-9_-]+$/.test(value)) return 'Cuma Bisa Huruf, Angka, Garis Bawah & tanda Hubung Master J!';
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    Swal.fire({ title: 'Renaming...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
                    
                    const formData = new FormData();
                    formData.append('old_filename', filename);
                    formData.append('new_filename', result.value);
                    formData.append('<?php echo CSRF_TOKEN_NAME; ?>', '<?php echo CSRF::getToken(); ?>');
                    
                    fetch('admin/rename.php', { method: 'POST', body: formData })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({ icon: 'success', title: 'Success!', text: data.message, confirmButtonColor: '#8B5CF6' })
                            .then(() => location.reload());
                        } else {
                            Swal.fire({ icon: 'error', title: 'Error', text: data.message, confirmButtonColor: '#8B5CF6' });
                        }
                    });
                }
            });
        }

        function deleteImage(filename) {
            Swal.fire({
                title: '<strong>Yakin Hapus Ga Master J ?</strong>',
                text: 'Nanti Pusing Cari Lagi!',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#EF4444',
                confirmButtonText: 'Iya, Hapus Aja!'
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = 'admin/delete.php?file=' + encodeURIComponent(filename) + '&confirm=1';
                }
            });
        }
    </script>
</body>
</html>
