<?php
require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/csrf.php';
require_once __DIR__ . '/../../includes/UserManager.php';

// Require admin
Auth::requireAdmin();

$userManager = new UserManager();
$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    CSRF::verifyOrDie();
    
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create':
            $result = $userManager->createUser(
                $_POST['username'] ?? '',
                $_POST['password'] ?? '',
                $_POST['display_name'] ?? '',
                $_POST['email'] ?? '',
                $_POST['role'] ?? UserManager::ROLE_USER
            );
            
            if (isset($result['success'])) {
                $message = 'User created successfully!';
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'update':
            $result = $userManager->updateUser($_POST['username'] ?? '', [
                'display_name' => $_POST['display_name'] ?? '',
                'email' => $_POST['email'] ?? '',
                'role' => $_POST['role'] ?? '',
                'status' => $_POST['status'] ?? ''
            ]);
            
            if (isset($result['success'])) {
                $message = 'User updated successfully!';
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'change_password':
            $result = $userManager->changePassword(
                $_POST['username'] ?? '',
                $_POST['new_password'] ?? ''
            );
            
            if (isset($result['success'])) {
                $message = 'Password changed successfully!';
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'delete':
            $result = $userManager->deleteUser($_POST['username'] ?? '');
            
            if (isset($result['success'])) {
                $message = 'User deleted successfully!';
            } else {
                $error = $result['error'];
            }
            break;
    }
}

$users = $userManager->getAllUsers();
$stats = $userManager->getUserStats();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Galeri Yesus V1 Created By Immortaldev</title>
    <meta name="robots" content="noindex">  
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link href="https://fonts.googleapis.com/css2?family=Questrial&display=swap" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Questrial', sans-serif;
            background: linear-gradient(180deg, #FAF5FF 0%, #F5F3FF 100%);
            min-height: 100vh;
            padding: 2rem;
        }
        
        .container { max-width: 1200px; margin: 0 auto; }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        h1 {
            font-size: 2rem;
            background: linear-gradient(135deg, #8B5CF6 0%, #6D28D9 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 12px;
            font-weight: 600;
            cursor: pointer;
            border: none;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #8B5CF6 0%, #6D28D9 100%);
            color: white;
        }
        
        .btn-secondary {
            background: white;
            color: #374151;
            border: 1px solid #E5E7EB;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px rgba(0,0,0,0.2);
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 16px;
            border: 1px solid #E5E7EB;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            background: linear-gradient(135deg, #8B5CF6 0%, #6D28D9 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .stat-label {
            color: #6B7280;
            font-size: 0.875rem;
            margin-top: 0.5rem;
        }
        
        .users-table {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            border: 1px solid #E5E7EB;
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid #E5E7EB;
        }
        
        th {
            font-weight: 600;
            color: #374151;
        }
        
        .avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .badge {
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .badge-admin {
            background: #DBEAFE;
            color: #1E40AF;
        }
        
        .badge-user {
            background: #D1FAE5;
            color: #065F46;
        }
        
        .badge-active {
            background: #D1FAE5;
            color: #065F46;
        }
        
        .badge-inactive {
            background: #FEE2E2;
            color: #991B1B;
        }
        
        .actions {
            display: flex;
            gap: 0.5rem;
        }
        
        .btn-small {
            padding: 0.5rem 0.75rem;
            font-size: 0.75rem;
            border-radius: 8px;
        }
        
        .alert {
            padding: 1rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
        }
        
        .alert-success {
            background: #D1FAE5;
            color: #065F46;
        }
        
        .alert-error {
            background: #FEE2E2;
            color: #991B1B;
        }
        
        @media (max-width: 768px) {
            .users-table { padding: 1rem; }
            th, td { padding: 0.5rem; font-size: 0.875rem; }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>Manage User</h1>
                <p style="color: #6B7280;">kelola User Anda Disini Master J</p>
            </div>
            <div style="display: flex; gap: 1rem;">
                <button onclick="createUser()" class="btn btn-primary">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                        <circle cx="8.5" cy="7" r="4"/>
                        <line x1="20" y1="8" x2="20" y2="14"/>
                        <line x1="23" y1="11" x2="17" y2="11"/>
                    </svg>
                    Add User
                </button>
                <a href="../index.php" class="btn btn-secondary">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M19 12H5M12 19l-7-7 7-7"/>
                    </svg>
                    Back
                </a>
            </div>
        </div>
        
        <?php if ($message): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="stats">
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['total']; ?></div>
                <div class="stat-label">Total Users</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['admins']; ?></div>
                <div class="stat-label">Administrators</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['users']; ?></div>
                <div class="stat-label">Regular Users</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['active']; ?></div>
                <div class="stat-label">Active Users</div>
            </div>
        </div>
        
        <div class="users-table">
            <table>
                <thead>
                    <tr>
                        <th>User</th>
                        <th>Email</th>
                        <th>Role</th>
                        <th>Status</th>
                        <th>Uploads</th>
                        <th>Last Login</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                        <tr>
                            <td>
                                <div style="display: flex; align-items: center; gap: 0.75rem;">
                                    <img src="../profile-avatar.php?t=<?php echo time(); ?>" 
                                         alt="Avatar" class="avatar">
                                    <div>
                                        <div style="font-weight: 600;"><?php echo htmlspecialchars($user['display_name']); ?></div>
                                        <div style="font-size: 0.75rem; color: #6B7280;">@<?php echo htmlspecialchars($user['username']); ?></div>
                                    </div>
                                </div>
                            </td>
                            <td><?php echo htmlspecialchars($user['email'] ?: '-'); ?></td>
                            <td>
                                <span class="badge badge-<?php echo $user['role']; ?>">
                                    <?php echo ucfirst($user['role']); ?>
                                </span>
                            </td>
                            <td>
                                <span class="badge badge-<?php echo $user['status']; ?>">
                                    <?php echo ucfirst($user['status']); ?>
                                </span>
                            </td>
                            <td><?php echo $user['total_uploads']; ?></td>
                            <td>
                                <?php 
                                if ($user['last_login']) {
                                    echo date('Y-m-d H:i', $user['last_login']);
                                } else {
                                    echo 'Never';
                                }
                                ?>
                            </td>
                            <td>
                                <div class="actions">
                                    <button onclick="editUser('<?php echo htmlspecialchars($user['username']); ?>')" 
                                            class="btn btn-small btn-secondary">Edit</button>
                                    <?php if ($user['username'] !== 'admin' && $user['username'] !== Auth::getUsername()): ?>
                                        <button onclick="deleteUser('<?php echo htmlspecialchars($user['username']); ?>')" 
                                                class="btn btn-small" style="background: #EF4444; color: white;">Delete</button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <script>
        function createUser() {
            Swal.fire({
                title: 'Create New User',
                html: `
                    <div style="text-align: left;">
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Username</label>
                            <input id="new-username" class="swal2-input" placeholder="Username" style="width: 100%;">
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Display Name</label>
                            <input id="new-display-name" class="swal2-input" placeholder="Display Name" style="width: 100%;">
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Email</label>
                            <input id="new-email" type="email" class="swal2-input" placeholder="Email" style="width: 100%;">
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Password</label>
                            <input id="new-password" type="password" class="swal2-input" placeholder="Password" style="width: 100%;">
                        </div>
                        <div style="margin-bottom: 1rem;">
                            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Role</label>
                            <select id="new-role" class="swal2-input" style="width: 100%;">
                                <option value="user">User</option>
                                <option value="admin">Administrator</option>
                            </select>
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: 'Create User',
                confirmButtonColor: '#8B5CF6',
                preConfirm: () => {
                    return {
                        username: document.getElementById('new-username').value,
                        display_name: document.getElementById('new-display-name').value,
                        email: document.getElementById('new-email').value,
                        password: document.getElementById('new-password').value,
                        role: document.getElementById('new-role').value
                    };
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="create">
                        <input type="hidden" name="username" value="${result.value.username}">
                        <input type="hidden" name="display_name" value="${result.value.display_name}">
                        <input type="hidden" name="email" value="${result.value.email}">
                        <input type="hidden" name="password" value="${result.value.password}">
                        <input type="hidden" name="role" value="${result.value.role}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }
        
        function editUser(username) {
            // Get user data via AJAX or from hidden data
            Swal.fire({
                title: 'Edit User: ' + username,
                html: `<p>Fitur Ini Belom Anda Buat Master J</p>`,
                icon: 'info'
            });
        }
        
        function deleteUser(username) {
            Swal.fire({
                title: 'Yakin Mau Hapus Akun Ini Master J?',
                text: 'Saya Akan Hapus Akun ' + username + ' Dan Semua Data Dia Master J!',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#EF4444',
                confirmButtonText: 'Hapus Sekarang!'
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="delete">
                        <input type="hidden" name="username" value="${username}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }
        
        // Theme
        const savedTheme = localStorage.getItem('theme') || 'light';
        document.documentElement.setAttribute('data-theme', savedTheme);
    </script>
</body>
</html>
