<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/FolderManager.php';
require_once __DIR__ . '/../includes/ImageManager.php';
require_once __DIR__ . '/../includes/PermissionManager.php';

Auth::requireLogin();

$username = Auth::getUsername();
$isAdmin = Auth::isAdmin();
$folderManager = new FolderManager();
$imageManager = new ImageManager();
$permManager = new PermissionManager();

$folderId = $_GET['id'] ?? '';

if (empty($folderId)) {
    header('Location: folders.php?error=invalid_folder');
    exit;
}

$folder = $folderManager->getFolder($folderId);

if (!$folder || !$folderManager->canAccessFolder($folderId, $username, $isAdmin)) {
    header('Location: folders.php?error=access_denied');
    exit;
}

$images = $imageManager->getImagesByFolder($folderId);
$canUpload = $permManager->canUpload($username);
$canDelete = $permManager->canDelete($username);
$isOwner = $folderManager->ownsFolder($folderId, $username);

$pageTitle = htmlspecialchars($folder['name']) . ' - Galeriku ProMax';
$pageHeader = htmlspecialchars($folder['name']);
include __DIR__ . '/components/header.php';
?>

<!-- Folder Info Card -->
<div class="card" style="margin-bottom: var(--spacing-xl);">
    <div class="flex items-center justify-between" style="margin-bottom: var(--spacing-md);">
        <div class="flex items-center gap-md">
            <a href="folders.php" class="btn btn-secondary btn-sm">
                <?php echo UIIcons::chevronLeft('icon-sm'); ?>
                <span>Back</span>
            </a>
            <span class="badge badge-<?php echo $folder['visibility'] === 'private' ? 'danger' : 'success'; ?>">
                <?php echo $folder['visibility'] === 'private' ? UIIcons::lock('icon-sm') : UIIcons::globe('icon-sm'); ?>
                <?php echo ucfirst($folder['visibility']); ?>
            </span>
        </div>
        
        <div class="flex gap-sm">
            <?php if ($canUpload && $isOwner): ?>
                <a href="admin/upload.php?folder=<?php echo urlencode($folderId); ?>" class="btn btn-primary btn-sm">
                    <?php echo UIIcons::upload('icon-sm'); ?>
                    <span>Upload Here</span>
                </a>
            <?php endif; ?>
            
            <?php if ($isOwner || $isAdmin): ?>
                <a href="admin/folder-edit.php?id=<?php echo urlencode($folderId); ?>" class="btn btn-secondary btn-sm">
                    <?php echo UIIcons::edit('icon-sm'); ?>
                </a>
                <a href="admin/folder-delete.php?id=<?php echo urlencode($folderId); ?>" class="btn btn-secondary btn-sm">
                    <?php echo UIIcons::trash('icon-sm'); ?>
                </a>
            <?php endif; ?>
        </div>
    </div>
    
    <?php if ($folder['description']): ?>
        <p style="color: var(--text-secondary); margin-bottom: var(--spacing-md);">
            <?php echo htmlspecialchars($folder['description']); ?>
        </p>
    <?php endif; ?>
    
    <div class="flex items-center gap-lg" style="padding-top: var(--spacing-md); border-top: 1px solid var(--border);">
        <div class="flex items-center gap-sm">
            <?php echo UIIcons::image('icon-sm'); ?>
            <span style="font-size: 0.875rem; color: var(--text-secondary);"><?php echo count($images); ?> images</span>
        </div>
        <div class="flex items-center gap-sm">
            <?php echo UIIcons::users('icon-sm'); ?>
            <span style="font-size: 0.875rem; color: var(--text-secondary);"><?php echo htmlspecialchars($folder['owner']); ?></span>
        </div>
    </div>
</div>

<!-- Images Grid -->
<?php if (!empty($images)): ?>
    <div class="gallery-grid">
        <?php foreach ($images as $image): ?>
            <div class="gallery-item" onclick="openLightbox('<?php echo htmlspecialchars($image['filename']); ?>', '<?php echo htmlspecialchars($image['owner']); ?>')">
                <div class="gallery-image">
                    <img src="image.php?file=<?php echo urlencode($image['filename']); ?>&username=<?php echo urlencode($image['owner']); ?>" 
                         alt="<?php echo htmlspecialchars($image['filename']); ?>" 
                         loading="lazy">
                </div>
                <div class="gallery-info">
                    <div class="gallery-title"><?php echo htmlspecialchars($image['filename']); ?></div>
                    <div class="gallery-meta">
                        <span class="badge badge-<?php echo $image['visibility'] === 'private' ? 'danger' : 'success'; ?>">
                            <?php echo $image['visibility'] === 'private' ? UIIcons::lock('icon-sm') : UIIcons::globe('icon-sm'); ?>
                        </span>
                        <div class="flex gap-xs">
                            <button class="btn-icon-sm" onclick="event.stopPropagation(); downloadImage('<?php echo htmlspecialchars($image['filename']); ?>', '<?php echo htmlspecialchars($image['owner']); ?>');" title="Download">
                                <?php echo UIIcons::download('icon-sm'); ?>
                            </button>
                            <?php if ($canDelete && ($image['owner'] === $username || $isAdmin)): ?>
                                <button class="btn-icon-sm" onclick="event.stopPropagation(); deleteImage('<?php echo htmlspecialchars($image['filename']); ?>', '<?php echo urlencode($folderId); ?>');" title="Delete">
                                    <?php echo UIIcons::trash('icon-sm'); ?>
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
<?php else: ?>
    <div class="empty-state">
        <?php echo UIIcons::image('empty-icon'); ?>
        <h3 class="empty-title">No images in this folder</h3>
        <p class="empty-text">Upload images to this folder to see them here</p>
        <?php if ($canUpload && $isOwner): ?>
            <a href="admin/upload.php?folder=<?php echo urlencode($folderId); ?>" class="btn btn-primary">
                <?php echo UIIcons::upload(); ?>
                <span>Upload First Image</span>
            </a>
        <?php endif; ?>
    </div>
<?php endif; ?>

<!-- Lightbox -->
<div id="lightbox" class="lightbox" onclick="closeLightbox()">
    <button class="lightbox-close" onclick="closeLightbox()">
        <?php echo UIIcons::x(); ?>
    </button>
    <img id="lightbox-img" src="" alt="">
</div>

<style>
.gallery-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(260px, 1fr));
    gap: var(--spacing-lg);
}
.gallery-item {
    background: var(--bg-card);
    border: 1px solid var(--border);
    border-radius: var(--radius-xl);
    overflow: hidden;
    cursor: pointer;
    transition: all var(--transition-base);
}
.gallery-item:hover {
    border-color: var(--primary);
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}
.gallery-image {
    width: 100%;
    height: 220px;
    background: repeating-conic-gradient(var(--bg-secondary) 0% 25%, var(--bg-tertiary) 0% 50%) 50% / 20px 20px;
    display: flex;
    align-items: center;
    justify-content: center;
}
.gallery-image img {
    max-width: 100%;
    max-height: 100%;
    object-fit: contain;
}
.gallery-info {
    padding: var(--spacing-md);
}
.gallery-title {
    font-weight: 600;
    margin-bottom: var(--spacing-sm);
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}
.gallery-meta {
    display: flex;
    align-items: center;
    justify-content: space-between;
}
.btn-icon-sm {
    width: 32px;
    height: 32px;
    border: 1px solid var(--border);
    background: var(--bg-secondary);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all var(--transition-fast);
    color: var(--text-secondary);
}
.btn-icon-sm:hover {
    background: var(--primary);
    border-color: var(--primary);
    color: white;
}
.lightbox {
    position: fixed;
    inset: 0;
    background: rgba(0, 0, 0, 0.95);
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}
.lightbox.active {
    display: flex;
}
.lightbox img {
    max-width: 90%;
    max-height: 90%;
    object-fit: contain;
}
.lightbox-close {
    position: absolute;
    top: var(--spacing-xl);
    right: var(--spacing-xl);
    width: 48px;
    height: 48px;
    background: var(--primary);
    border: none;
    border-radius: 50%;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all var(--transition-base);
}
.lightbox-close:hover {
    background: var(--primary-dark);
    transform: scale(1.1);
}
</style>

<script>
function openLightbox(filename, owner) {
    document.getElementById('lightbox').classList.add('active');
    document.getElementById('lightbox-img').src = 'image.php?file=' + encodeURIComponent(filename) + '&username=' + encodeURIComponent(owner);
}
function closeLightbox() {
    document.getElementById('lightbox').classList.remove('active');
}
function downloadImage(filename, owner) {
    window.location.href = 'image.php?file=' + encodeURIComponent(filename) + '&username=' + encodeURIComponent(owner) + '&download=1';
}
function deleteImage(filename, folderId) {
    if (confirm('Are you sure you want to delete this image?')) {
        window.location.href = 'admin/delete.php?file=' + encodeURIComponent(filename) + '&folder=' + folderId;
    }
}
document.addEventListener('keydown', e => { if (e.key === 'Escape') closeLightbox(); });
</script>

<?php include __DIR__ . '/components/footer.php'; ?>
