<?php
require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/Auth.php';
require_once __DIR__ . '/../../includes/CSRF.php';
require_once __DIR__ . '/../../includes/PngHandler.php';
require_once __DIR__ . '/../../includes/FolderManager.php';
require_once __DIR__ . '/../../includes/ImageManager.php';
require_once __DIR__ . '/../../includes/PermissionManager.php';

Auth::requireLogin();

$username = Auth::getUsername();
$userId = Auth::getUserId();
$folderManager = new FolderManager();
$imageManager = new ImageManager();
$permManager = new PermissionManager();

// Get folder from URL
$preselectedFolder = $_GET['folder'] ?? '';

// Get user's folders
$userFolders = $folderManager->getUserFolders($username);
$canShare = $permManager->canShare($username);

$success = false;
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    CSRF::verifyOrDie();
    
    $handler = new PngHandler($userId, $username);
    $result = $handler->process($_FILES['file']);
    
    if ($result && isset($result['success'])) {
        $folderId = $_POST['folder_id'] ?? null;
        $visibility = ($canShare && isset($_POST['visibility'])) ? $_POST['visibility'] : 'private';
        $filename = $result['filename'] ?? '';
        $filesize = $result['size'] ?? 0;
        
        if ($folderId === 'root') $folderId = null;
        
        $imageManager->addImage($filename, $username, $folderId, $visibility, $filesize);
        $success = true;
    } else {
        $error = implode(', ', $handler->getErrors());
    }
}

$pageTitle = 'Upload - Galeriku ProMax';
$pageHeader = 'Upload Image';
include __DIR__ . '/../components/header.php';
?>

<?php if ($success): ?>
<div class="alert alert-success" style="margin-bottom: var(--spacing-xl);">
    <?php echo UIIcons::check('icon-sm'); ?>
    <span>Image uploaded successfully!</span>
</div>
<?php endif; ?>

<?php if ($error): ?>
<div class="alert alert-error" style="margin-bottom: var(--spacing-xl);">
    <?php echo UIIcons::alertCircle('icon-sm'); ?>
    <span><?php echo htmlspecialchars($error); ?></span>
</div>
<?php endif; ?>

<div class="grid grid-2">
    <!-- Upload Form -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Select Image</h2>
        </div>
        <div class="card-body">
            <form method="POST" enctype="multipart/form-data" id="uploadForm">
                <?php echo CSRF::field(); ?>
                
                <!-- Folder Selection -->
                <div class="form-group">
                    <label class="form-label" for="folder_id">
                        <?php echo UIIcons::folder('icon-sm'); ?>
                        Upload to Folder (Optional)
                    </label>
                    <select name="folder_id" id="folder_id" class="form-select">
                        <option value="root">No Folder (Gallery Root)</option>
                        <?php foreach ($userFolders as $folder): ?>
                            <option value="<?php echo htmlspecialchars($folder['id']); ?>" 
                                    <?php echo $preselectedFolder === $folder['id'] ? 'selected' : ''; ?>>
                                <?php echo $folder['visibility'] === 'private' ? '🔒' : '🌐'; ?> 
                                <?php echo htmlspecialchars($folder['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Visibility -->
                <?php if ($canShare): ?>
                <div class="form-group">
                    <label class="form-label">
                        <?php echo UIIcons::eye('icon-sm'); ?>
                        Visibility
                    </label>
                    <div class="radio-group">
                        <label class="radio-card">
                            <input type="radio" name="visibility" value="private" checked>
                            <div class="radio-content">
                                <?php echo UIIcons::lock(); ?>
                                <strong>Private</strong>
                                <span>Only you can see</span>
                            </div>
                        </label>
                        <label class="radio-card">
                            <input type="radio" name="visibility" value="shared">
                            <div class="radio-content">
                                <?php echo UIIcons::globe(); ?>
                                <strong>Shared</strong>
                                <span>Everyone can see</span>
                            </div>
                        </label>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- File Input -->
                <div class="upload-zone" id="dropZone">
                    <input type="file" name="file" id="fileInput" accept="image/png" hidden>
                    <?php echo UIIcons::upload('icon-xl'); ?>
                    <h3 style="margin: var(--spacing-md) 0 var(--spacing-sm);">Drop PNG here</h3>
                    <p style="color: var(--text-secondary); margin-bottom: var(--spacing-md);">or click to browse</p>
                    <button type="button" class="btn btn-primary" onclick="document.getElementById('fileInput').click()">
                        <?php echo UIIcons::folder('icon-sm'); ?>
                        <span>Choose File</span>
                    </button>
                    <div style="margin-top: var(--spacing-md); font-size: 0.8125rem; color: var(--text-tertiary);">
                        Max size: <?php echo formatBytes(MAX_FILE_SIZE); ?> • PNG only
                    </div>
                </div>
                
                <div id="preview" style="display: none; margin-top: var(--spacing-lg);">
                    <div class="flex items-center justify-between" style="margin-bottom: var(--spacing-md);">
                        <strong>Selected:</strong>
                        <button type="button" class="btn btn-secondary btn-sm" onclick="clearFile()">
                            <?php echo UIIcons::x('icon-sm'); ?>
                        </button>
                    </div>
                    <div id="previewImage" style="max-height: 200px; overflow: hidden; border-radius: var(--radius-lg); background: var(--bg-secondary);"></div>
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: var(--spacing-md);">
                        <?php echo UIIcons::upload(); ?>
                        <span>Upload Image</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Info Card -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Upload Guidelines</h2>
        </div>
        <div class="card-body">
            <div class="info-list">
                <div class="info-item">
                    <?php echo UIIcons::check('icon-sm'); ?>
                    <div>
                        <strong>Format</strong>
                        <p>Only PNG images are supported</p>
                    </div>
                </div>
                <div class="info-item">
                    <?php echo UIIcons::check('icon-sm'); ?>
                    <div>
                        <strong>File Size</strong>
                        <p>Maximum <?php echo formatBytes(MAX_FILE_SIZE); ?> per image</p>
                    </div>
                </div>
                <div class="info-item">
                    <?php echo UIIcons::check('icon-sm'); ?>
                    <div>
                        <strong>Transparency</strong>
                        <p>PNG transparency is preserved</p>
                    </div>
                </div>
                <div class="info-item">
                    <?php echo UIIcons::check('icon-sm'); ?>
                    <div>
                        <strong>Optimization</strong>
                        <p>Images are automatically optimized</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.alert {
    display: flex;
    align-items: center;
    gap: var(--spacing-sm);
    padding: var(--spacing-md);
    border-radius: var(--radius-md);
    border: 1px solid;
}
.alert-success {
    background: rgba(16, 185, 129, 0.1);
    border-color: var(--success);
    color: var(--success);
}
.alert-error {
    background: rgba(239, 68, 68, 0.1);
    border-color: var(--danger);
    color: var(--danger);
}
.radio-group {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: var(--spacing-md);
}
.radio-card {
    position: relative;
    cursor: pointer;
}
.radio-card input {
    position: absolute;
    opacity: 0;
}
.radio-content {
    padding: var(--spacing-lg);
    border: 2px solid var(--border);
    border-radius: var(--radius-lg);
    text-align: center;
    transition: all var(--transition-base);
}
.radio-content svg {
    width: 32px;
    height: 32px;
    margin-bottom: var(--spacing-sm);
    color: var(--text-tertiary);
}
.radio-content strong {
    display: block;
    margin-bottom: var(--spacing-xs);
}
.radio-content span {
    font-size: 0.8125rem;
    color: var(--text-secondary);
}
.radio-card input:checked + .radio-content {
    border-color: var(--primary);
    background: rgba(139, 92, 246, 0.05);
}
.radio-card input:checked + .radio-content svg {
    color: var(--primary);
}
.upload-zone {
    border: 2px dashed var(--border);
    border-radius: var(--radius-xl);
    padding: var(--spacing-2xl);
    text-align: center;
    transition: all var(--transition-base);
    cursor: pointer;
}
.upload-zone:hover {
    border-color: var(--primary);
    background: rgba(139, 92, 246, 0.02);
}
.upload-zone svg {
    color: var(--primary);
    margin: 0 auto;
}
.info-list {
    display: flex;
    flex-direction: column;
    gap: var(--spacing-lg);
}
.info-item {
    display: flex;
    gap: var(--spacing-md);
}
.info-item svg {
    color: var(--success);
    flex-shrink: 0;
}
.info-item strong {
    display: block;
    margin-bottom: var(--spacing-xs);
}
.info-item p {
    font-size: 0.875rem;
    color: var(--text-secondary);
    margin: 0;
}
@media (max-width: 768px) {
    .radio-group {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
const dropZone = document.getElementById('dropZone');
const fileInput = document.getElementById('fileInput');
const preview = document.getElementById('preview');
const previewImage = document.getElementById('previewImage');

dropZone.onclick = () => fileInput.click();

dropZone.ondragover = (e) => {
    e.preventDefault();
    dropZone.style.borderColor = 'var(--primary)';
    dropZone.style.background = 'rgba(139, 92, 246, 0.05)';
};

dropZone.ondragleave = () => {
    dropZone.style.borderColor = 'var(--border)';
    dropZone.style.background = 'transparent';
};

dropZone.ondrop = (e) => {
    e.preventDefault();
    dropZone.style.borderColor = 'var(--border)';
    dropZone.style.background = 'transparent';
    
    if (e.dataTransfer.files.length > 0) {
        fileInput.files = e.dataTransfer.files;
        showPreview(e.dataTransfer.files[0]);
    }
};

fileInput.onchange = (e) => {
    if (e.target.files.length > 0) {
        showPreview(e.target.files[0]);
    }
};

function showPreview(file) {
    if (!file.type.match('image/png')) {
        alert('Please select a PNG image');
        return;
    }
    
    const reader = new FileReader();
    reader.onload = (e) => {
        previewImage.innerHTML = '<img src="' + e.target.result + '" style="max-width: 100%; height: auto; display: block;">';
        preview.style.display = 'block';
        dropZone.style.display = 'none';
    };
    reader.readAsDataURL(file);
}

function clearFile() {
    fileInput.value = '';
    preview.style.display = 'none';
    dropZone.style.display = 'block';
    previewImage.innerHTML = '';
}
</script>

<?php include __DIR__ . '/../components/footer.php'; ?>
