<?php
/**
 * Galeriku ProMax Configuration
 * Version: 2.0
 * Enhanced with better structure and security
 */

// Error reporting (disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../storage/error.log');

// Session configuration
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', 0); // Set 1 if using HTTPS
ini_set('session.cookie_samesite', 'Strict');

// Timezone
date_default_timezone_set('Asia/Jakarta');

// Path configuration
define('BASE_PATH', dirname(__DIR__));
define('PUBLIC_PATH', BASE_PATH . '/public');
define('STORAGE_PATH', BASE_PATH . '/storage');
define('UPLOADS_PATH', STORAGE_PATH . '/uploads');
define('ORIGINALS_PATH', UPLOADS_PATH . '/originals');
define('OPTIMIZED_PATH', UPLOADS_PATH . '/optimized');
define('THUMBS_PATH', STORAGE_PATH . '/thumbs');
define('USERS_PATH', STORAGE_PATH . '/users');

// Create directories if not exist
$directories = [
    STORAGE_PATH,
    UPLOADS_PATH,
    ORIGINALS_PATH,
    OPTIMIZED_PATH,
    THUMBS_PATH,
    USERS_PATH
];

foreach ($directories as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// URL configuration
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';
$scriptDir = dirname($_SERVER['SCRIPT_NAME']);
define('BASE_URL', $protocol . '://' . $host . $scriptDir);
define('PUBLIC_URL', BASE_URL . '/public');

// Admin credentials
define('ADMIN_USERNAME', 'admin');
define('ADMIN_PASSWORD_HASH', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi'); // password

// PNG Settings
define('PNG_COMPRESSION', 6); // 0-9, balanced between size and speed
define('THUMBNAIL_SIZE', 300); // pixels
define('MAX_FILE_SIZE', 10 * 1024 * 1024); // 10MB
define('ALLOWED_MIME', 'image/png');

// User Settings
define('MAX_USERNAME_LENGTH', 20);
define('MIN_USERNAME_LENGTH', 3);
define('MIN_PASSWORD_LENGTH', 6);
define('MAX_AVATAR_SIZE', 2 * 1024 * 1024); // 2MB

// Session name
define('SESSION_NAME', 'GALERIKU_PROMAX');

// CSRF token name
define('CSRF_TOKEN_NAME', 'csrf_token');

// Pagination
define('ITEMS_PER_PAGE', 12);

// Activity log limits
define('MAX_LOGIN_HISTORY', 50);
define('MAX_ACTIVITY_LOG', 100);

/**
 * Autoload classes
 */
spl_autoload_register(function ($class) {
    $file = BASE_PATH . '/includes/' . $class . '.php';
    if (file_exists($file)) {
        require_once $file;
    }
});

/**
 * Start session with security
 */
if (session_status() === PHP_SESSION_NONE) {
    session_name(SESSION_NAME);
    session_start();
    
    // Regenerate session ID on first visit
    if (!isset($_SESSION['initiated'])) {
        session_regenerate_id(true);
        $_SESSION['initiated'] = true;
        $_SESSION['created_at'] = time();
    }
    
    // Session timeout (30 minutes)
    if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > 1800)) {
        session_unset();
        session_destroy();
        session_start();
    }
    $_SESSION['last_activity'] = time();
}

/**
 * Helper functions
 */

/**
 * Sanitize filename
 */
function sanitizeFilename($filename) {
    $filename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $filename);
    $filename = preg_replace('/_+/', '_', $filename);
    $filename = trim($filename, '_-');
    return substr($filename, 0, 50);
}

/**
 * Format bytes to human readable
 */
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= (1 << (10 * $pow));
    return round($bytes, $precision) . ' ' . $units[$pow];
}

/**
 * Get time ago string
 */
function timeAgo($timestamp) {
    $diff = time() - $timestamp;
    
    if ($diff < 60) return 'just now';
    if ($diff < 3600) return floor($diff / 60) . ' minutes ago';
    if ($diff < 86400) return floor($diff / 3600) . ' hours ago';
    if ($diff < 604800) return floor($diff / 86400) . ' days ago';
    if ($diff < 2592000) return floor($diff / 604800) . ' weeks ago';
    if ($diff < 31536000) return floor($diff / 2592000) . ' months ago';
    return floor($diff / 31536000) . ' years ago';
}

/**
 * Generate unique ID
 */
function generateUniqueId($prefix = '') {
    return $prefix . bin2hex(random_bytes(8)) . '_' . time();
}

/**
 * JSON response helper
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Get client IP
 */
function getClientIP() {
    $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($headers as $header) {
        if (isset($_SERVER[$header]) && filter_var($_SERVER[$header], FILTER_VALIDATE_IP)) {
            return $_SERVER[$header];
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
}

/**
 * Validate email
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Create .htaccess for storage protection
 */
$htaccessContent = "# Deny all direct access\nOrder Deny,Allow\nDeny from all";
$storageHtaccess = STORAGE_PATH . '/.htaccess';
if (!file_exists($storageHtaccess)) {
    file_put_contents($storageHtaccess, $htaccessContent);
}
