<?php
/**
 * User Manager Class - ProMax Version 2.0
 * Fixed all bugs, enhanced features, better structure
 */

class UserManager
{
    private $usersFile;
    private $storageDir;
    private $loginHistoryFile;
    private $activityLogFile;
    
    const ROLE_ADMIN = 'admin';
    const ROLE_USER = 'user';
    
    public function __construct()
    {
        $this->usersFile = STORAGE_PATH . '/users.json';
        $this->storageDir = USERS_PATH;
        $this->loginHistoryFile = STORAGE_PATH . '/login_history.json';
        $this->activityLogFile = STORAGE_PATH . '/activity_log.json';
        
        // Create storage directory
        if (!is_dir($this->storageDir)) {
            mkdir($this->storageDir, 0755, true);
        }
        
        // Initialize files
        $this->initializeFiles();
        
        // Initialize default admin if no users exist
        if (!file_exists($this->usersFile) || empty($this->getAllUsers())) {
            $this->initializeDefaultAdmin();
        }
    }
    
    /**
     * Initialize data files
     */
    private function initializeFiles()
    {
        $files = [
            $this->usersFile => [],
            $this->loginHistoryFile => [],
            $this->activityLogFile => []
        ];
        
        foreach ($files as $file => $defaultData) {
            if (!file_exists($file)) {
                file_put_contents($file, json_encode($defaultData, JSON_PRETTY_PRINT));
                chmod($file, 0644);
            }
        }
    }
    
    /**
     * Initialize default admin user
     */
    private function initializeDefaultAdmin()
    {
        $adminId = generateUniqueId('user_');
        
        $users = [
            'admin' => [
                'id' => $adminId,
                'username' => 'admin',
                'password_hash' => ADMIN_PASSWORD_HASH,
                'display_name' => 'Administrator',
                'role' => self::ROLE_ADMIN,
                'email' => 'admin@galeriku.local',
                'bio' => 'System Administrator',
                'has_avatar' => false,
                'created_at' => time(),
                'last_login' => null,
                'last_login_ip' => null,
                'last_login_user_agent' => null,
                'total_uploads' => 0,
                'total_logins' => 0,
                'storage_used' => 0,
                'status' => 'active'
            ]
        ];
        
        $this->saveUsers($users);
        
        // Create admin directory
        $adminDir = $this->storageDir . '/admin';
        if (!is_dir($adminDir)) {
            mkdir($adminDir, 0755, true);
        }
    }
    
    /**
     * Get all users
     */
    public function getAllUsers()
    {
        if (!file_exists($this->usersFile)) {
            return [];
        }
        
        $data = file_get_contents($this->usersFile);
        return json_decode($data, true) ?: [];
    }
    
    /**
     * Get user by username
     */
    public function getUser($username)
    {
        $users = $this->getAllUsers();
        return $users[$username] ?? null;
    }
    
    /**
     * Get user by ID
     */
    public function getUserById($userId)
    {
        $users = $this->getAllUsers();
        foreach ($users as $user) {
            if ($user['id'] === $userId) {
                return $user;
            }
        }
        return null;
    }
    
    /**
     * Save users to file with backup
     */
    private function saveUsers($users)
    {
        // Create backup before saving
        if (file_exists($this->usersFile)) {
            $backupFile = $this->usersFile . '.backup_' . date('YmdHis');
            copy($this->usersFile, $backupFile);
            
            // Keep only last 5 backups
            $backups = glob($this->usersFile . '.backup_*');
            if (count($backups) > 5) {
                usort($backups, function($a, $b) {
                    return filemtime($a) - filemtime($b);
                });
                unlink($backups[0]);
            }
        }
        
        return file_put_contents($this->usersFile, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    /**
     * Create new user
     */
    public function createUser($username, $password, $displayName = '', $email = '', $role = self::ROLE_USER)
    {
        // Validate username
        if (empty($username)) {
            return ['error' => 'Username is required'];
        }
        
        if (!preg_match('/^[a-zA-Z0-9_-]{' . MIN_USERNAME_LENGTH . ',' . MAX_USERNAME_LENGTH . '}$/', $username)) {
            return ['error' => 'Username must be ' . MIN_USERNAME_LENGTH . '-' . MAX_USERNAME_LENGTH . ' characters (letters, numbers, underscore, dash only)'];
        }
        
        // Check if username exists
        if ($this->getUser($username)) {
            return ['error' => 'Username already exists'];
        }
        
        // Validate password
        if (strlen($password) < MIN_PASSWORD_LENGTH) {
            return ['error' => 'Password must be at least ' . MIN_PASSWORD_LENGTH . ' characters'];
        }
        
        // Validate email
        if (!empty($email) && !isValidEmail($email)) {
            return ['error' => 'Invalid email address'];
        }
        
        // Generate unique user ID
        $userId = generateUniqueId('user_');
        
        // Create user data
        $users = $this->getAllUsers();
        $users[$username] = [
            'id' => $userId,
            'username' => $username,
            'password_hash' => password_hash($password, PASSWORD_DEFAULT),
            'display_name' => $displayName ?: $username,
            'role' => $role,
            'email' => $email,
            'bio' => '',
            'has_avatar' => false,
            'created_at' => time(),
            'last_login' => null,
            'last_login_ip' => null,
            'last_login_user_agent' => null,
            'total_uploads' => 0,
            'total_logins' => 0,
            'storage_used' => 0,
            'status' => 'active'
        ];
        
        // Save users
        if ($this->saveUsers($users)) {
            // Create user directory
            $userDir = $this->storageDir . '/' . $username;
            if (!is_dir($userDir)) {
                mkdir($userDir, 0755, true);
            }
            
            // Log activity
            $this->logActivity($username, 'user_created', 'User account created');
            
            return ['success' => true, 'user_id' => $userId, 'username' => $username];
        }
        
        return ['error' => 'Failed to create user'];
    }
    
    /**
     * Update user profile
     */
    public function updateUser($username, $data)
    {
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        // Allowed fields to update
        $allowedFields = ['display_name', 'email', 'bio', 'role', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                // Validate email if being updated
                if ($field === 'email' && !empty($data[$field])) {
                    if (!isValidEmail($data[$field])) {
                        return ['error' => 'Invalid email address'];
                    }
                }
                
                // Validate role
                if ($field === 'role' && !in_array($data[$field], [self::ROLE_ADMIN, self::ROLE_USER])) {
                    return ['error' => 'Invalid role'];
                }
                
                // Validate status
                if ($field === 'status' && !in_array($data[$field], ['active', 'inactive', 'suspended'])) {
                    return ['error' => 'Invalid status'];
                }
                
                $users[$username][$field] = $data[$field];
            }
        }
        
        if ($this->saveUsers($users)) {
            $this->logActivity($username, 'profile_updated', 'User profile updated');
            return ['success' => true];
        }
        
        return ['error' => 'Failed to update user'];
    }
    
    /**
     * Change password
     */
    public function changePassword($username, $currentPassword, $newPassword)
    {
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        // Verify current password (skip for admin changing other users)
        if (isset($currentPassword) && !empty($currentPassword)) {
            if (!password_verify($currentPassword, $users[$username]['password_hash'])) {
                return ['error' => 'Current password is incorrect'];
            }
        }
        
        // Validate new password
        if (strlen($newPassword) < MIN_PASSWORD_LENGTH) {
            return ['error' => 'New password must be at least ' . MIN_PASSWORD_LENGTH . ' characters'];
        }
        
        // Update password
        $users[$username]['password_hash'] = password_hash($newPassword, PASSWORD_DEFAULT);
        
        if ($this->saveUsers($users)) {
            $this->logActivity($username, 'password_changed', 'Password changed successfully');
            return ['success' => true];
        }
        
        return ['error' => 'Failed to change password'];
    }
    
    /**
     * Change username
     */
    public function changeUsername($oldUsername, $newUsername)
    {
        // Validate new username
        if (empty($newUsername)) {
            return ['error' => 'Username cannot be empty'];
        }
        
        if (!preg_match('/^[a-zA-Z0-9_-]{' . MIN_USERNAME_LENGTH . ',' . MAX_USERNAME_LENGTH . '}$/', $newUsername)) {
            return ['error' => 'Username must be ' . MIN_USERNAME_LENGTH . '-' . MAX_USERNAME_LENGTH . ' characters'];
        }
        
        // Cannot change admin username
        if ($oldUsername === 'admin') {
            return ['error' => 'Cannot change admin username'];
        }
        
        $users = $this->getAllUsers();
        
        // Check if old user exists
        if (!isset($users[$oldUsername])) {
            return ['error' => 'User not found'];
        }
        
        // Check if new username exists
        if (isset($users[$newUsername])) {
            return ['error' => 'Username already taken'];
        }
        
        // Get user data and update username
        $userData = $users[$oldUsername];
        $userData['username'] = $newUsername;
        
        // Remove old, add new
        unset($users[$oldUsername]);
        $users[$newUsername] = $userData;
        
        // Save users
        if (!$this->saveUsers($users)) {
            return ['error' => 'Failed to update username'];
        }
        
        // Rename user directory
        $oldDir = $this->storageDir . '/' . $oldUsername;
        $newDir = $this->storageDir . '/' . $newUsername;
        
        if (is_dir($oldDir)) {
            if (!rename($oldDir, $newDir)) {
                // Rollback
                unset($users[$newUsername]);
                $users[$oldUsername] = $userData;
                $this->saveUsers($users);
                return ['error' => 'Failed to rename user directory'];
            }
        }
        
        // Migrate login history
        $this->migrateLoginHistory($oldUsername, $newUsername);
        
        // Migrate activity log
        $this->migrateActivityLog($oldUsername, $newUsername);
        
        // Log activity
        $this->logActivity($newUsername, 'username_changed', "Username changed from $oldUsername to $newUsername");
        
        return ['success' => true, 'new_username' => $newUsername];
    }
    
    /**
     * Delete user
     */
    public function deleteUser($username)
    {
        // Cannot delete admin or self
        if ($username === 'admin') {
            return ['error' => 'Cannot delete admin user'];
        }
        
        if (isset($_SESSION['username']) && $_SESSION['username'] === $username) {
            return ['error' => 'Cannot delete yourself'];
        }
        
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        // Remove user
        unset($users[$username]);
        
        // Delete user directory
        $userDir = $this->storageDir . '/' . $username;
        if (is_dir($userDir)) {
            $this->deleteDirectory($userDir);
        }
        
        // Save
        if ($this->saveUsers($users)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to delete user'];
    }
    
    /**
     * Upload user avatar
     */
    public function uploadAvatar($username, $file)
    {
        // Validate file
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            return ['error' => 'Invalid file'];
        }
        
        // Check size
        if ($file['size'] > MAX_AVATAR_SIZE) {
            return ['error' => 'File too large (max ' . formatBytes(MAX_AVATAR_SIZE) . ')'];
        }
        
        // Check MIME
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, ['image/png', 'image/jpeg', 'image/jpg'])) {
            return ['error' => 'Only PNG/JPG allowed'];
        }
        
        // Load image
        if ($mimeType === 'image/png') {
            $img = @imagecreatefrompng($file['tmp_name']);
        } else {
            $img = @imagecreatefromjpeg($file['tmp_name']);
        }
        
        if (!$img) {
            return ['error' => 'Failed to load image'];
        }
        
        // Create square thumbnail (200x200)
        $size = 200;
        $width = imagesx($img);
        $height = imagesy($img);
        
        // Calculate crop dimensions
        $cropSize = min($width, $height);
        $cropX = ($width - $cropSize) / 2;
        $cropY = ($height - $cropSize) / 2;
        
        // Create avatar
        $avatar = imagecreatetruecolor($size, $size);
        imagealphablending($avatar, false);
        imagesavealpha($avatar, true);
        
        // Transparent background for PNG
        $transparent = imagecolorallocatealpha($avatar, 0, 0, 0, 127);
        imagefill($avatar, 0, 0, $transparent);
        
        // Resize and crop (cast to int to avoid deprecated warning)
        imagecopyresampled($avatar, $img, 0, 0, (int)$cropX, (int)$cropY, $size, $size, (int)$cropSize, (int)$cropSize);
        
        // Save avatar
        $userDir = $this->storageDir . '/' . $username;
        if (!is_dir($userDir)) {
            mkdir($userDir, 0755, true);
        }
        
        $avatarPath = $userDir . '/avatar.png';
        $success = imagepng($avatar, $avatarPath, 6);
        
        imagedestroy($img);
        imagedestroy($avatar);
        
        if ($success) {
            // Update user
            $users = $this->getAllUsers();
            if (isset($users[$username])) {
                $users[$username]['has_avatar'] = true;
                $this->saveUsers($users);
            }
            
            $this->logActivity($username, 'avatar_updated', 'Profile avatar updated');
            
            return ['success' => true, 'avatar_url' => PUBLIC_URL . '/user-avatar.php?username=' . urlencode($username) . '&t=' . time()];
        }
        
        return ['error' => 'Failed to save avatar'];
    }
    
    /**
     * Delete user avatar
     */
    public function deleteAvatar($username)
    {
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        
        if (file_exists($avatarPath)) {
            if (unlink($avatarPath)) {
                // Update user
                $users = $this->getAllUsers();
                if (isset($users[$username])) {
                    $users[$username]['has_avatar'] = false;
                    $this->saveUsers($users);
                }
                
                $this->logActivity($username, 'avatar_deleted', 'Profile avatar deleted');
                
                return ['success' => true];
            }
        }
        
        return ['error' => 'Failed to delete avatar'];
    }
    
    /**
     * Get avatar path
     */
    public function getAvatarPath($username)
    {
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        return file_exists($avatarPath) ? $avatarPath : null;
    }
    
    /**
     * Verify credentials
     */
    public function verifyCredentials($username, $password)
    {
        $user = $this->getUser($username);
        
        if (!$user) {
            return false;
        }
        
        if ($user['status'] !== 'active') {
            return false;
        }
        
        return password_verify($password, $user['password_hash']);
    }
    
    /**
     * Update last login
     */
    public function updateLastLogin($username)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            $users[$username]['last_login'] = time();
            $users[$username]['last_login_ip'] = getClientIP();
            $users[$username]['last_login_user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
            $users[$username]['total_logins'] = ($users[$username]['total_logins'] ?? 0) + 1;
            
            $this->saveUsers($users);
        }
    }
    
    /**
     * Increment upload count
     */
    public function incrementUploads($username)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            $users[$username]['total_uploads'] = ($users[$username]['total_uploads'] ?? 0) + 1;
            $this->saveUsers($users);
        }
    }
    
    /**
     * Update storage used
     */
    public function updateStorageUsed($username)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            // Calculate total storage
            $userDir = $this->storageDir . '/' . $username;
            $totalSize = 0;
            
            if (is_dir($userDir)) {
                $totalSize = $this->getDirectorySize($userDir);
            }
            
            $users[$username]['storage_used'] = $totalSize;
            $this->saveUsers($users);
        }
    }
    
    /**
     * Get directory size
     */
    private function getDirectorySize($dir)
    {
        $size = 0;
        
        if (is_dir($dir)) {
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
            );
            
            foreach ($files as $file) {
                $size += $file->getSize();
            }
        }
        
        return $size;
    }
    
    /**
     * Log login attempt
     */
    public function logLoginAttempt($username, $success)
    {
        if (!file_exists($this->loginHistoryFile)) {
            $this->initializeFiles();
        }
        
        $history = json_decode(file_get_contents($this->loginHistoryFile), true) ?: [];
        
        if (!isset($history[$username])) {
            $history[$username] = [];
        }
        
        array_unshift($history[$username], [
            'timestamp' => time(),
            'success' => $success,
            'ip' => getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
        ]);
        
        // Keep only last N attempts
        $history[$username] = array_slice($history[$username], 0, MAX_LOGIN_HISTORY);
        
        file_put_contents($this->loginHistoryFile, json_encode($history, JSON_PRETTY_PRINT));
    }
    
    /**
     * Get login history
     */
    public function getLoginHistory($username, $limit = 20)
    {
        if (!file_exists($this->loginHistoryFile)) {
            return [];
        }
        
        $history = json_decode(file_get_contents($this->loginHistoryFile), true) ?: [];
        $userHistory = $history[$username] ?? [];
        
        return array_slice($userHistory, 0, $limit);
    }
    
    /**
     * Log activity
     */
    public function logActivity($username, $action, $description = '', $metadata = [])
    {
        if (!file_exists($this->activityLogFile)) {
            $this->initializeFiles();
        }
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        
        if (!isset($log[$username])) {
            $log[$username] = [];
        }
        
        array_unshift($log[$username], [
            'timestamp' => time(),
            'action' => $action,
            'description' => $description,
            'ip' => getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'metadata' => $metadata
        ]);
        
        // Keep only last N activities
        $log[$username] = array_slice($log[$username], 0, MAX_ACTIVITY_LOG);
        
        file_put_contents($this->activityLogFile, json_encode($log, JSON_PRETTY_PRINT));
    }
    
    /**
     * Get activity log
     */
    public function getActivityLog($username, $limit = 20)
    {
        if (!file_exists($this->activityLogFile)) {
            return [];
        }
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        $userLog = $log[$username] ?? [];
        
        return array_slice($userLog, 0, $limit);
    }
    
    /**
     * Get all activity logs (admin only)
     */
    public function getAllActivityLogs($limit = 50)
    {
        if (!file_exists($this->activityLogFile)) {
            return [];
        }
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        
        $allLogs = [];
        foreach ($log as $username => $userLogs) {
            foreach ($userLogs as $activity) {
                $activity['username'] = $username;
                $allLogs[] = $activity;
            }
        }
        
        // Sort by timestamp desc
        usort($allLogs, function($a, $b) {
            return $b['timestamp'] - $a['timestamp'];
        });
        
        return array_slice($allLogs, 0, $limit);
    }
    
    /**
     * Get user stats
     */
    public function getUserStats()
    {
        $users = $this->getAllUsers();
        
        $stats = [
            'total_users' => count($users),
            'active_users' => 0,
            'inactive_users' => 0,
            'admin_users' => 0,
            'regular_users' => 0,
            'total_uploads' => 0,
            'total_logins' => 0,
            'total_storage' => 0
        ];
        
        foreach ($users as $user) {
            if ($user['status'] === 'active') {
                $stats['active_users']++;
            } else {
                $stats['inactive_users']++;
            }
            
            if ($user['role'] === self::ROLE_ADMIN) {
                $stats['admin_users']++;
            } else {
                $stats['regular_users']++;
            }
            
            $stats['total_uploads'] += $user['total_uploads'] ?? 0;
            $stats['total_logins'] += $user['total_logins'] ?? 0;
            $stats['total_storage'] += $user['storage_used'] ?? 0;
        }
        
        return $stats;
    }
    
    /**
     * Migrate login history
     */
    private function migrateLoginHistory($oldUsername, $newUsername)
    {
        if (!file_exists($this->loginHistoryFile)) {
            return;
        }
        
        $history = json_decode(file_get_contents($this->loginHistoryFile), true) ?: [];
        
        if (isset($history[$oldUsername])) {
            $history[$newUsername] = $history[$oldUsername];
            unset($history[$oldUsername]);
            file_put_contents($this->loginHistoryFile, json_encode($history, JSON_PRETTY_PRINT));
        }
    }
    
    /**
     * Migrate activity log
     */
    private function migrateActivityLog($oldUsername, $newUsername)
    {
        if (!file_exists($this->activityLogFile)) {
            return;
        }
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        
        if (isset($log[$oldUsername])) {
            $log[$newUsername] = $log[$oldUsername];
            unset($log[$oldUsername]);
            file_put_contents($this->activityLogFile, json_encode($log, JSON_PRETTY_PRINT));
        }
    }
    
    /**
     * Delete directory recursively
     */
    private function deleteDirectory($dir)
    {
        if (!is_dir($dir)) {
            return false;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->deleteDirectory($path) : unlink($path);
        }
        
        return rmdir($dir);
    }
}
