<?php
/**
 * FolderManager.php
 * Handles all folder operations: create, read, update, delete, organize
 */

class FolderManager {
    private $dataFile;
    private $storageDir;
    
    public function __construct() {
        $this->dataFile = STORAGE_PATH . '/folders.json';
        $this->storageDir = STORAGE_PATH . '/users';
        $this->initDataFile();
    }
    
    /**
     * Initialize folders.json if not exists
     */
    private function initDataFile() {
        if (!file_exists($this->dataFile)) {
            file_put_contents($this->dataFile, json_encode([], JSON_PRETTY_PRINT));
        }
    }
    
    /**
     * Load all folders
     */
    private function loadFolders() {
        if (!file_exists($this->dataFile)) {
            return [];
        }
        
        $data = file_get_contents($this->dataFile);
        return json_decode($data, true) ?: [];
    }
    
    /**
     * Save folders
     */
    private function saveFolders($folders) {
        // Backup before save
        if (file_exists($this->dataFile)) {
            $backup = $this->dataFile . '.backup_' . date('YmdHis');
            copy($this->dataFile, $backup);
            
            // Keep only last 5 backups
            $backups = glob($this->dataFile . '.backup_*');
            if (count($backups) > 5) {
                array_map('unlink', array_slice($backups, 0, -5));
            }
        }
        
        return file_put_contents($this->dataFile, json_encode($folders, JSON_PRETTY_PRINT));
    }
    
    /**
     * Generate unique folder ID
     */
    private function generateFolderId() {
        return 'folder_' . bin2hex(random_bytes(8)) . '_' . time();
    }
    
    /**
     * Create new folder
     */
    public function createFolder($username, $name, $visibility = 'private', $description = '', $parentId = null) {
        $folders = $this->loadFolders();
        
        // Validate name
        if (empty(trim($name))) {
            return ['error' => 'Folder name cannot be empty'];
        }
        
        // Check for duplicate name (same owner, same parent)
        foreach ($folders as $folder) {
            if ($folder['owner'] === $username && 
                $folder['name'] === $name && 
                $folder['parent_id'] === $parentId) {
                return ['error' => 'Folder with this name already exists'];
            }
        }
        
        $folderId = $this->generateFolderId();
        
        $folders[$folderId] = [
            'id' => $folderId,
            'name' => sanitizeFilename($name),
            'owner' => $username,
            'visibility' => in_array($visibility, ['private', 'shared']) ? $visibility : 'private',
            'description' => htmlspecialchars($description),
            'parent_id' => $parentId,
            'created' => time(),
            'updated' => time(),
            'image_count' => 0,
            'thumbnail' => null
        ];
        
        if ($this->saveFolders($folders)) {
            // Log activity
            $userManager = new UserManager();
            $userManager->logActivity($username, 'folder_created', "Created folder: {$name}");
            
            return ['success' => true, 'folder_id' => $folderId, 'folder' => $folders[$folderId]];
        }
        
        return ['error' => 'Failed to create folder'];
    }
    
    /**
     * Get folder by ID
     */
    public function getFolder($folderId) {
        $folders = $this->loadFolders();
        return $folders[$folderId] ?? null;
    }
    
    /**
     * Update folder
     */
    public function updateFolder($folderId, $data) {
        $folders = $this->loadFolders();
        
        if (!isset($folders[$folderId])) {
            return ['error' => 'Folder not found'];
        }
        
        $folder = $folders[$folderId];
        
        // Update allowed fields
        if (isset($data['name'])) {
            $folder['name'] = sanitizeFilename($data['name']);
        }
        
        if (isset($data['description'])) {
            $folder['description'] = htmlspecialchars($data['description']);
        }
        
        if (isset($data['visibility']) && in_array($data['visibility'], ['private', 'shared'])) {
            $folder['visibility'] = $data['visibility'];
        }
        
        if (isset($data['image_count'])) {
            $folder['image_count'] = (int)$data['image_count'];
        }
        
        if (isset($data['thumbnail'])) {
            $folder['thumbnail'] = $data['thumbnail'];
        }
        
        $folder['updated'] = time();
        $folders[$folderId] = $folder;
        
        if ($this->saveFolders($folders)) {
            // Log activity
            $userManager = new UserManager();
            $userManager->logActivity($folder['owner'], 'folder_updated', "Updated folder: {$folder['name']}");
            
            return ['success' => true, 'folder' => $folder];
        }
        
        return ['error' => 'Failed to update folder'];
    }
    
    /**
     * Delete folder
     */
    public function deleteFolder($folderId, $deleteImages = false) {
        $folders = $this->loadFolders();
        
        if (!isset($folders[$folderId])) {
            return ['error' => 'Folder not found'];
        }
        
        $folder = $folders[$folderId];
        
        // If deleteImages is true, delete all images in folder
        if ($deleteImages) {
            $imageManager = new ImageManager();
            $images = $imageManager->getImagesByFolder($folderId);
            
            foreach ($images as $image) {
                $imageManager->deleteImage($image['id']);
            }
        } else {
            // Move images to no folder (root)
            $imageManager = new ImageManager();
            $imageManager->moveImagesToRoot($folderId);
        }
        
        unset($folders[$folderId]);
        
        if ($this->saveFolders($folders)) {
            // Log activity
            $userManager = new UserManager();
            $userManager->logActivity($folder['owner'], 'folder_deleted', "Deleted folder: {$folder['name']}");
            
            return ['success' => true];
        }
        
        return ['error' => 'Failed to delete folder'];
    }
    
    /**
     * Get all folders for a user
     */
    public function getUserFolders($username, $includeShared = false) {
        $folders = $this->loadFolders();
        $userFolders = [];
        
        foreach ($folders as $folder) {
            // User's own folders
            if ($folder['owner'] === $username) {
                $userFolders[] = $folder;
            }
            // Shared folders from others (if includeShared)
            elseif ($includeShared && $folder['visibility'] === 'shared') {
                $userFolders[] = $folder;
            }
        }
        
        // Sort by created date (newest first)
        usort($userFolders, function($a, $b) {
            return $b['created'] - $a['created'];
        });
        
        return $userFolders;
    }
    
    /**
     * Get all shared folders (public gallery)
     */
    public function getSharedFolders() {
        $folders = $this->loadFolders();
        $sharedFolders = [];
        
        foreach ($folders as $folder) {
            if ($folder['visibility'] === 'shared') {
                $sharedFolders[] = $folder;
            }
        }
        
        // Sort by created date (newest first)
        usort($sharedFolders, function($a, $b) {
            return $b['created'] - $a['created'];
        });
        
        return $sharedFolders;
    }
    
    /**
     * Check if user owns folder
     */
    public function ownsFolder($folderId, $username) {
        $folder = $this->getFolder($folderId);
        return $folder && $folder['owner'] === $username;
    }
    
    /**
     * Check if user can access folder
     */
    public function canAccessFolder($folderId, $username, $isAdmin = false) {
        $folder = $this->getFolder($folderId);
        
        if (!$folder) {
            return false;
        }
        
        // Admin can access all
        if ($isAdmin) {
            return true;
        }
        
        // Owner can access
        if ($folder['owner'] === $username) {
            return true;
        }
        
        // Anyone can access shared
        if ($folder['visibility'] === 'shared') {
            return true;
        }
        
        return false;
    }
    
    /**
     * Search folders
     */
    public function searchFolders($query, $username = null, $isAdmin = false) {
        $folders = $this->loadFolders();
        $results = [];
        $query = strtolower(trim($query));
        
        if (empty($query)) {
            return [];
        }
        
        foreach ($folders as $folder) {
            // Check access
            if (!$isAdmin) {
                if ($folder['owner'] !== $username && $folder['visibility'] !== 'shared') {
                    continue;
                }
            }
            
            // Search in name and description
            if (strpos(strtolower($folder['name']), $query) !== false ||
                strpos(strtolower($folder['description']), $query) !== false) {
                $results[] = $folder;
            }
        }
        
        return $results;
    }
    
    /**
     * Get folder statistics
     */
    public function getFolderStats($username) {
        $folders = $this->getUserFolders($username);
        
        $stats = [
            'total_folders' => count($folders),
            'private_folders' => 0,
            'shared_folders' => 0,
            'total_images_in_folders' => 0
        ];
        
        foreach ($folders as $folder) {
            if ($folder['visibility'] === 'private') {
                $stats['private_folders']++;
            } else {
                $stats['shared_folders']++;
            }
            $stats['total_images_in_folders'] += $folder['image_count'];
        }
        
        return $stats;
    }
    
    /**
     * Increment folder image count
     */
    public function incrementImageCount($folderId) {
        $folder = $this->getFolder($folderId);
        if ($folder) {
            return $this->updateFolder($folderId, [
                'image_count' => $folder['image_count'] + 1
            ]);
        }
        return false;
    }
    
    /**
     * Decrement folder image count
     */
    public function decrementImageCount($folderId) {
        $folder = $this->getFolder($folderId);
        if ($folder && $folder['image_count'] > 0) {
            return $this->updateFolder($folderId, [
                'image_count' => $folder['image_count'] - 1
            ]);
        }
        return false;
    }
}
