<?php
/**
 * CSRF Protection Class
 * Enhanced security for form submissions
 */

class CSRF
{
    /**
     * Generate CSRF token
     */
    public static function generateToken()
    {
        if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
        }
        return $_SESSION[CSRF_TOKEN_NAME];
    }
    
    /**
     * Get current token
     */
    public static function getToken()
    {
        return self::generateToken();
    }
    
    /**
     * Generate hidden input field
     */
    public static function field()
    {
        $token = self::getToken();
        return '<input type="hidden" name="' . CSRF_TOKEN_NAME . '" value="' . htmlspecialchars($token) . '">';
    }
    
    /**
     * Verify CSRF token
     */
    public static function verify()
    {
        $token = $_POST[CSRF_TOKEN_NAME] ?? $_GET[CSRF_TOKEN_NAME] ?? '';
        
        if (empty($token)) {
            return false;
        }
        
        if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
            return false;
        }
        
        return hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }
    
    /**
     * Verify or die with error
     */
    public static function verifyOrDie()
    {
        if (!self::verify()) {
            http_response_code(403);
            die('CSRF token validation failed');
        }
    }
    
    /**
     * Verify or JSON error
     */
    public static function verifyOrJsonError()
    {
        if (!self::verify()) {
            jsonResponse(['error' => 'CSRF token validation failed'], 403);
        }
    }
    
    /**
     * Regenerate token (use after form submission)
     */
    public static function regenerate()
    {
        unset($_SESSION[CSRF_TOKEN_NAME]);
        return self::generateToken();
    }
}
