<?php
/**
 * Authentication Class - ProMax Version
 * Enhanced security and user management
 */

class Auth
{
    private static $userManager;
    
    /**
     * Initialize UserManager
     */
    private static function getUserManager()
    {
        if (!self::$userManager) {
            self::$userManager = new UserManager();
        }
        return self::$userManager;
    }
    
    /**
     * Login user
     */
    public static function login($username, $password, $rememberMe = false)
    {
        $userManager = self::getUserManager();
        
        // Verify credentials
        if (!$userManager->verifyCredentials($username, $password)) {
            // Log failed attempt
            $userManager->logLoginAttempt($username, false);
            return false;
        }
        
        // Get user data
        $user = $userManager->getUser($username);
        
        // Check if user is active
        if ($user['status'] !== 'active') {
            return false;
        }
        
        // Regenerate session ID for security
        session_regenerate_id(true);
        
        // Set session variables
        $_SESSION['logged_in'] = true;
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $username;
        $_SESSION['role'] = $user['role'];
        $_SESSION['display_name'] = $user['display_name'];
        $_SESSION['login_time'] = time();
        $_SESSION['login_ip'] = getClientIP();
        
        // Update last login
        $userManager->updateLastLogin($username);
        
        // Log successful login
        $userManager->logLoginAttempt($username, true);
        $userManager->logActivity($username, 'login', 'User logged in successfully');
        
        return true;
    }
    
    /**
     * Logout user
     */
    public static function logout()
    {
        if (self::isLoggedIn()) {
            $username = $_SESSION['username'] ?? null;
            
            if ($username) {
                $userManager = self::getUserManager();
                $userManager->logActivity($username, 'logout', 'User logged out');
            }
        }
        
        // Clear session
        $_SESSION = [];
        
        // Destroy session cookie
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time() - 3600, '/');
        }
        
        // Destroy session
        session_destroy();
        
        // Start new clean session
        session_start();
    }
    
    /**
     * Check if user is logged in
     */
    public static function isLoggedIn()
    {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }
    
    /**
     * Check if user is admin
     */
    public static function isAdmin()
    {
        return self::isLoggedIn() && isset($_SESSION['role']) && $_SESSION['role'] === 'admin';
    }
    
    /**
     * Get current user data
     */
    public static function getCurrentUser()
    {
        if (!self::isLoggedIn()) {
            return null;
        }
        
        $userManager = self::getUserManager();
        return $userManager->getUser($_SESSION['username']);
    }
    
    /**
     * Get current username
     */
    public static function getUsername()
    {
        return $_SESSION['username'] ?? null;
    }
    
    /**
     * Get current user ID
     */
    public static function getUserId()
    {
        return $_SESSION['user_id'] ?? null;
    }
    
    /**
     * Get current user role
     */
    public static function getRole()
    {
        return $_SESSION['role'] ?? null;
    }
    
    /**
     * Require login (redirect to login if not logged in)
     */
    public static function requireLogin()
    {
        if (!self::isLoggedIn()) {
            header('Location: ' . PUBLIC_URL . '/login.php');
            exit;
        }
    }
    
    /**
     * Require admin (show 403 if not admin)
     */
    public static function requireAdmin()
    {
        self::requireLogin();
        
        if (!self::isAdmin()) {
            http_response_code(403);
            die('Access Denied: Admin privileges required');
        }
    }
    
    /**
     * Redirect if already logged in
     */
    public static function redirectIfLoggedIn()
    {
        if (self::isLoggedIn()) {
            header('Location: ' . PUBLIC_URL . '/index.php');
            exit;
        }
    }
    
    /**
     * Check if current user owns resource
     */
    public static function ownsResource($username)
    {
        if (!self::isLoggedIn()) {
            return false;
        }
        
        // Admin can access everything
        if (self::isAdmin()) {
            return true;
        }
        
        // Check if current user is the owner
        return self::getUsername() === $username;
    }
    
    /**
     * Validate session integrity
     */
    public static function validateSession()
    {
        if (!self::isLoggedIn()) {
            return false;
        }
        
        // Check session timeout (30 minutes)
        if (isset($_SESSION['login_time']) && (time() - $_SESSION['login_time'] > 1800)) {
            self::logout();
            return false;
        }
        
        // Verify user still exists and is active
        $userManager = self::getUserManager();
        $user = $userManager->getUser(self::getUsername());
        
        if (!$user || $user['status'] !== 'active') {
            self::logout();
            return false;
        }
        
        return true;
    }
}
