<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/CSRF.php';
require_once __DIR__ . '/../includes/UserManager.php';

Auth::requireLogin();

$username = Auth::getUsername();
$userManager = new UserManager();
$user = $userManager->getUser($username);

$success = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    CSRF::verifyOrDie();
    
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_profile':
                $result = $userManager->updateUser($username, [
                    'display_name' => $_POST['display_name'],
                    'email' => $_POST['email'],
                    'bio' => $_POST['bio']
                ]);
                
                if (isset($result['success'])) {
                    $success = 'Profile updated successfully!';
                    $user = $userManager->getUser($username);
                } else {
                    $error = $result['error'];
                }
                break;
                
            case 'change_password':
                if ($_POST['new_password'] !== $_POST['confirm_password']) {
                    $error = 'Passwords do not match!';
                } else {
                    $result = $userManager->changePassword(
                        $username,
                        $_POST['current_password'],
                        $_POST['new_password']
                    );
                    
                    if (isset($result['success'])) {
                        $success = 'Password changed successfully!';
                    } else {
                        $error = $result['error'];
                    }
                }
                break;
                
            case 'upload_avatar':
                if (isset($_FILES['avatar'])) {
                    $result = $userManager->uploadAvatar($username, $_FILES['avatar']);
                    
                    if (isset($result['success'])) {
                        $success = 'Avatar uploaded successfully!';
                        $user = $userManager->getUser($username);
                    } else {
                        $error = $result['error'];
                    }
                }
                break;
                
            case 'delete_avatar':
                $result = $userManager->deleteAvatar($username);
                
                if (isset($result['success'])) {
                    $success = 'Avatar deleted successfully!';
                    $user = $userManager->getUser($username);
                } else {
                    $error = $result['error'];
                }
                break;
        }
    }
}

$activityLog = $userManager->getActivityLog($username, 10);
$loginHistory = $userManager->getLoginHistory($username, 10);
?>
<!DOCTYPE html>
<html lang="en" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - Galeriku ProMax</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --bg-primary: #FFFFFF;
            --bg-secondary: #F9FAFB;
            --bg-card: #FFFFFF;
            --text-primary: #111827;
            --text-secondary: #6B7280;
            --border: #E5E7EB;
            --success: #10B981;
            --error: #EF4444;
            --shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        
        [data-theme="dark"] {
            --primary: #A78BFA;
            --bg-primary: #111827;
            --bg-secondary: #1F2937;
            --bg-card: #374151;
            --text-primary: #F9FAFB;
            --text-secondary: #D1D5DB;
            --border: #4B5563;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-secondary);
            color: var(--text-primary);
            min-height: 100vh;
        }
        
        .header {
            background: var(--bg-primary);
            border-bottom: 1px solid var(--border);
            padding: 1rem 1.5rem;
            box-shadow: var(--shadow);
        }
        
        .header-content {
            max-width: 1000px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
            text-decoration: none;
        }
        
        .logo svg { width: 32px; height: 32px; }
        
        .btn {
            padding: 0.625rem 1rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            border: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.2s;
            text-decoration: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: white;
        }
        
        .btn-secondary {
            background: var(--bg-card);
            color: var(--text-primary);
            border: 1px solid var(--border);
        }
        
        .btn-danger {
            background: var(--error);
            color: white;
        }
        
        .btn svg { width: 18px; height: 18px; }
        
        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 2rem 1.5rem;
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 2rem;
        }
        
        .settings-grid {
            display: grid;
            gap: 1.5rem;
        }
        
        .setting-section {
            background: var(--bg-card);
            border-radius: 12px;
            border: 1px solid var(--border);
            padding: 1.5rem;
        }
        
        .section-title {
            font-size: 1.25rem;
            font-weight: 700;
            margin-bottom: 1rem;
        }
        
        .section-description {
            color: var(--text-secondary);
            font-size: 0.875rem;
            margin-bottom: 1.5rem;
        }
        
        .form-group {
            margin-bottom: 1.25rem;
        }
        
        label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            font-size: 0.875rem;
        }
        
        input, textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-size: 1rem;
            font-family: inherit;
            transition: all 0.2s;
        }
        
        input:focus, textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(139, 92, 246, 0.1);
        }
        
        textarea {
            resize: vertical;
            min-height: 100px;
        }
        
        .avatar-section {
            display: flex;
            gap: 1.5rem;
            align-items: flex-start;
        }
        
        .avatar-preview {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid var(--border);
        }
        
        .avatar-actions {
            flex: 1;
        }
        
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
            border: 1px solid var(--success);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--error);
            border: 1px solid var(--error);
        }
        
        .activity-list {
            list-style: none;
        }
        
        .activity-item {
            padding: 0.75rem 0;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .activity-item:last-child {
            border-bottom: none;
        }
        
        .activity-action {
            font-weight: 600;
        }
        
        .activity-time {
            font-size: 0.875rem;
            color: var(--text-secondary);
        }
        
        @media (max-width: 768px) {
            .avatar-section {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <a href="index.php" class="logo">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <circle cx="12" cy="12" r="3"/>
                    <path d="M12 1v6m0 6v6"/>
                    <path d="m4.93 4.93 4.24 4.24m5.66 5.66 4.24 4.24"/>
                    <path d="M1 12h6m6 0h6"/>
                    <path d="m4.93 19.07 4.24-4.24m5.66-5.66 4.24-4.24"/>
                </svg>
                Settings
            </a>
            <a href="index.php" class="btn btn-secondary">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <line x1="19" y1="12" x2="5" y2="12"/>
                    <polyline points="12 19 5 12 12 5"/>
                </svg>
                Back to Gallery
            </a>
        </div>
    </div>
    
    <div class="container">
        <h1 class="page-title">Account Settings</h1>
        
        <?php if ($success): ?>
        <div class="alert alert-success">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 20px; height: 20px;">
                <polyline points="20 6 9 17 4 12"/>
            </svg>
            <span><?php echo htmlspecialchars($success); ?></span>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="alert alert-error">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 20px; height: 20px;">
                <circle cx="12" cy="12" r="10"/>
                <line x1="12" y1="8" x2="12" y2="12"/>
                <line x1="12" y1="16" x2="12.01" y2="16"/>
            </svg>
            <span><?php echo htmlspecialchars($error); ?></span>
        </div>
        <?php endif; ?>
        
        <div class="settings-grid">
            <!-- Avatar Section -->
            <div class="setting-section">
                <h2 class="section-title">Profile Picture</h2>
                <p class="section-description">Upload a custom avatar or use the auto-generated one</p>
                
                <div class="avatar-section">
                    <img src="user-avatar.php?username=<?php echo urlencode($username); ?>&t=<?php echo time(); ?>" alt="Avatar" class="avatar-preview">
                    
                    <div class="avatar-actions">
                        <form method="POST" enctype="multipart/form-data" style="margin-bottom: 1rem;">
                            <?php echo CSRF::field(); ?>
                            <input type="hidden" name="action" value="upload_avatar">
                            <div class="form-group">
                                <input type="file" name="avatar" accept="image/png,image/jpeg,image/jpg" required>
                            </div>
                            <button type="submit" class="btn btn-primary">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                    <polyline points="17 8 12 3 7 8"/>
                                    <line x1="12" y1="3" x2="12" y2="15"/>
                                </svg>
                                Upload New Avatar
                            </button>
                        </form>
                        
                        <?php if ($user['has_avatar']): ?>
                        <form method="POST">
                            <?php echo CSRF::field(); ?>
                            <input type="hidden" name="action" value="delete_avatar">
                            <button type="submit" class="btn btn-danger">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="3 6 5 6 21 6"/>
                                    <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
                                </svg>
                                Delete Avatar
                            </button>
                        </form>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Profile Section -->
            <div class="setting-section">
                <h2 class="section-title">Profile Information</h2>
                <p class="section-description">Update your public profile details</p>
                
                <form method="POST">
                    <?php echo CSRF::field(); ?>
                    <input type="hidden" name="action" value="update_profile">
                    
                    <div class="form-group">
                        <label>Username (cannot be changed)</label>
                        <input type="text" value="<?php echo htmlspecialchars($user['username']); ?>" disabled>
                    </div>
                    
                    <div class="form-group">
                        <label>Display Name</label>
                        <input type="text" name="display_name" value="<?php echo htmlspecialchars($user['display_name']); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Email</label>
                        <input type="email" name="email" value="<?php echo htmlspecialchars($user['email']); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label>Bio</label>
                        <textarea name="bio"><?php echo htmlspecialchars($user['bio']); ?></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="20 6 9 17 4 12"/>
                        </svg>
                        Save Changes
                    </button>
                </form>
            </div>
            
            <!-- Password Section -->
            <div class="setting-section">
                <h2 class="section-title">Change Password</h2>
                <p class="section-description">Update your password to keep your account secure</p>
                
                <form method="POST">
                    <?php echo CSRF::field(); ?>
                    <input type="hidden" name="action" value="change_password">
                    
                    <div class="form-group">
                        <label>Current Password</label>
                        <input type="password" name="current_password" required>
                    </div>
                    
                    <div class="form-group">
                        <label>New Password</label>
                        <input type="password" name="new_password" required minlength="6">
                    </div>
                    
                    <div class="form-group">
                        <label>Confirm New Password</label>
                        <input type="password" name="confirm_password" required minlength="6">
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                            <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                        </svg>
                        Update Password
                    </button>
                </form>
            </div>
            
            <!-- Activity Log -->
            <div class="setting-section">
                <h2 class="section-title">Recent Activity</h2>
                <p class="section-description">Your recent account activity</p>
                
                <?php if (!empty($activityLog)): ?>
                <ul class="activity-list">
                    <?php foreach ($activityLog as $activity): ?>
                    <li class="activity-item">
                        <div>
                            <div class="activity-action"><?php echo htmlspecialchars($activity['description']); ?></div>
                            <div class="activity-time"><?php echo timeAgo($activity['timestamp']); ?></div>
                        </div>
                    </li>
                    <?php endforeach; ?>
                </ul>
                <?php else: ?>
                <p style="color: var(--text-secondary);">No recent activity</p>
                <?php endif; ?>
            </div>
            
            <!-- Account Stats -->
            <div class="setting-section">
                <h2 class="section-title">Account Statistics</h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 1rem; margin-top: 1rem;">
                    <div style="text-align: center; padding: 1rem; background: var(--bg-secondary); border-radius: 8px;">
                        <div style="font-size: 1.5rem; font-weight: 700; color: var(--primary);"><?php echo $user['total_uploads']; ?></div>
                        <div style="font-size: 0.875rem; color: var(--text-secondary);">Total Uploads</div>
                    </div>
                    <div style="text-align: center; padding: 1rem; background: var(--bg-secondary); border-radius: 8px;">
                        <div style="font-size: 1.5rem; font-weight: 700; color: var(--primary);"><?php echo formatBytes($user['storage_used']); ?></div>
                        <div style="font-size: 0.875rem; color: var(--text-secondary);">Storage Used</div>
                    </div>
                    <div style="text-align: center; padding: 1rem; background: var(--bg-secondary); border-radius: 8px;">
                        <div style="font-size: 1.5rem; font-weight: 700; color: var(--primary);"><?php echo $user['total_logins']; ?></div>
                        <div style="font-size: 0.875rem; color: var(--text-secondary);">Total Logins</div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
