<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/CSRF.php';
require_once __DIR__ . '/../includes/PngHandler.php';
require_once __DIR__ . '/../includes/UserManager.php';
require_once __DIR__ . '/../includes/ImageManager.php';

Auth::requireLogin();

$username = Auth::getUsername();
$userId = Auth::getUserId();
$isAdmin = Auth::isAdmin();
$currentUser = Auth::getCurrentUser();

$userManager = new UserManager();
$imageManager = new ImageManager();
$handler = new PngHandler($userId, $username);

// Get view mode: my (private) or shared (public gallery)
$view = $_GET['view'] ?? 'my';

// Get pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = ITEMS_PER_PAGE;

// Get images based on view mode
if ($view === 'shared') {
    $allImages = $imageManager->getSharedImages();
} else {
    $allImages = $imageManager->getUserImages($username);
}

// Manual pagination
$totalImages = count($allImages);
$totalPages = ceil($totalImages / $perPage);
$offset = ($page - 1) * $perPage;
$images = array_slice($allImages, $offset, $perPage);

// Convert to format expected by template
$pagination = [
    'images' => $images,
    'current_page' => $page,
    'total_pages' => $totalPages,
    'total_images' => $totalImages
];

// Get user stats
$userStats = $isAdmin ? $userManager->getUserStats() : null;
$storageUsed = $currentUser['storage_used'] ?? 0;
$totalUploads = $currentUser['total_uploads'] ?? 0;
?>
<!DOCTYPE html>
<html lang="en" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gallery - Galeriku ProMax</title>
    <meta name="robots" content="noindex">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --primary-darker: #6D28D9;
            --bg-primary: #FFFFFF;
            --bg-secondary: #F9FAFB;
            --bg-card: #FFFFFF;
            --text-primary: #111827;
            --text-secondary: #6B7280;
            --border: #E5E7EB;
            --shadow: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0,0,0,0.1);
        }
        
        [data-theme="dark"] {
            --primary: #A78BFA;
            --primary-dark: #8B5CF6;
            --bg-primary: #111827;
            --bg-secondary: #1F2937;
            --bg-card: #374151;
            --text-primary: #F9FAFB;
            --text-secondary: #D1D5DB;
            --border: #4B5563;
            --shadow: 0 1px 3px rgba(0,0,0,0.3);
            --shadow-lg: 0 10px 15px -3px rgba(0,0,0,0.3);
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-secondary);
            color: var(--text-primary);
            transition: background 0.3s, color 0.3s;
        }
        
        /* Header */
        .header {
            background: var(--bg-primary);
            border-bottom: 1px solid var(--border);
            padding: 1rem 1.5rem;
            position: sticky;
            top: 0;
            z-index: 100;
            backdrop-filter: blur(10px);
            box-shadow: var(--shadow);
        }
        
        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 1rem;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
            text-decoration: none;
        }
        
        .logo svg {
            width: 32px;
            height: 32px;
        }
        
        .header-actions {
            display: flex;
            gap: 0.5rem;
            align-items: center;
        }
        
        .btn {
            padding: 0.625rem 1rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            border: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.2s;
            text-decoration: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(139, 92, 246, 0.4);
        }
        
        .btn-secondary {
            background: var(--bg-card);
            color: var(--text-primary);
            border: 1px solid var(--border);
        }
        
        .btn-secondary:hover {
            background: var(--bg-secondary);
        }
        
        .btn svg {
            width: 18px;
            height: 18px;
        }
        
        /* User dropdown */
        .user-menu {
            position: relative;
        }
        
        .user-btn {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem;
            border-radius: 8px;
            background: var(--bg-card);
            border: 1px solid var(--border);
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .user-btn:hover {
            background: var(--bg-secondary);
        }
        
        .user-avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .dropdown {
            position: absolute;
            top: calc(100% + 0.5rem);
            right: 0;
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 12px;
            box-shadow: var(--shadow-lg);
            min-width: 200px;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.2s;
        }
        
        .dropdown.active {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }
        
        .dropdown-item {
            padding: 0.75rem 1rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            color: var(--text-primary);
            text-decoration: none;
            transition: background 0.2s;
            border-bottom: 1px solid var(--border);
        }
        
        .dropdown-item:last-child {
            border-bottom: none;
        }
        
        .dropdown-item:hover {
            background: var(--bg-secondary);
        }
        
        .dropdown-item svg {
            width: 18px;
            height: 18px;
        }
        
        /* Stats Bar */
        .stats-bar {
            background: var(--bg-primary);
            border-bottom: 1px solid var(--border);
            padding: 1rem 1.5rem;
        }
        
        .stats-content {
            max-width: 1400px;
            margin: 0 auto;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }
        
        .stat-card {
            background: var(--bg-card);
            padding: 1rem;
            border-radius: 12px;
            border: 1px solid var(--border);
        }
        
        .stat-label {
            font-size: 0.75rem;
            color: var(--text-secondary);
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            margin-bottom: 0.25rem;
        }
        
        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--primary);
        }
        
        /* Gallery */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem 1.5rem;
        }
        
        .gallery-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        .gallery-title {
            font-size: 1.75rem;
            font-weight: 700;
        }
        
        .gallery-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 3rem;
        }
        
        .image-card {
            background: var(--bg-card);
            border-radius: 12px;
            overflow: hidden;
            border: 1px solid var(--border);
            transition: all 0.3s;
            cursor: pointer;
        }
        
        .image-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
        }
        
        .image-wrapper {
            aspect-ratio: 1;
            background: repeating-conic-gradient(#f0f0f0 0% 25%, transparent 0% 50%) 50% / 20px 20px;
            position: relative;
            overflow: hidden;
        }
        
        [data-theme="dark"] .image-wrapper {
            background: repeating-conic-gradient(#374151 0% 25%, transparent 0% 50%) 50% / 20px 20px;
        }
        
        .image-wrapper img {
            width: 100%;
            height: 100%;
            object-fit: contain;
            transition: transform 0.3s;
        }
        
        .image-card:hover .image-wrapper img {
            transform: scale(1.05);
        }
        
        .image-info {
            padding: 1rem;
        }
        
        .image-name {
            font-weight: 600;
            font-size: 0.875rem;
            margin-bottom: 0.5rem;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        .image-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.75rem;
            color: var(--text-secondary);
            margin-bottom: 0.75rem;
        }
        
        .image-actions {
            display: flex;
            gap: 0.5rem;
        }
        
        .btn-icon {
            padding: 0.5rem;
            border-radius: 6px;
            border: 1px solid var(--border);
            background: var(--bg-secondary);
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .btn-icon:hover {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }
        
        .btn-icon svg {
            width: 16px;
            height: 16px;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
        }
        
        .empty-icon {
            width: 80px;
            height: 80px;
            margin: 0 auto 1.5rem;
            color: var(--text-secondary);
        }
        
        .empty-state h2 {
            font-size: 1.5rem;
            margin-bottom: 0.5rem;
        }
        
        .empty-state p {
            color: var(--text-secondary);
            margin-bottom: 1.5rem;
        }
        
        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 0.5rem;
            margin-top: 2rem;
        }
        
        .pagination a,
        .pagination span {
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            border: 1px solid var(--border);
            background: var(--bg-card);
            color: var(--text-primary);
            text-decoration: none;
            font-weight: 600;
            font-size: 0.875rem;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .pagination a:hover {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }
        
        .pagination .active {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }
        
        .pagination .disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Lightbox */
        .lightbox {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.9);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 1000;
            backdrop-filter: blur(10px);
        }
        
        .lightbox.active {
            display: flex;
        }
        
        .lightbox-content {
            max-width: 90vw;
            max-height: 90vh;
            position: relative;
        }
        
        .lightbox img {
            max-width: 100%;
            max-height: 90vh;
            border-radius: 8px;
        }
        
        .lightbox-close {
            position: absolute;
            top: -40px;
            right: 0;
            background: white;
            color: #111;
            border: none;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 1.5rem;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-wrap: wrap;
            }
            
            .gallery-grid {
                grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
                gap: 1rem;
            }
            
            .stats-content {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-content">
            <a href="index.php" class="logo">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                    <polyline points="3.27 6.96 12 12.01 20.73 6.96"/>
                    <line x1="12" y1="22.08" x2="12" y2="12"/>
                </svg>
                Galeriku ProMax
            </a>
            
            <div class="header-actions">
                <a href="admin/upload.php" class="btn btn-primary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="17 8 12 3 7 8"/>
                        <line x1="12" y1="3" x2="12" y2="15"/>
                    </svg>
                    Upload
                </a>
                
                <button class="btn btn-secondary" onclick="toggleTheme()">
                    <svg id="theme-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <circle cx="12" cy="12" r="5"/>
                        <line x1="12" y1="1" x2="12" y2="3"/>
                        <line x1="12" y1="21" x2="12" y2="23"/>
                        <line x1="4.22" y1="4.22" x2="5.64" y2="5.64"/>
                        <line x1="18.36" y1="18.36" x2="19.78" y2="19.78"/>
                        <line x1="1" y1="12" x2="3" y2="12"/>
                        <line x1="21" y1="12" x2="23" y2="12"/>
                        <line x1="4.22" y1="19.78" x2="5.64" y2="18.36"/>
                        <line x1="18.36" y1="5.64" x2="19.78" y2="4.22"/>
                    </svg>
                </button>
                
                <div class="user-menu">
                    <button class="user-btn" onclick="toggleDropdown()">
                        <img src="user-avatar.php?username=<?php echo urlencode($username); ?>" alt="Avatar" class="user-avatar">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 16px; height: 16px;">
                            <polyline points="6 9 12 15 18 9"/>
                        </svg>
                    </button>
                    
                    <div class="dropdown" id="userDropdown">
                        <a href="settings.php" class="dropdown-item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="12" cy="12" r="3"/>
                                <path d="M12 1v6m0 6v6"/>
                                <path d="m4.93 4.93 4.24 4.24m5.66 5.66 4.24 4.24"/>
                                <path d="M1 12h6m6 0h6"/>
                                <path d="m4.93 19.07 4.24-4.24m5.66-5.66 4.24-4.24"/>
                            </svg>
                            Settings
                        </a>
                        <a href="folders.php" class="dropdown-item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"/>
                            </svg>
                            Folders
                        </a>
                        <a href="search.php" class="dropdown-item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="11" cy="11" r="8"/>
                                <path d="m21 21-4.35-4.35"/>
                            </svg>
                            Search
                        </a>
                        <?php if ($isAdmin): ?>
                        <a href="admin/users.php" class="dropdown-item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                                <circle cx="9" cy="7" r="4"/>
                                <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                                <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                            </svg>
                            Manage Users
                        </a>
                        <?php endif; ?>
                        <a href="logout.php" class="dropdown-item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"/>
                                <polyline points="16 17 21 12 16 7"/>
                                <line x1="21" y1="12" x2="9" y2="12"/>
                            </svg>
                            Logout
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Stats Bar -->
    <div class="stats-bar">
        <div class="stats-content">
            <div class="stat-card">
                <div class="stat-label">Total Images</div>
                <div class="stat-value"><?php echo $totalUploads; ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Storage Used</div>
                <div class="stat-value"><?php echo formatBytes($storageUsed); ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Current Page</div>
                <div class="stat-value"><?php echo $page; ?> / <?php echo max(1, $pagination['total_pages']); ?></div>
            </div>
        </div>
    </div>
    
    <!-- View Toggle -->
    <div class="stats-bar" style="border-top: 1px solid var(--border); padding: 0.75rem 1.5rem;">
        <div class="stats-content" style="display: flex; justify-content: center; gap: 0.5rem;">
            <a href="?view=my" class="btn <?php echo $view === 'my' ? 'btn-primary' : 'btn-secondary'; ?>" style="padding: 0.5rem 1rem;">
                🔒 My Photos
            </a>
            <a href="?view=shared" class="btn <?php echo $view === 'shared' ? 'btn-primary' : 'btn-secondary'; ?>" style="padding: 0.5rem 1rem;">
                🌐 Shared Gallery
            </a>
        </div>
    </div>
    
    <!-- Gallery -->
    <div class="container">
        <?php if (!empty($images)): ?>
            <div class="gallery-grid">
                <?php foreach ($images as $image): ?>
                <div class="image-card" onclick="openLightbox('<?php echo htmlspecialchars($image['filename']); ?>')">
                    <div class="image-wrapper">
                        <img src="image.php?type=thumb&file=<?php echo urlencode($image['filename']); ?>" alt="<?php echo htmlspecialchars($image['filename']); ?>">
                    </div>
                    <div class="image-info">
                        <div class="image-name" title="<?php echo htmlspecialchars($image['filename']); ?>">
                            <?php echo htmlspecialchars($image['filename']); ?>
                        </div>
                        <div class="image-meta">
                            <span><?php echo formatBytes($image['size']); ?></span>
                            <span><?php echo date('M d, Y', $image['uploaded']); ?></span>
                        </div>
                        <div class="image-actions">
                            <button class="btn-icon" onclick="event.stopPropagation(); downloadImage('<?php echo htmlspecialchars($image['filename']); ?>')" title="Download">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                    <polyline points="7 10 12 15 17 10"/>
                                    <line x1="12" y1="15" x2="12" y2="3"/>
                                </svg>
                            </button>
                            <button class="btn-icon" onclick="event.stopPropagation(); renameImage('<?php echo htmlspecialchars($image['filename']); ?>')" title="Rename">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
                                    <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
                                </svg>
                            </button>
                            <button class="btn-icon" onclick="event.stopPropagation(); deleteImage('<?php echo htmlspecialchars($image['filename']); ?>')" title="Delete">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="3 6 5 6 21 6"/>
                                    <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            
            <!-- Pagination -->
            <?php if ($pagination['total_pages'] > 1): ?>
            <div class="pagination">
                <?php if ($page > 1): ?>
                    <a href="?page=1&view=<?php echo $view; ?>">&laquo;&laquo;</a>
                    <a href="?page=<?php echo $page - 1; ?>&view=<?php echo $view; ?>">&laquo;</a>
                <?php else: ?>
                    <span class="disabled">&laquo;&laquo;</span>
                    <span class="disabled">&laquo;</span>
                <?php endif; ?>
                
                <?php
                $start = max(1, $page - 2);
                $end = min($pagination['total_pages'], $page + 2);
                for ($i = $start; $i <= $end; $i++):
                ?>
                    <?php if ($i == $page): ?>
                        <span class="active"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?page=<?php echo $i; ?>&view=<?php echo $view; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                
                <?php if ($page < $pagination['total_pages']): ?>
                    <a href="?page=<?php echo $page + 1; ?>&view=<?php echo $view; ?>">&raquo;</a>
                    <a href="?page=<?php echo $pagination['total_pages']; ?>&view=<?php echo $view; ?>">&raquo;&raquo;</a>
                <?php else: ?>
                    <span class="disabled">&raquo;</span>
                    <span class="disabled">&raquo;&raquo;</span>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        <?php else: ?>
            <div class="empty-state">
                <svg class="empty-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                    <circle cx="8.5" cy="8.5" r="1.5"/>
                    <polyline points="21 15 16 10 5 21"/>
                </svg>
                <h2>No Images Yet</h2>
                <p>Start by uploading your first PNG image</p>
                <a href="admin/upload.php" class="btn btn-primary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="17 8 12 3 7 8"/>
                        <line x1="12" y1="3" x2="12" y2="15"/>
                    </svg>
                    Upload Now
                </a>
            </div>
        <?php endif; ?>
    </div>
    
    <!-- Lightbox -->
    <div class="lightbox" id="lightbox" onclick="closeLightbox()">
        <div class="lightbox-content" onclick="event.stopPropagation()">
            <button class="lightbox-close" onclick="closeLightbox()">×</button>
            <img id="lightboxImage" src="" alt="">
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
        // Theme Toggle
        function toggleTheme() {
            const html = document.documentElement;
            const current = html.getAttribute('data-theme');
            const newTheme = current === 'dark' ? 'light' : 'dark';
            html.setAttribute('data-theme', newTheme);
            localStorage.setItem('theme', newTheme);
        }
        
        // Load theme
        (function() {
            const saved = localStorage.getItem('theme') || 'light';
            document.documentElement.setAttribute('data-theme', saved);
        })();
        
        // Dropdown
        function toggleDropdown() {
            document.getElementById('userDropdown').classList.toggle('active');
        }
        
        document.addEventListener('click', (e) => {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').classList.remove('active');
            }
        });
        
        // Lightbox
        function openLightbox(filename) {
            const lightbox = document.getElementById('lightbox');
            const img = document.getElementById('lightboxImage');
            img.src = 'image.php?type=optimized&file=' + encodeURIComponent(filename);
            lightbox.classList.add('active');
        }
        
        function closeLightbox() {
            document.getElementById('lightbox').classList.remove('active');
        }
        
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') closeLightbox();
        });
        
        // Download
        function downloadImage(filename) {
            window.location.href = 'image.php?type=original&file=' + encodeURIComponent(filename) + '&download=1';
        }
        
        // Rename
        function renameImage(filename) {
            const name = filename.replace('.png', '');
            Swal.fire({
                title: 'Rename Image',
                input: 'text',
                inputValue: name,
                showCancelButton: true,
                confirmButtonText: 'Rename',
                confirmButtonColor: '#8B5CF6',
                inputValidator: (value) => {
                    if (!value) return 'Name cannot be empty';
                    if (!/^[a-zA-Z0-9_-]+$/.test(value)) return 'Only letters, numbers, underscore and dash allowed';
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = 'admin/rename.php?file=' + encodeURIComponent(filename) + '&new_name=' + encodeURIComponent(result.value);
                }
            });
        }
        
        // Delete
        function deleteImage(filename) {
            Swal.fire({
                title: 'Delete Image?',
                text: 'This action cannot be undone!',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#EF4444',
                confirmButtonText: 'Yes, delete it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = 'admin/delete.php?file=' + encodeURIComponent(filename) + '&confirm=1';
                }
            });
        }
    </script>
</body>
</html>
