<?php
/**
 * Folders Page
 * Display user's folders in grid view
 */

require_once '../includes/config.php';
require_once '../includes/Auth.php';
require_once '../includes/FolderManager.php';
require_once '../includes/PermissionManager.php';
require_once '../includes/ImageManager.php';

// Require login
Auth::requireLogin();

$username = Auth::getUsername();
$isAdmin = Auth::isAdmin();
$folderManager = new FolderManager();
$permManager = new PermissionManager();
$imageManager = new ImageManager();

// Get view mode
$view = $_GET['view'] ?? 'my'; // my|shared

// Get folders
if ($view === 'shared') {
    $folders = $folderManager->getSharedFolders();
} else {
    $folders = $folderManager->getUserFolders($username, false);
}

// Get statistics
$stats = $folderManager->getFolderStats($username);
$canCreateFolders = $permManager->canCreateFolders($username);

// Success/Error messages
$messages = [
    'folder_created' => ['type' => 'success', 'text' => 'Folder created successfully!'],
    'folder_updated' => ['type' => 'success', 'text' => 'Folder updated successfully!'],
    'folder_deleted' => ['type' => 'success', 'text' => 'Folder deleted successfully!']
];

$message = isset($_GET['success']) && isset($messages[$_GET['success']]) ? $messages[$_GET['success']] : null;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Folders - Galeriku ProMax</title>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --success: #10B981;
            --warning: #F59E0B;
            --danger: #EF4444;
            --bg-dark: #0F172A;
            --bg-card: #1E293B;
            --bg-hover: #334155;
            --text-primary: #F1F5F9;
            --text-secondary: #94A3B8;
            --border: #334155;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, var(--bg-dark) 0%, #1a1a2e 100%);
            color: var(--text-primary);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border);
            padding: 1rem 2rem;
            position: sticky;
            top: 0;
            z-index: 100;
            backdrop-filter: blur(10px);
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 1rem;
        }

        .logo {
            font-size: 1.5rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary) 0%, #EC4899 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .nav-links {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .nav-link {
            padding: 0.5rem 1rem;
            text-decoration: none;
            color: var(--text-secondary);
            border-radius: 0.5rem;
            transition: all 0.3s;
            font-size: 0.875rem;
        }

        .nav-link:hover,
        .nav-link.active {
            color: var(--text-primary);
            background: var(--bg-hover);
        }

        .btn {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 0.5rem;
            cursor: pointer;
            font-size: 0.875rem;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        /* Main Content */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }

        .page-header {
            margin-bottom: 2rem;
        }

        .page-title {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }

        .page-subtitle {
            color: var(--text-secondary);
        }

        /* Stats Bar */
        .stats-bar {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 0.75rem;
            padding: 1.5rem;
            transition: all 0.3s;
        }

        .stat-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary) 0%, #EC4899 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .stat-label {
            color: var(--text-secondary);
            font-size: 0.875rem;
            margin-top: 0.5rem;
        }

        /* View Toggle */
        .view-toggle {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 2rem;
        }

        .view-btn {
            padding: 0.75rem 1.5rem;
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 0.5rem;
            color: var(--text-secondary);
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
            font-size: 0.875rem;
        }

        .view-btn.active {
            background: var(--primary);
            border-color: var(--primary);
            color: white;
        }

        .view-btn:hover {
            border-color: var(--primary);
        }

        /* Folders Grid */
        .folders-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .folder-card {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 1rem;
            overflow: hidden;
            transition: all 0.3s;
            cursor: pointer;
            text-decoration: none;
            color: inherit;
        }

        .folder-card:hover {
            border-color: var(--primary);
            transform: translateY(-4px);
            box-shadow: 0 12px 24px rgba(139, 92, 246, 0.2);
        }

        .folder-thumbnail {
            width: 100%;
            height: 180px;
            background: linear-gradient(135deg, rgba(139, 92, 246, 0.1) 0%, rgba(236, 72, 153, 0.1) 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 4rem;
            position: relative;
            overflow: hidden;
        }

        .folder-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .folder-icon {
            font-size: 4rem;
        }

        .folder-content {
            padding: 1.25rem;
        }

        .folder-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 0.75rem;
        }

        .folder-name {
            font-size: 1.125rem;
            font-weight: 600;
            margin-bottom: 0.25rem;
        }

        .folder-description {
            color: var(--text-secondary);
            font-size: 0.875rem;
            margin-bottom: 0.75rem;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .folder-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.875rem;
            color: var(--text-secondary);
        }

        .folder-count {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }

        .badge {
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .badge-private {
            background: rgba(239, 68, 68, 0.2);
            color: var(--danger);
        }

        .badge-shared {
            background: rgba(16, 185, 129, 0.2);
            color: var(--success);
        }

        .folder-actions {
            position: absolute;
            top: 0.75rem;
            right: 0.75rem;
            display: flex;
            gap: 0.5rem;
            opacity: 0;
            transition: opacity 0.3s;
        }

        .folder-card:hover .folder-actions {
            opacity: 1;
        }

        .action-btn {
            width: 32px;
            height: 32px;
            border-radius: 0.5rem;
            background: rgba(15, 23, 42, 0.9);
            backdrop-filter: blur(10px);
            border: 1px solid var(--border);
            color: var(--text-primary);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s;
        }

        .action-btn:hover {
            background: var(--primary);
            border-color: var(--primary);
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
        }

        .empty-icon {
            font-size: 5rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        .empty-title {
            font-size: 1.5rem;
            margin-bottom: 0.5rem;
        }

        .empty-text {
            color: var(--text-secondary);
            margin-bottom: 2rem;
        }

        @media (max-width: 768px) {
            .container {
                padding: 1rem;
            }

            .header {
                padding: 1rem;
            }

            .page-title {
                font-size: 1.5rem;
            }

            .folders-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-content">
            <div class="logo">📁 Galeriku ProMax</div>
            <div class="nav-links">
                <a href="index.php" class="nav-link">Photos</a>
                <a href="folders.php" class="nav-link active">Folders</a>
                <?php if ($isAdmin): ?>
                    <a href="admin/users.php" class="nav-link">Users</a>
                    <a href="admin/permissions.php" class="nav-link">Permissions</a>
                <?php endif; ?>
                <a href="settings.php" class="nav-link">Settings</a>
                <a href="logout.php" class="nav-link">Logout</a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">My Folders</h1>
            <p class="page-subtitle">Organize your images into folders</p>
        </div>

        <!-- Stats Bar -->
        <div class="stats-bar">
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['total_folders']; ?></div>
                <div class="stat-label">Total Folders</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['private_folders']; ?></div>
                <div class="stat-label">Private Folders</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['shared_folders']; ?></div>
                <div class="stat-label">Shared Folders</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['total_images_in_folders']; ?></div>
                <div class="stat-label">Images in Folders</div>
            </div>
        </div>

        <!-- View Toggle -->
        <div class="view-toggle">
            <a href="?view=my" class="view-btn <?php echo $view === 'my' ? 'active' : ''; ?>">
                🔒 My Folders
            </a>
            <a href="?view=shared" class="view-btn <?php echo $view === 'shared' ? 'active' : ''; ?>">
                🌐 Shared Gallery
            </a>
            <?php if ($canCreateFolders): ?>
                <a href="admin/folder-create.php" class="btn btn-primary" style="margin-left: auto;">
                    ➕ Create Folder
                </a>
            <?php endif; ?>
        </div>

        <!-- Folders Grid -->
        <?php if (empty($folders)): ?>
            <div class="empty-state">
                <div class="empty-icon">📁</div>
                <h2 class="empty-title">No folders yet</h2>
                <p class="empty-text">
                    <?php if ($view === 'shared'): ?>
                        No shared folders available at the moment
                    <?php else: ?>
                        Create your first folder to organize your images
                    <?php endif; ?>
                </p>
                <?php if ($view === 'my' && $canCreateFolders): ?>
                    <a href="admin/folder-create.php" class="btn btn-primary">
                        Create First Folder
                    </a>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="folders-grid">
                <?php foreach ($folders as $folder): ?>
                    <a href="folder-view.php?id=<?php echo urlencode($folder['id']); ?>" class="folder-card">
                        <div class="folder-thumbnail">
                            <?php if ($folder['thumbnail']): ?>
                                <img src="image.php?file=<?php echo urlencode($folder['thumbnail']); ?>&username=<?php echo urlencode($folder['owner']); ?>" 
                                     alt="<?php echo htmlspecialchars($folder['name']); ?>">
                            <?php else: ?>
                                <span class="folder-icon">📁</span>
                            <?php endif; ?>
                            
                            <?php if ($folder['owner'] === $username): ?>
                                <div class="folder-actions" onclick="event.preventDefault(); event.stopPropagation();">
                                    <button class="action-btn" onclick="window.location.href='admin/folder-edit.php?id=<?php echo urlencode($folder['id']); ?>'" title="Edit">
                                        ✏️
                                    </button>
                                    <button class="action-btn" onclick="window.location.href='admin/folder-delete.php?id=<?php echo urlencode($folder['id']); ?>'" title="Delete">
                                        🗑️
                                    </button>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="folder-content">
                            <div class="folder-header">
                                <div>
                                    <div class="folder-name"><?php echo htmlspecialchars($folder['name']); ?></div>
                                    <?php if ($folder['description']): ?>
                                        <div class="folder-description"><?php echo htmlspecialchars($folder['description']); ?></div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="folder-meta">
                                <span class="folder-count">
                                    🖼️ <?php echo $folder['image_count']; ?> image<?php echo $folder['image_count'] !== 1 ? 's' : ''; ?>
                                </span>
                                <span class="badge <?php echo $folder['visibility'] === 'private' ? 'badge-private' : 'badge-shared'; ?>">
                                    <?php echo $folder['visibility'] === 'private' ? '🔒 Private' : '🌐 Shared'; ?>
                                </span>
                            </div>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <?php if ($message): ?>
        <script>
            Swal.fire({
                icon: '<?php echo $message['type']; ?>',
                title: '<?php echo $message['type'] === 'success' ? 'Success!' : 'Info'; ?>',
                text: '<?php echo $message['text']; ?>',
                timer: 2000,
                showConfirmButton: false
            });
        </script>
    <?php endif; ?>
</body>
</html>
