<?php
require_once '../includes/config.php';
require_once '../includes/Auth.php';
require_once '../includes/FolderManager.php';
require_once '../includes/ImageManager.php';
require_once '../includes/PngHandler.php';
require_once '../includes/PermissionManager.php';

Auth::requireLogin();

$username = Auth::getUsername();
$isAdmin = Auth::isAdmin();
$folderManager = new FolderManager();
$imageManager = new ImageManager();
$pngHandler = new PngHandler($username);
$permManager = new PermissionManager();

$folderId = $_GET['id'] ?? '';

if (empty($folderId)) {
    header('Location: folders.php?error=invalid_folder');
    exit;
}

$folder = $folderManager->getFolder($folderId);

if (!$folder) {
    header('Location: folders.php?error=folder_not_found');
    exit;
}

// Check access
if (!$folderManager->canAccessFolder($folderId, $username, $isAdmin)) {
    die('You do not have permission to view this folder');
}

// Get images in folder
$images = $imageManager->getImagesByFolder($folderId);
$canUpload = $permManager->canUpload($username);
$canDelete = $permManager->canDelete($username);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($folder['name']); ?> - Galeriku ProMax</title>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        :root {
            --primary: #8B5CF6; --bg-dark: #0F172A; --bg-card: #1E293B;
            --bg-hover: #334155; --text-primary: #F1F5F9; --text-secondary: #94A3B8;
            --border: #334155;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, var(--bg-dark) 0%, #1a1a2e 100%);
            color: var(--text-primary); min-height: 100vh;
        }
        .header {
            background: var(--bg-card); border-bottom: 1px solid var(--border);
            padding: 1rem 2rem; position: sticky; top: 0; z-index: 100;
        }
        .header-content {
            max-width: 1400px; margin: 0 auto; display: flex;
            justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem;
        }
        .back-btn {
            padding: 0.5rem 1rem; background: var(--bg-hover);
            color: var(--text-primary); text-decoration: none;
            border-radius: 0.5rem; transition: all 0.3s;
        }
        .back-btn:hover { background: var(--primary); transform: translateY(-2px); }
        .container { max-width: 1400px; margin: 0 auto; padding: 2rem; }
        .folder-header {
            background: var(--bg-card); border: 1px solid var(--border);
            border-radius: 1rem; padding: 2rem; margin-bottom: 2rem;
        }
        .folder-title {
            font-size: 2rem; margin-bottom: 0.5rem;
            background: linear-gradient(135deg, var(--primary) 0%, #EC4899 100%);
            -webkit-background-clip: text; -webkit-text-fill-color: transparent;
        }
        .folder-meta {
            display: flex; gap: 2rem; color: var(--text-secondary);
            font-size: 0.875rem; margin-top: 1rem;
        }
        .badge {
            padding: 0.25rem 0.75rem; border-radius: 0.375rem;
            font-size: 0.75rem; font-weight: 600;
        }
        .badge-private { background: rgba(239, 68, 68, 0.2); color: #EF4444; }
        .badge-shared { background: rgba(16, 185, 129, 0.2); color: #10B981; }
        .btn {
            padding: 0.75rem 1.5rem; border: none; border-radius: 0.5rem;
            cursor: pointer; font-weight: 600; text-decoration: none;
            display: inline-flex; align-items: center; gap: 0.5rem;
            transition: all 0.3s;
        }
        .btn-primary { background: var(--primary); color: white; }
        .btn-primary:hover { background: #7C3AED; transform: translateY(-2px); }
        .images-grid {
            display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 1.5rem; margin-top: 2rem;
        }
        .image-card {
            background: var(--bg-card); border: 1px solid var(--border);
            border-radius: 1rem; overflow: hidden; transition: all 0.3s; cursor: pointer;
        }
        .image-card:hover {
            border-color: var(--primary); transform: translateY(-4px);
            box-shadow: 0 12px 24px rgba(139, 92, 246, 0.2);
        }
        .image-container {
            width: 100%; height: 200px; background: repeating-conic-gradient(#1E293B 0% 25%, #334155 0% 50%) 50% / 20px 20px;
            display: flex; align-items: center; justify-content: center; overflow: hidden;
        }
        .image-container img {
            max-width: 100%; max-height: 100%; object-fit: contain;
        }
        .image-info {
            padding: 1rem; display: flex; justify-content: space-between;
            align-items: center;
        }
        .image-name {
            font-size: 0.875rem; font-weight: 600;
            overflow: hidden; text-overflow: ellipsis; white-space: nowrap;
        }
        .image-actions {
            display: flex; gap: 0.5rem;
        }
        .action-btn {
            width: 32px; height: 32px; border-radius: 0.375rem;
            background: var(--bg-hover); border: 1px solid var(--border);
            color: var(--text-primary); cursor: pointer;
            display: flex; align-items: center; justify-content: center;
            transition: all 0.3s;
        }
        .action-btn:hover { background: var(--primary); border-color: var(--primary); }
        .empty-state {
            text-align: center; padding: 4rem 2rem;
        }
        .empty-icon { font-size: 5rem; margin-bottom: 1rem; opacity: 0.5; }
        .empty-title { font-size: 1.5rem; margin-bottom: 0.5rem; }
        .empty-text { color: var(--text-secondary); margin-bottom: 2rem; }
        .lightbox {
            display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0;
            background: rgba(0, 0, 0, 0.95); z-index: 9999;
            align-items: center; justify-content: center;
        }
        .lightbox.active { display: flex; }
        .lightbox img {
            max-width: 90%; max-height: 90%; object-fit: contain;
        }
        .lightbox-close {
            position: absolute; top: 2rem; right: 2rem;
            background: var(--primary); color: white;
            width: 48px; height: 48px; border-radius: 50%;
            display: flex; align-items: center; justify-content: center;
            cursor: pointer; font-size: 1.5rem; border: none;
            transition: all 0.3s;
        }
        .lightbox-close:hover { background: #7C3AED; transform: scale(1.1); }
        @media (max-width: 768px) {
            .container { padding: 1rem; }
            .images-grid { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <a href="folders.php" class="back-btn">← Back to Folders</a>
            <?php if ($canUpload && $folderManager->ownsFolder($folderId, $username)): ?>
                <a href="admin/upload.php?folder=<?php echo urlencode($folderId); ?>" class="btn btn-primary">
                    ➕ Upload to Folder
                </a>
            <?php endif; ?>
        </div>
    </div>

    <div class="container">
        <div class="folder-header">
            <h1 class="folder-title">📁 <?php echo htmlspecialchars($folder['name']); ?></h1>
            <?php if ($folder['description']): ?>
                <p style="color: var(--text-secondary); margin-top: 0.5rem;">
                    <?php echo htmlspecialchars($folder['description']); ?>
                </p>
            <?php endif; ?>
            <div class="folder-meta">
                <span>🖼️ <?php echo count($images); ?> image(s)</span>
                <span>👤 <?php echo htmlspecialchars($folder['owner']); ?></span>
                <span class="badge <?php echo $folder['visibility'] === 'private' ? 'badge-private' : 'badge-shared'; ?>">
                    <?php echo $folder['visibility'] === 'private' ? '🔒 Private' : '🌐 Shared'; ?>
                </span>
            </div>
        </div>

        <?php if (empty($images)): ?>
            <div class="empty-state">
                <div class="empty-icon">🖼️</div>
                <h2 class="empty-title">No images in this folder</h2>
                <p class="empty-text">Upload images to this folder to see them here</p>
                <?php if ($canUpload && $folderManager->ownsFolder($folderId, $username)): ?>
                    <a href="admin/upload.php?folder=<?php echo urlencode($folderId); ?>" class="btn btn-primary">
                        Upload First Image
                    </a>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="images-grid">
                <?php foreach ($images as $image): ?>
                    <div class="image-card">
                        <div class="image-container" onclick="openLightbox('<?php echo htmlspecialchars($image['filename']); ?>', '<?php echo htmlspecialchars($image['owner']); ?>')">
                            <img src="image.php?file=<?php echo urlencode($image['filename']); ?>&username=<?php echo urlencode($image['owner']); ?>" 
                                 alt="<?php echo htmlspecialchars($image['filename']); ?>" loading="lazy">
                        </div>
                        <div class="image-info">
                            <span class="image-name" title="<?php echo htmlspecialchars($image['filename']); ?>">
                                <?php echo htmlspecialchars($image['filename']); ?>
                            </span>
                            <div class="image-actions">
                                <button class="action-btn" onclick="downloadImage('<?php echo htmlspecialchars($image['filename']); ?>', '<?php echo htmlspecialchars($image['owner']); ?>')" title="Download">
                                    ⬇️
                                </button>
                                <?php if ($canDelete && ($image['owner'] === $username || $isAdmin)): ?>
                                    <button class="action-btn" onclick="deleteImage('<?php echo htmlspecialchars($image['filename']); ?>')" title="Delete">
                                        🗑️
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <div class="lightbox" id="lightbox" onclick="closeLightbox()">
        <button class="lightbox-close" onclick="closeLightbox()">✕</button>
        <img id="lightbox-img" src="" alt="">
    </div>

    <script>
        function openLightbox(filename, owner) {
            document.getElementById('lightbox').classList.add('active');
            document.getElementById('lightbox-img').src = 'image.php?file=' + encodeURIComponent(filename) + '&username=' + encodeURIComponent(owner);
        }
        
        function closeLightbox() {
            document.getElementById('lightbox').classList.remove('active');
        }
        
        function downloadImage(filename, owner) {
            window.location.href = 'image.php?file=' + encodeURIComponent(filename) + '&username=' + encodeURIComponent(owner) + '&download=1';
        }
        
        function deleteImage(filename) {
            Swal.fire({
                title: 'Delete Image?',
                text: 'This action cannot be undone',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#EF4444',
                confirmButtonText: 'Yes, delete it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = 'admin/delete.php?file=' + encodeURIComponent(filename) + '&folder=<?php echo urlencode($folderId); ?>';
                }
            });
        }
        
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') closeLightbox();
        });
    </script>
</body>
</html>
