<?php
require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/Auth.php';
require_once __DIR__ . '/../../includes/CSRF.php';
require_once __DIR__ . '/../../includes/UserManager.php';

Auth::requireAdmin();

$userManager = new UserManager();
$users = $userManager->getAllUsers();
$stats = $userManager->getUserStats();

$success = $_GET['success'] ?? '';
$error = $_GET['error'] ?? '';

// Handle POST actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    CSRF::verifyOrDie();
    
    switch ($_POST['action']) {
        case 'create':
            $result = $userManager->createUser(
                $_POST['username'],
                $_POST['password'],
                $_POST['display_name'],
                $_POST['email'],
                $_POST['role'] ?? 'user'
            );
            
            if (isset($result['success'])) {
                header('Location: users.php?success=user_created');
                exit;
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'edit':
            $result = $userManager->updateUser($_POST['edit_username'], [
                'display_name' => $_POST['display_name'],
                'email' => $_POST['email'],
                'bio' => $_POST['bio'],
                'role' => $_POST['role'],
                'status' => $_POST['status']
            ]);
            
            if (isset($result['success'])) {
                header('Location: users.php?success=user_updated');
                exit;
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'change_password':
            $result = $userManager->changePassword(
                $_POST['password_username'],
                null, // Admin doesn't need current password
                $_POST['new_password']
            );
            
            if (isset($result['success'])) {
                header('Location: users.php?success=password_changed');
                exit;
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'delete':
            $result = $userManager->deleteUser($_POST['delete_username']);
            
            if (isset($result['success'])) {
                header('Location: users.php?success=user_deleted');
                exit;
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'toggle_status':
            $targetUser = $_POST['toggle_username'];
            $currentUser = $userManager->getUser($targetUser);
            $newStatus = $currentUser['status'] === 'active' ? 'suspended' : 'active';
            
            $result = $userManager->updateUser($targetUser, ['status' => $newStatus]);
            
            if (isset($result['success'])) {
                $action = $newStatus === 'suspended' ? 'locked' : 'unlocked';
                header("Location: users.php?success=user_{$action}");
                exit;
            } else {
                $error = $result['error'];
            }
            break;
    }
}

// Get user for editing if edit parameter exists
$editUser = null;
if (isset($_GET['edit'])) {
    $editUser = $userManager->getUser($_GET['edit']);
}
?>
<!DOCTYPE html>
<html lang="en" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users - Galeriku ProMax</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --bg-primary: #FFFFFF;
            --bg-secondary: #F9FAFB;
            --bg-card: #FFFFFF;
            --text-primary: #111827;
            --text-secondary: #6B7280;
            --border: #E5E7EB;
            --success: #10B981;
            --warning: #F59E0B;
            --error: #EF4444;
            --shadow: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0,0,0,0.1);
        }
        
        [data-theme="dark"] {
            --primary: #A78BFA;
            --bg-primary: #111827;
            --bg-secondary: #1F2937;
            --bg-card: #374151;
            --text-primary: #F9FAFB;
            --text-secondary: #D1D5DB;
            --border: #4B5563;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-secondary);
            color: var(--text-primary);
            min-height: 100vh;
        }
        
        .header {
            background: var(--bg-primary);
            border-bottom: 1px solid var(--border);
            padding: 1rem 1.5rem;
            box-shadow: var(--shadow);
        }
        
        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
            text-decoration: none;
        }
        
        .logo svg { width: 32px; height: 32px; }
        
        .btn {
            padding: 0.625rem 1rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            border: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.2s;
            text-decoration: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(139, 92, 246, 0.4);
        }
        
        .btn-secondary {
            background: var(--bg-card);
            color: var(--text-primary);
            border: 1px solid var(--border);
        }
        
        .btn-warning {
            background: var(--warning);
            color: white;
        }
        
        .btn-danger {
            background: var(--error);
            color: white;
        }
        
        .btn-success {
            background: var(--success);
            color: white;
        }
        
        .btn-sm {
            padding: 0.375rem 0.75rem;
            font-size: 0.8125rem;
        }
        
        .btn svg { width: 18px; height: 18px; }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem 1.5rem;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 700;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: var(--bg-card);
            padding: 1.5rem;
            border-radius: 12px;
            border: 1px solid var(--border);
        }
        
        .stat-label {
            font-size: 0.75rem;
            color: var(--text-secondary);
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            margin-bottom: 0.5rem;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--primary);
        }
        
        .users-table {
            background: var(--bg-card);
            border-radius: 12px;
            border: 1px solid var(--border);
            overflow: hidden;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        thead {
            background: var(--bg-secondary);
        }
        
        th {
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-secondary);
            border-bottom: 1px solid var(--border);
        }
        
        td {
            padding: 1rem;
            border-bottom: 1px solid var(--border);
        }
        
        tbody tr:last-child td {
            border-bottom: none;
        }
        
        tbody tr:hover {
            background: var(--bg-secondary);
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .user-details {
            flex: 1;
        }
        
        .user-name {
            font-weight: 600;
            margin-bottom: 0.125rem;
        }
        
        .user-email {
            font-size: 0.875rem;
            color: var(--text-secondary);
        }
        
        .badge {
            padding: 0.25rem 0.75rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
            display: inline-block;
        }
        
        .badge-admin {
            background: rgba(139, 92, 246, 0.1);
            color: var(--primary);
        }
        
        .badge-user {
            background: var(--bg-secondary);
            color: var(--text-secondary);
        }
        
        .badge-active {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }
        
        .badge-suspended {
            background: rgba(239, 68, 68, 0.1);
            color: var(--error);
        }
        
        .badge-inactive {
            background: rgba(107, 114, 128, 0.1);
            color: var(--text-secondary);
        }
        
        .action-buttons {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 1000;
            backdrop-filter: blur(4px);
        }
        
        .modal.active {
            display: flex;
        }
        
        .modal-content {
            background: var(--bg-card);
            border-radius: 16px;
            max-width: 500px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }
        
        .modal-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-title {
            font-size: 1.25rem;
            font-weight: 700;
        }
        
        .modal-close {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            border: none;
            background: var(--bg-secondary);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: var(--text-secondary);
        }
        
        .modal-body {
            padding: 1.5rem;
        }
        
        .form-group {
            margin-bottom: 1.25rem;
        }
        
        label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            font-size: 0.875rem;
        }
        
        input, select, textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-size: 1rem;
            transition: all 0.2s;
            font-family: inherit;
        }
        
        input:focus, select:focus, textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(139, 92, 246, 0.1);
        }
        
        textarea {
            resize: vertical;
            min-height: 80px;
        }
        
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
            border: 1px solid var(--success);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--error);
            border: 1px solid var(--error);
        }
        
        @media (max-width: 768px) {
            .stats-grid { grid-template-columns: 1fr; }
            .users-table { overflow-x: auto; }
            table { min-width: 800px; }
            .action-buttons { flex-direction: column; }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <a href="../index.php" class="logo">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                    <circle cx="9" cy="7" r="4"/>
                    <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                    <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                </svg>
                Manage Users
            </a>
            <div style="display: flex; gap: 0.5rem;">
                <button class="btn btn-primary" onclick="openModal('addUserModal')">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                        <circle cx="8.5" cy="7" r="4"/>
                        <line x1="20" y1="8" x2="20" y2="14"/>
                        <line x1="23" y1="11" x2="17" y2="11"/>
                    </svg>
                    Add User
                </button>
                <a href="../index.php" class="btn btn-secondary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="19" y1="12" x2="5" y2="12"/>
                        <polyline points="12 19 5 12 12 5"/>
                    </svg>
                    Back
                </a>
            </div>
        </div>
    </div>
    
    <div class="container">
        <?php if ($success): ?>
        <div class="alert alert-success">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 20px; height: 20px;">
                <polyline points="20 6 9 17 4 12"/>
            </svg>
            <span>
                <?php
                $messages = [
                    'user_created' => 'User created successfully!',
                    'user_updated' => 'User updated successfully!',
                    'user_deleted' => 'User deleted successfully!',
                    'user_locked' => 'User locked successfully!',
                    'user_unlocked' => 'User unlocked successfully!',
                    'password_changed' => 'Password changed successfully!'
                ];
                echo $messages[$success] ?? 'Operation completed successfully!';
                ?>
            </span>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="alert alert-error">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 20px; height: 20px;">
                <circle cx="12" cy="12" r="10"/>
                <line x1="12" y1="8" x2="12" y2="12"/>
                <line x1="12" y1="16" x2="12.01" y2="16"/>
            </svg>
            <span><?php echo htmlspecialchars($error); ?></span>
        </div>
        <?php endif; ?>
        
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-label">Total Users</div>
                <div class="stat-value"><?php echo $stats['total_users']; ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Active Users</div>
                <div class="stat-value"><?php echo $stats['active_users']; ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Total Uploads</div>
                <div class="stat-value"><?php echo $stats['total_uploads']; ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Storage Used</div>
                <div class="stat-value"><?php echo formatBytes($stats['total_storage']); ?></div>
            </div>
        </div>
        
        <div class="users-table">
            <table>
                <thead>
                    <tr>
                        <th>User</th>
                        <th>Role</th>
                        <th>Status</th>
                        <th>Uploads</th>
                        <th>Storage</th>
                        <th>Last Login</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                    <tr>
                        <td>
                            <div class="user-info">
                                <img src="../user-avatar.php?username=<?php echo urlencode($user['username']); ?>" alt="Avatar" class="user-avatar">
                                <div class="user-details">
                                    <div class="user-name"><?php echo htmlspecialchars($user['display_name']); ?></div>
                                    <div class="user-email"><?php echo htmlspecialchars($user['email'] ?: $user['username']); ?></div>
                                </div>
                            </div>
                        </td>
                        <td>
                            <span class="badge <?php echo $user['role'] === 'admin' ? 'badge-admin' : 'badge-user'; ?>">
                                <?php echo ucfirst($user['role']); ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-<?php echo $user['status']; ?>">
                                <?php echo ucfirst($user['status']); ?>
                            </span>
                        </td>
                        <td><?php echo $user['total_uploads'] ?? 0; ?></td>
                        <td><?php echo formatBytes($user['storage_used'] ?? 0); ?></td>
                        <td><?php echo $user['last_login'] ? timeAgo($user['last_login']) : 'Never'; ?></td>
                        <td>
                            <div class="action-buttons">
                                <button class="btn btn-primary btn-sm" onclick='editUser(<?php echo json_encode($user); ?>)' title="Edit User">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
                                        <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
                                    </svg>
                                    Edit
                                </button>
                                
                                <button class="btn btn-warning btn-sm" onclick="changePassword('<?php echo htmlspecialchars($user['username']); ?>')" title="Change Password">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                                        <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                                    </svg>
                                    Password
                                </button>
                                
                                <?php if ($user['username'] !== 'admin' && $user['username'] !== Auth::getUsername()): ?>
                                <button class="btn <?php echo $user['status'] === 'active' ? 'btn-warning' : 'btn-success'; ?> btn-sm" onclick="toggleStatus('<?php echo htmlspecialchars($user['username']); ?>', '<?php echo $user['status']; ?>')" title="<?php echo $user['status'] === 'active' ? 'Lock' : 'Unlock'; ?> User">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <?php if ($user['status'] === 'active'): ?>
                                        <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                                        <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                                        <?php else: ?>
                                        <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                                        <path d="M7 11V7a5 5 0 0 1 9.9-1"/>
                                        <?php endif; ?>
                                    </svg>
                                    <?php echo $user['status'] === 'active' ? 'Lock' : 'Unlock'; ?>
                                </button>
                                
                                <button class="btn btn-danger btn-sm" onclick="deleteUser('<?php echo htmlspecialchars($user['username']); ?>', '<?php echo htmlspecialchars($user['display_name']); ?>')" title="Delete User">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <polyline points="3 6 5 6 21 6"/>
                                        <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
                                    </svg>
                                    Delete
                                </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Add User Modal -->
    <div class="modal" id="addUserModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Add New User</h3>
                <button class="modal-close" onclick="closeModal('addUserModal')">×</button>
            </div>
            <div class="modal-body">
                <form method="POST">
                    <?php echo CSRF::field(); ?>
                    <input type="hidden" name="action" value="create">
                    
                    <div class="form-group">
                        <label>Username *</label>
                        <input type="text" name="username" required pattern="[a-zA-Z0-9_-]{3,20}" title="3-20 characters, letters, numbers, underscore, dash only">
                    </div>
                    
                    <div class="form-group">
                        <label>Password *</label>
                        <input type="password" name="password" required minlength="6">
                    </div>
                    
                    <div class="form-group">
                        <label>Display Name</label>
                        <input type="text" name="display_name">
                    </div>
                    
                    <div class="form-group">
                        <label>Email *</label>
                        <input type="email" name="email" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Role</label>
                        <select name="role">
                            <option value="user">User</option>
                            <option value="admin">Admin</option>
                        </select>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; justify-content: center;">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="20 6 9 17 4 12"/>
                        </svg>
                        Create User
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit User Modal -->
    <div class="modal" id="editUserModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Edit User</h3>
                <button class="modal-close" onclick="closeModal('editUserModal')">×</button>
            </div>
            <div class="modal-body">
                <form method="POST" id="editUserForm">
                    <?php echo CSRF::field(); ?>
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="edit_username" id="edit_username">
                    
                    <div class="form-group">
                        <label>Username (cannot be changed)</label>
                        <input type="text" id="edit_username_display" disabled>
                    </div>
                    
                    <div class="form-group">
                        <label>Display Name</label>
                        <input type="text" name="display_name" id="edit_display_name" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Email</label>
                        <input type="email" name="email" id="edit_email" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Bio</label>
                        <textarea name="bio" id="edit_bio"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>Role</label>
                        <select name="role" id="edit_role">
                            <option value="user">User</option>
                            <option value="admin">Admin</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Status</label>
                        <select name="status" id="edit_status">
                            <option value="active">Active</option>
                            <option value="suspended">Suspended</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; justify-content: center;">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="20 6 9 17 4 12"/>
                        </svg>
                        Update User
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Change Password Modal -->
    <div class="modal" id="passwordModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Change Password</h3>
                <button class="modal-close" onclick="closeModal('passwordModal')">×</button>
            </div>
            <div class="modal-body">
                <form method="POST" id="passwordForm">
                    <?php echo CSRF::field(); ?>
                    <input type="hidden" name="action" value="change_password">
                    <input type="hidden" name="password_username" id="password_username">
                    
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" id="password_username_display" disabled>
                    </div>
                    
                    <div class="form-group">
                        <label>New Password</label>
                        <input type="password" name="new_password" required minlength="6">
                    </div>
                    
                    <div class="form-group">
                        <label>Confirm Password</label>
                        <input type="password" name="confirm_password" required minlength="6">
                    </div>
                    
                    <button type="submit" class="btn btn-warning" style="width: 100%; justify-content: center;">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                            <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                        </svg>
                        Change Password
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        function openModal(id) {
            document.getElementById(id).classList.add('active');
        }
        
        function closeModal(id) {
            document.getElementById(id).classList.remove('active');
        }
        
        // Close modal on outside click
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    closeModal(modal.id);
                }
            });
        });
        
        function editUser(user) {
            document.getElementById('edit_username').value = user.username;
            document.getElementById('edit_username_display').value = user.username;
            document.getElementById('edit_display_name').value = user.display_name;
            document.getElementById('edit_email').value = user.email || '';
            document.getElementById('edit_bio').value = user.bio || '';
            document.getElementById('edit_role').value = user.role;
            document.getElementById('edit_status').value = user.status;
            openModal('editUserModal');
        }
        
        function changePassword(username) {
            document.getElementById('password_username').value = username;
            document.getElementById('password_username_display').value = username;
            document.getElementById('passwordForm').reset();
            document.getElementById('password_username').value = username;
            openModal('passwordModal');
        }
        
        function toggleStatus(username, currentStatus) {
            const action = currentStatus === 'active' ? 'lock' : 'unlock';
            const actionText = currentStatus === 'active' ? 'Lock' : 'Unlock';
            
            Swal.fire({
                title: `${actionText} User?`,
                text: `Are you sure you want to ${action} ${username}?`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: action === 'lock' ? '#F59E0B' : '#10B981',
                confirmButtonText: `Yes, ${action} user!`
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="toggle_status">
                        <input type="hidden" name="toggle_username" value="${username}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }
        
        function deleteUser(username, displayName) {
            Swal.fire({
                title: 'Delete User?',
                html: `Are you sure you want to delete <strong>${displayName}</strong>?<br><small>This action cannot be undone!</small>`,
                icon: 'error',
                showCancelButton: true,
                confirmButtonColor: '#EF4444',
                confirmButtonText: 'Yes, delete user!'
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="delete">
                        <input type="hidden" name="delete_username" value="${username}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }
        
        // Validate password match
        document.getElementById('passwordForm').addEventListener('submit', function(e) {
            const pass = this.querySelector('[name="new_password"]').value;
            const confirm = this.querySelector('[name="confirm_password"]').value;
            
            if (pass !== confirm) {
                e.preventDefault();
                Swal.fire({
                    icon: 'error',
                    title: 'Password Mismatch',
                    text: 'Passwords do not match!',
                    confirmButtonColor: '#8B5CF6'
                });
            }
        });
    </script>
</body>
</html>
