<?php
require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/Auth.php';
require_once __DIR__ . '/../../includes/CSRF.php';
require_once __DIR__ . '/../../includes/PngHandler.php';
require_once __DIR__ . '/../../includes/UserManager.php';
require_once __DIR__ . '/../../includes/FolderManager.php';
require_once __DIR__ . '/../../includes/ImageManager.php';
require_once __DIR__ . '/../../includes/PermissionManager.php';

Auth::requireLogin();

$username = Auth::getUsername();
$userId = Auth::getUserId();
$folderManager = new FolderManager();
$imageManager = new ImageManager();
$permManager = new PermissionManager();

// Get folder from URL or form
$preselectedFolder = $_GET['folder'] ?? $_POST['folder_id'] ?? '';

// Get user's folders for dropdown
$userFolders = $folderManager->getUserFolders($username);

// Check permissions
$canShare = $permManager->canShare($username);

$success = false;
$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    CSRF::verifyOrDie();
    
    $handler = new PngHandler($userId, $username);
    $result = $handler->process($_FILES['file']);
    
    if ($result && isset($result['success'])) {
        // Add image metadata
        $folderId = $_POST['folder_id'] ?? null;
        $visibility = ($canShare && isset($_POST['visibility'])) ? $_POST['visibility'] : 'private';
        $filename = $result['filename'] ?? '';
        $filesize = $result['size'] ?? 0;
        
        if ($folderId === 'root' || empty($folderId)) {
            $folderId = null;
        }
        
        $imageManager->addImage($filename, $username, $folderId, $visibility, $filesize);
        
        $success = true;
    } else {
        $errors = $handler->getErrors();
    }
}
?>
<!DOCTYPE html>
<html lang="en" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload - Galeriku ProMax</title>
    <meta name="robots" content="noindex">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --bg-primary: #FFFFFF;
            --bg-secondary: #F9FAFB;
            --bg-card: #FFFFFF;
            --text-primary: #111827;
            --text-secondary: #6B7280;
            --border: #E5E7EB;
            --success: #10B981;
            --error: #EF4444;
            --shadow: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0,0,0,0.1);
        }
        
        [data-theme="dark"] {
            --primary: #A78BFA;
            --bg-primary: #111827;
            --bg-secondary: #1F2937;
            --bg-card: #374151;
            --text-primary: #F9FAFB;
            --text-secondary: #D1D5DB;
            --border: #4B5563;
        }
        
        .upload-options {
            background: var(--bg-card); border: 1px solid var(--border);
            border-radius: 16px; padding: 1.5rem; margin-bottom: 1.5rem;
            box-shadow: var(--shadow);
        }
        
        .form-group { margin-bottom: 1.5rem; }
        .form-group:last-child { margin-bottom: 0; }
        
        .form-group label {
            display: block; font-weight: 600; margin-bottom: 0.5rem;
            color: var(--text-primary);
        }
        
        .form-select {
            width: 100%; padding: 0.75rem; background: var(--bg-secondary);
            border: 1px solid var(--border); border-radius: 8px;
            color: var(--text-primary); font-size: 0.875rem;
            transition: all 0.3s;
        }
        
        .form-select:focus {
            outline: none; border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(139, 92, 246, 0.1);
        }
        
        .radio-group {
            display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;
        }
        
        .radio-option {
            position: relative; display: flex; align-items: center;
            padding: 1rem; background: var(--bg-secondary);
            border: 2px solid var(--border); border-radius: 8px;
            cursor: pointer; transition: all 0.3s;
        }
        
        .radio-option:hover { border-color: var(--primary); }
        
        .radio-option input[type="radio"] {
            position: absolute; opacity: 0;
        }
        
        .radio-option:has(input[type="radio"]:checked) {
            border-color: var(--primary); background: rgba(139, 92, 246, 0.05);
        }
        
        .radio-label { flex: 1; display: flex; flex-direction: column; gap: 0.25rem; }
        .radio-label strong { display: block; }
        .radio-label small { color: var(--text-secondary); font-size: 0.75rem; }
        
        @media (max-width: 640px) {
            .radio-group { grid-template-columns: 1fr; }
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-secondary);
            color: var(--text-primary);
            min-height: 100vh;
            transition: all 0.3s;
        }
        
        .header {
            background: var(--bg-primary);
            border-bottom: 1px solid var(--border);
            padding: 1rem 1.5rem;
            box-shadow: var(--shadow);
        }
        
        .header-content {
            max-width: 1000px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
            text-decoration: none;
        }
        
        .logo svg { width: 32px; height: 32px; }
        
        .btn {
            padding: 0.625rem 1rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            border: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.2s;
            text-decoration: none;
        }
        
        .btn-secondary {
            background: var(--bg-card);
            color: var(--text-primary);
            border: 1px solid var(--border);
        }
        
        .btn-secondary:hover { background: var(--bg-secondary); }
        .btn svg { width: 18px; height: 18px; }
        
        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 2rem 1.5rem;
        }
        
        .page-header {
            margin-bottom: 2rem;
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }
        
        .page-subtitle {
            color: var(--text-secondary);
        }
        
        .upload-card {
            background: var(--bg-card);
            border-radius: 16px;
            border: 2px dashed var(--border);
            padding: 3rem;
            text-align: center;
            margin-bottom: 2rem;
            transition: all 0.3s;
        }
        
        .upload-card.drag-over {
            border-color: var(--primary);
            background: rgba(139, 92, 246, 0.05);
            transform: scale(1.02);
        }
        
        .upload-icon {
            width: 80px;
            height: 80px;
            margin: 0 auto 1.5rem;
            color: var(--primary);
            opacity: 0.5;
        }
        
        .upload-title {
            font-size: 1.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }
        
        .upload-subtitle {
            color: var(--text-secondary);
            margin-bottom: 1.5rem;
        }
        
        .upload-input {
            display: none;
        }
        
        .btn-upload {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: white;
            padding: 0.875rem 2rem;
            font-size: 1rem;
        }
        
        .btn-upload:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 16px rgba(139, 92, 246, 0.4);
        }
        
        .upload-info {
            margin-top: 1.5rem;
            padding: 1rem;
            background: var(--bg-secondary);
            border-radius: 8px;
            font-size: 0.875rem;
            color: var(--text-secondary);
        }
        
        .preview-section {
            background: var(--bg-card);
            border-radius: 16px;
            border: 1px solid var(--border);
            padding: 2rem;
            margin-bottom: 2rem;
            display: none;
        }
        
        .preview-section.active {
            display: block;
        }
        
        .preview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }
        
        .preview-title {
            font-size: 1.25rem;
            font-weight: 700;
        }
        
        .preview-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 1rem;
        }
        
        .preview-item {
            background: var(--bg-secondary);
            border-radius: 12px;
            padding: 1rem;
            border: 1px solid var(--border);
        }
        
        .preview-image {
            aspect-ratio: 1;
            background: repeating-conic-gradient(#f0f0f0 0% 25%, transparent 0% 50%) 50% / 20px 20px;
            border-radius: 8px;
            margin-bottom: 0.75rem;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
        }
        
        .preview-image img {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
        }
        
        .preview-name {
            font-size: 0.875rem;
            font-weight: 600;
            margin-bottom: 0.25rem;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        .preview-size {
            font-size: 0.75rem;
            color: var(--text-secondary);
        }
        
        .progress-bar {
            width: 100%;
            height: 8px;
            background: var(--bg-secondary);
            border-radius: 4px;
            margin-top: 0.5rem;
            overflow: hidden;
            display: none;
        }
        
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, var(--primary) 0%, var(--primary-dark) 100%);
            border-radius: 4px;
            transition: width 0.3s;
            width: 0%;
        }
        
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
            border: 1px solid var(--success);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--error);
            border: 1px solid var(--error);
        }
        
        .alert svg {
            width: 20px;
            height: 20px;
            flex-shrink: 0;
        }
        
        @media (max-width: 768px) {
            .upload-card { padding: 2rem 1rem; }
            .preview-grid { grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <a href="../index.php" class="logo">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                    <polyline points="3.27 6.96 12 12.01 20.73 6.96"/>
                    <line x1="12" y1="22.08" x2="12" y2="12"/>
                </svg>
                Upload Image
            </a>
            <a href="../index.php" class="btn btn-secondary">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <line x1="19" y1="12" x2="5" y2="12"/>
                    <polyline points="12 19 5 12 12 5"/>
                </svg>
                Back to Gallery
            </a>
        </div>
    </div>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Upload PNG Image</h1>
            <p class="page-subtitle">Drag & drop your PNG file or click to browse</p>
        </div>
        
        <?php if ($success): ?>
        <div class="alert alert-success">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <polyline points="20 6 9 17 4 12"/>
            </svg>
            <span>Image uploaded successfully!</span>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($errors)): ?>
        <div class="alert alert-error">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <circle cx="12" cy="12" r="10"/>
                <line x1="12" y1="8" x2="12" y2="12"/>
                <line x1="12" y1="16" x2="12.01" y2="16"/>
            </svg>
            <div>
                <?php foreach ($errors as $error): ?>
                    <div><?php echo htmlspecialchars($error); ?></div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <form method="POST" enctype="multipart/form-data" id="uploadForm">
            <?php echo CSRF::field(); ?>
            
            <!-- Folder and Visibility Options -->
            <div class="upload-options">
                <div class="form-group">
                    <label for="folder_id">📁 Upload to Folder (Optional)</label>
                    <select name="folder_id" id="folder_id" class="form-select">
                        <option value="root">📷 No Folder (Gallery Root)</option>
                        <?php foreach ($userFolders as $folder): ?>
                            <option value="<?php echo htmlspecialchars($folder['id']); ?>" 
                                    <?php echo $preselectedFolder === $folder['id'] ? 'selected' : ''; ?>>
                                <?php echo $folder['visibility'] === 'private' ? '🔒' : '🌐'; ?> 
                                <?php echo htmlspecialchars($folder['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <?php if ($canShare): ?>
                    <div class="form-group">
                        <label>🌐 Visibility</label>
                        <div class="radio-group">
                            <label class="radio-option">
                                <input type="radio" name="visibility" value="private" checked>
                                <span class="radio-label">
                                    <strong>🔒 Private</strong>
                                    <small>Only you can see</small>
                                </span>
                            </label>
                            <label class="radio-option">
                                <input type="radio" name="visibility" value="shared">
                                <span class="radio-label">
                                    <strong>🌐 Shared</strong>
                                    <small>Everyone can see</small>
                                </span>
                            </label>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="upload-card" id="dropZone">
                <svg class="upload-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                    <polyline points="17 8 12 3 7 8"/>
                    <line x1="12" y1="3" x2="12" y2="15"/>
                </svg>
                
                <h2 class="upload-title">Drop your PNG here</h2>
                <p class="upload-subtitle">or click the button below to browse</p>
                
                <input type="file" name="file" id="fileInput" class="upload-input" accept="image/png">
                <label for="fileInput" class="btn btn-upload">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="17 8 12 3 7 8"/>
                        <line x1="12" y1="3" x2="12" y2="15"/>
                    </svg>
                    Choose File
                </label>
                
                <div class="upload-info">
                    <strong>Supported format:</strong> PNG only<br>
                    <strong>Max file size:</strong> <?php echo formatBytes(MAX_FILE_SIZE); ?><br>
                    <strong>Transparency:</strong> Preserved automatically
                </div>
            </div>
        </form>
        
        <div class="preview-section" id="previewSection">
            <div class="preview-header">
                <h3 class="preview-title">Selected Image</h3>
                <button class="btn btn-secondary" onclick="clearPreview()">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"/>
                        <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                    Clear
                </button>
            </div>
            
            <div class="preview-grid" id="previewGrid"></div>
            
            <button type="button" class="btn btn-upload" onclick="submitForm()" style="margin-top: 1.5rem; width: 100%; justify-content: center;">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <polyline points="20 6 9 17 4 12"/>
                </svg>
                Upload Image
            </button>
            
            <div class="progress-bar" id="progressBar">
                <div class="progress-fill" id="progressFill"></div>
            </div>
        </div>
    </div>
    
    <script>
        const dropZone = document.getElementById('dropZone');
        const fileInput = document.getElementById('fileInput');
        const previewSection = document.getElementById('previewSection');
        const previewGrid = document.getElementById('previewGrid');
        const uploadForm = document.getElementById('uploadForm');
        const progressBar = document.getElementById('progressBar');
        const progressFill = document.getElementById('progressFill');
        
        // Prevent default drag behaviors
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, preventDefaults, false);
            document.body.addEventListener(eventName, preventDefaults, false);
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        // Highlight drop zone
        ['dragenter', 'dragover'].forEach(eventName => {
            dropZone.addEventListener(eventName, () => {
                dropZone.classList.add('drag-over');
            });
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, () => {
                dropZone.classList.remove('drag-over');
            });
        });
        
        // Handle dropped files
        dropZone.addEventListener('drop', handleDrop);
        fileInput.addEventListener('change', handleFiles);
        
        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;
            handleFiles({ target: { files } });
        }
        
        function handleFiles(e) {
            const files = e.target.files;
            if (files.length === 0) return;
            
            const file = files[0];
            
            // Validate file type
            if (!file.type.match('image/png')) {
                alert('Please upload PNG files only!');
                return;
            }
            
            // Validate file size
            if (file.size > <?php echo MAX_FILE_SIZE; ?>) {
                alert('File size exceeds <?php echo formatBytes(MAX_FILE_SIZE); ?>!');
                return;
            }
            
            // Show preview
            showPreview(file);
        }
        
        function showPreview(file) {
            const reader = new FileReader();
            reader.onload = (e) => {
                previewGrid.innerHTML = `
                    <div class="preview-item">
                        <div class="preview-image">
                            <img src="${e.target.result}" alt="Preview">
                        </div>
                        <div class="preview-name" title="${file.name}">${file.name}</div>
                        <div class="preview-size">${formatBytes(file.size)}</div>
                    </div>
                `;
                previewSection.classList.add('active');
            };
            reader.readAsDataURL(file);
        }
        
        function clearPreview() {
            fileInput.value = '';
            previewGrid.innerHTML = '';
            previewSection.classList.remove('active');
            progressBar.style.display = 'none';
            progressFill.style.width = '0%';
        }
        
        function submitForm() {
            if (fileInput.files.length === 0) {
                alert('Please select a file first!');
                return;
            }
            
            // Show progress bar
            progressBar.style.display = 'block';
            
            // Simulate progress (in real app, use XMLHttpRequest for real progress)
            let progress = 0;
            const interval = setInterval(() => {
                progress += 10;
                progressFill.style.width = progress + '%';
                
                if (progress >= 90) {
                    clearInterval(interval);
                }
            }, 100);
            
            // Submit form
            setTimeout(() => {
                uploadForm.submit();
            }, 1000);
        }
        
        function formatBytes(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
        }
    </script>
</body>
</html>
