<?php
/**
 * Permissions Management
 * Admin can control user permissions
 */

require_once '../../includes/config.php';
require_once '../../includes/Auth.php';
require_once '../../includes/CSRF.php';
require_once '../../includes/UserManager.php';
require_once '../../includes/PermissionManager.php';

// Require admin
Auth::requireAdmin();

$userManager = new UserManager();
$permManager = new PermissionManager();
$username = Auth::getUsername();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && CSRF::validate()) {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'update_permissions') {
        $targetUser = $_POST['username'] ?? '';
        
        if ($targetUser && $targetUser !== 'admin') {
            $permissions = [
                'can_upload' => isset($_POST['can_upload']),
                'can_create_folders' => isset($_POST['can_create_folders']),
                'can_delete' => isset($_POST['can_delete']),
                'can_share' => isset($_POST['can_share']),
                'can_download_original' => isset($_POST['can_download_original']),
                'storage_quota' => (int)($_POST['storage_quota'] ?? 1073741824),
                'max_folders' => (int)($_POST['max_folders'] ?? 50),
                'max_images_per_folder' => (int)($_POST['max_images_per_folder'] ?? 1000)
            ];
            
            $result = $permManager->setUserPermissions($targetUser, $permissions);
            
            if (isset($result['success'])) {
                $userManager->logActivity($username, 'permissions_updated', "Updated permissions for user: {$targetUser}");
                header('Location: permissions.php?success=updated');
                exit;
            }
        }
    }
    
    if ($action === 'apply_preset') {
        $targetUser = $_POST['username'] ?? '';
        $preset = $_POST['preset'] ?? '';
        
        if ($targetUser && $targetUser !== 'admin') {
            $result = $permManager->applyPreset($targetUser, $preset);
            
            if (isset($result['success'])) {
                $userManager->logActivity($username, 'preset_applied', "Applied {$preset} preset to user: {$targetUser}");
                header('Location: permissions.php?success=preset_applied');
                exit;
            }
        }
    }
    
    if ($action === 'reset_permissions') {
        $targetUser = $_POST['username'] ?? '';
        
        if ($targetUser && $targetUser !== 'admin') {
            $result = $permManager->resetToDefault($targetUser);
            
            if (isset($result['success'])) {
                $userManager->logActivity($username, 'permissions_reset', "Reset permissions for user: {$targetUser}");
                header('Location: permissions.php?success=reset');
                exit;
            }
        }
    }
}

// Get all users and their permissions
$users = $userManager->getAllUsers();
$allPermissions = $permManager->getAllPermissions();
$presets = $permManager->getPresets();

// Success messages
$successMessages = [
    'updated' => 'Permissions updated successfully!',
    'preset_applied' => 'Permission preset applied successfully!',
    'reset' => 'Permissions reset to default successfully!'
];

$successMessage = isset($_GET['success']) ? ($successMessages[$_GET['success']] ?? '') : '';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Permissions - Galeriku ProMax</title>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --success: #10B981;
            --warning: #F59E0B;
            --danger: #EF4444;
            --bg-dark: #0F172A;
            --bg-card: #1E293B;
            --bg-hover: #334155;
            --text-primary: #F1F5F9;
            --text-secondary: #94A3B8;
            --border: #334155;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, var(--bg-dark) 0%, #1a1a2e 100%);
            color: var(--text-primary);
            min-height: 100vh;
            padding: 2rem;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
        }

        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            flex-wrap: wrap;
            gap: 1rem;
        }

        .header h1 {
            font-size: 2rem;
            background: linear-gradient(135deg, var(--primary) 0%, #EC4899 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 0.5rem;
            cursor: pointer;
            font-size: 0.875rem;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s;
        }

        .btn-back {
            background: var(--bg-card);
            color: var(--text-primary);
        }

        .btn-back:hover {
            background: var(--bg-hover);
            transform: translateY(-2px);
        }

        .permissions-grid {
            display: grid;
            gap: 1.5rem;
        }

        .user-card {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 1rem;
            padding: 1.5rem;
            transition: all 0.3s;
        }

        .user-card:hover {
            border-color: var(--primary);
            box-shadow: 0 8px 24px rgba(139, 92, 246, 0.15);
        }

        .user-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            flex-wrap: wrap;
            gap: 1rem;
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .user-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary) 0%, #EC4899 100%);
        }

        .user-details h3 {
            font-size: 1.125rem;
            margin-bottom: 0.25rem;
        }

        .user-details p {
            color: var(--text-secondary);
            font-size: 0.875rem;
        }

        .badge-admin {
            background: var(--primary);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 0.375rem;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .permissions-content {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 2rem;
        }

        .permissions-section {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }

        .section-title {
            font-size: 1rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        .permission-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            background: rgba(139, 92, 246, 0.1);
            border-radius: 0.5rem;
            border: 1px solid transparent;
            transition: all 0.3s;
        }

        .permission-item:hover {
            border-color: var(--primary);
            background: rgba(139, 92, 246, 0.15);
        }

        .permission-checkbox {
            width: 20px;
            height: 20px;
            cursor: pointer;
            accent-color: var(--primary);
        }

        .permission-label {
            font-size: 0.875rem;
            cursor: pointer;
            flex: 1;
        }

        .permission-status {
            font-size: 0.75rem;
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
            font-weight: 600;
        }

        .status-enabled {
            background: rgba(16, 185, 129, 0.2);
            color: var(--success);
        }

        .status-disabled {
            background: rgba(239, 68, 68, 0.2);
            color: var(--danger);
        }

        .quota-input {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .quota-input label {
            font-size: 0.875rem;
            color: var(--text-secondary);
        }

        .quota-input input,
        .quota-input select {
            padding: 0.75rem;
            background: rgba(139, 92, 246, 0.1);
            border: 1px solid var(--border);
            border-radius: 0.5rem;
            color: var(--text-primary);
            font-size: 0.875rem;
        }

        .quota-input input:focus,
        .quota-input select:focus {
            outline: none;
            border-color: var(--primary);
            background: rgba(139, 92, 246, 0.15);
        }

        .preset-section {
            grid-column: 1 / -1;
            padding-top: 1rem;
            border-top: 1px solid var(--border);
        }

        .preset-buttons {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }

        .btn-preset {
            padding: 0.5rem 1rem;
            background: rgba(139, 92, 246, 0.1);
            border: 1px solid var(--border);
            border-radius: 0.5rem;
            color: var(--text-primary);
            cursor: pointer;
            font-size: 0.875rem;
            transition: all 0.3s;
        }

        .btn-preset:hover {
            background: var(--primary);
            border-color: var(--primary);
        }

        .action-buttons {
            grid-column: 1 / -1;
            display: flex;
            gap: 1rem;
            justify-content: flex-end;
            padding-top: 1rem;
            border-top: 1px solid var(--border);
        }

        .btn-save {
            background: var(--primary);
            color: white;
        }

        .btn-save:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-reset {
            background: var(--warning);
            color: white;
        }

        .btn-reset:hover {
            background: #D97706;
            transform: translateY(-2px);
        }

        .unlimited-badge {
            background: linear-gradient(135deg, var(--primary) 0%, #EC4899 100%);
            color: white;
            padding: 0.125rem 0.5rem;
            border-radius: 0.25rem;
            font-size: 0.75rem;
            font-weight: 600;
        }

        @media (max-width: 768px) {
            body {
                padding: 1rem;
            }

            .header h1 {
                font-size: 1.5rem;
            }

            .permissions-content {
                grid-template-columns: 1fr;
            }

            .preset-buttons {
                flex-direction: column;
            }

            .action-buttons {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔐 User Permissions</h1>
            <a href="users.php" class="btn btn-back">
                ← Back to Users
            </a>
        </div>

        <?php if ($successMessage): ?>
            <script>
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: '<?php echo $successMessage; ?>',
                    timer: 2000,
                    showConfirmButton: false
                });
            </script>
        <?php endif; ?>

        <div class="permissions-grid">
            <?php foreach ($users as $user): ?>
                <?php if ($user['username'] === 'admin') continue; ?>
                <?php 
                    $userPerms = $permManager->getUserPermissions($user['username']);
                    $isUnlimited = $userPerms['storage_quota'] === -1;
                ?>
                
                <div class="user-card">
                    <form method="POST" id="form-<?php echo $user['username']; ?>">
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="update_permissions">
                        <input type="hidden" name="username" value="<?php echo htmlspecialchars($user['username']); ?>">
                        
                        <div class="user-header">
                            <div class="user-info">
                                <img src="../user-avatar.php?username=<?php echo urlencode($user['username']); ?>" 
                                     alt="<?php echo htmlspecialchars($user['display_name']); ?>" 
                                     class="user-avatar">
                                <div class="user-details">
                                    <h3><?php echo htmlspecialchars($user['display_name']); ?></h3>
                                    <p>@<?php echo htmlspecialchars($user['username']); ?></p>
                                </div>
                            </div>
                            <?php if ($user['role'] === 'admin'): ?>
                                <span class="badge-admin">ADMIN</span>
                            <?php endif; ?>
                        </div>

                        <div class="permissions-content">
                            <!-- Basic Permissions -->
                            <div class="permissions-section">
                                <h4 class="section-title">Basic Permissions</h4>
                                
                                <div class="permission-item">
                                    <input type="checkbox" 
                                           name="can_upload" 
                                           id="upload-<?php echo $user['username']; ?>"
                                           class="permission-checkbox"
                                           <?php echo $userPerms['can_upload'] ? 'checked' : ''; ?>>
                                    <label for="upload-<?php echo $user['username']; ?>" class="permission-label">
                                        Can Upload Images
                                    </label>
                                    <span class="permission-status <?php echo $userPerms['can_upload'] ? 'status-enabled' : 'status-disabled'; ?>">
                                        <?php echo $userPerms['can_upload'] ? 'Enabled' : 'Disabled'; ?>
                                    </span>
                                </div>

                                <div class="permission-item">
                                    <input type="checkbox" 
                                           name="can_create_folders" 
                                           id="folders-<?php echo $user['username']; ?>"
                                           class="permission-checkbox"
                                           <?php echo $userPerms['can_create_folders'] ? 'checked' : ''; ?>>
                                    <label for="folders-<?php echo $user['username']; ?>" class="permission-label">
                                        Can Create Folders
                                    </label>
                                    <span class="permission-status <?php echo $userPerms['can_create_folders'] ? 'status-enabled' : 'status-disabled'; ?>">
                                        <?php echo $userPerms['can_create_folders'] ? 'Enabled' : 'Disabled'; ?>
                                    </span>
                                </div>

                                <div class="permission-item">
                                    <input type="checkbox" 
                                           name="can_delete" 
                                           id="delete-<?php echo $user['username']; ?>"
                                           class="permission-checkbox"
                                           <?php echo $userPerms['can_delete'] ? 'checked' : ''; ?>>
                                    <label for="delete-<?php echo $user['username']; ?>" class="permission-label">
                                        Can Delete Own Content
                                    </label>
                                    <span class="permission-status <?php echo $userPerms['can_delete'] ? 'status-enabled' : 'status-disabled'; ?>">
                                        <?php echo $userPerms['can_delete'] ? 'Enabled' : 'Disabled'; ?>
                                    </span>
                                </div>

                                <div class="permission-item">
                                    <input type="checkbox" 
                                           name="can_share" 
                                           id="share-<?php echo $user['username']; ?>"
                                           class="permission-checkbox"
                                           <?php echo $userPerms['can_share'] ? 'checked' : ''; ?>>
                                    <label for="share-<?php echo $user['username']; ?>" class="permission-label">
                                        Can Share to Public
                                    </label>
                                    <span class="permission-status <?php echo $userPerms['can_share'] ? 'status-enabled' : 'status-disabled'; ?>">
                                        <?php echo $userPerms['can_share'] ? 'Enabled' : 'Disabled'; ?>
                                    </span>
                                </div>

                                <div class="permission-item">
                                    <input type="checkbox" 
                                           name="can_download_original" 
                                           id="download-<?php echo $user['username']; ?>"
                                           class="permission-checkbox"
                                           <?php echo $userPerms['can_download_original'] ? 'checked' : ''; ?>>
                                    <label for="download-<?php echo $user['username']; ?>" class="permission-label">
                                        Can Download Original Quality
                                    </label>
                                    <span class="permission-status <?php echo $userPerms['can_download_original'] ? 'status-enabled' : 'status-disabled'; ?>">
                                        <?php echo $userPerms['can_download_original'] ? 'Enabled' : 'Disabled'; ?>
                                    </span>
                                </div>
                            </div>

                            <!-- Quotas & Limits -->
                            <div class="permissions-section">
                                <h4 class="section-title">Quotas & Limits</h4>
                                
                                <div class="quota-input">
                                    <label for="quota-<?php echo $user['username']; ?>">
                                        Storage Quota 
                                        <?php if ($isUnlimited): ?>
                                            <span class="unlimited-badge">UNLIMITED</span>
                                        <?php endif; ?>
                                    </label>
                                    <select name="storage_quota" id="quota-<?php echo $user['username']; ?>">
                                        <option value="536870912" <?php echo $userPerms['storage_quota'] === 536870912 ? 'selected' : ''; ?>>512 MB</option>
                                        <option value="1073741824" <?php echo $userPerms['storage_quota'] === 1073741824 ? 'selected' : ''; ?>>1 GB</option>
                                        <option value="2147483648" <?php echo $userPerms['storage_quota'] === 2147483648 ? 'selected' : ''; ?>>2 GB</option>
                                        <option value="5368709120" <?php echo $userPerms['storage_quota'] === 5368709120 ? 'selected' : ''; ?>>5 GB</option>
                                        <option value="10737418240" <?php echo $userPerms['storage_quota'] === 10737418240 ? 'selected' : ''; ?>>10 GB</option>
                                        <option value="-1" <?php echo $userPerms['storage_quota'] === -1 ? 'selected' : ''; ?>>Unlimited</option>
                                    </select>
                                </div>

                                <div class="quota-input">
                                    <label for="max-folders-<?php echo $user['username']; ?>">
                                        Max Folders
                                        <?php if ($userPerms['max_folders'] === -1): ?>
                                            <span class="unlimited-badge">UNLIMITED</span>
                                        <?php endif; ?>
                                    </label>
                                    <input type="number" 
                                           name="max_folders" 
                                           id="max-folders-<?php echo $user['username']; ?>"
                                           value="<?php echo $userPerms['max_folders']; ?>"
                                           min="-1"
                                           placeholder="-1 for unlimited">
                                </div>

                                <div class="quota-input">
                                    <label for="max-images-<?php echo $user['username']; ?>">
                                        Max Images Per Folder
                                        <?php if ($userPerms['max_images_per_folder'] === -1): ?>
                                            <span class="unlimited-badge">UNLIMITED</span>
                                        <?php endif; ?>
                                    </label>
                                    <input type="number" 
                                           name="max_images_per_folder" 
                                           id="max-images-<?php echo $user['username']; ?>"
                                           value="<?php echo $userPerms['max_images_per_folder']; ?>"
                                           min="-1"
                                           placeholder="-1 for unlimited">
                                </div>
                            </div>

                            <!-- Presets -->
                            <div class="preset-section">
                                <h4 class="section-title">Quick Presets</h4>
                                <div class="preset-buttons">
                                    <?php foreach ($presets as $presetKey => $preset): ?>
                                        <button type="button" 
                                                class="btn-preset" 
                                                onclick="applyPreset('<?php echo $user['username']; ?>', '<?php echo $presetKey; ?>')">
                                            <?php echo htmlspecialchars($preset['name']); ?>
                                        </button>
                                    <?php endforeach; ?>
                                </div>
                            </div>

                            <!-- Action Buttons -->
                            <div class="action-buttons">
                                <button type="button" 
                                        class="btn btn-reset" 
                                        onclick="resetPermissions('<?php echo $user['username']; ?>')">
                                    Reset to Default
                                </button>
                                <button type="submit" class="btn btn-save">
                                    Save Permissions
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            <?php endforeach; ?>
        </div>
    </div>

    <script>
        function applyPreset(username, preset) {
            Swal.fire({
                title: 'Apply Preset?',
                text: 'This will override current permissions',
                icon: 'question',
                showCancelButton: true,
                confirmButtonColor: '#8B5CF6',
                confirmButtonText: 'Yes, apply it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="apply_preset">
                        <input type="hidden" name="username" value="${username}">
                        <input type="hidden" name="preset" value="${preset}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }

        function resetPermissions(username) {
            Swal.fire({
                title: 'Reset Permissions?',
                text: 'This will reset to default permissions',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#F59E0B',
                confirmButtonText: 'Yes, reset it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="reset_permissions">
                        <input type="hidden" name="username" value="${username}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }
    </script>
</body>
</html>
