<?php
/**
 * Edit Folder
 * Handles folder editing
 */

require_once '../../includes/config.php';
require_once '../../includes/Auth.php';
require_once '../../includes/CSRF.php';
require_once '../../includes/FolderManager.php';
require_once '../../includes/PermissionManager.php';

// Require login
Auth::requireLogin();

$username = Auth::getUsername();
$isAdmin = Auth::isAdmin();
$folderManager = new FolderManager();

// Get folder ID
$folderId = $_GET['id'] ?? '';

if (empty($folderId)) {
    header('Location: ../folders.php?error=invalid_folder');
    exit;
}

$folder = $folderManager->getFolder($folderId);

if (!$folder) {
    header('Location: ../folders.php?error=folder_not_found');
    exit;
}

// Check permission
if (!$folderManager->ownsFolder($folderId, $username) && !$isAdmin) {
    die('You do not have permission to edit this folder');
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && CSRF::validate()) {
    $name = trim($_POST['name'] ?? '');
    $visibility = $_POST['visibility'] ?? 'private';
    $description = trim($_POST['description'] ?? '');
    
    if (empty($name)) {
        $error = 'Folder name is required';
    } else {
        $result = $folderManager->updateFolder($folderId, [
            'name' => $name,
            'visibility' => $visibility,
            'description' => $description
        ]);
        
        if (isset($result['success'])) {
            header('Location: ../folders.php?success=folder_updated');
            exit;
        } else {
            $error = $result['error'] ?? 'Failed to update folder';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Folder - Galeriku ProMax</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #8B5CF6;
            --primary-dark: #7C3AED;
            --danger: #EF4444;
            --bg-dark: #0F172A;
            --bg-card: #1E293B;
            --bg-hover: #334155;
            --text-primary: #F1F5F9;
            --text-secondary: #94A3B8;
            --border: #334155;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, var(--bg-dark) 0%, #1a1a2e 100%);
            color: var(--text-primary);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }

        .container {
            max-width: 600px;
            width: 100%;
        }

        .card {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 1rem;
            padding: 2rem;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }

        .header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .header h1 {
            font-size: 2rem;
            background: linear-gradient(135deg, var(--primary) 0%, #EC4899 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 0.5rem;
        }

        .header p {
            color: var(--text-secondary);
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
            font-weight: 600;
        }

        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 0.875rem;
            background: rgba(139, 92, 246, 0.1);
            border: 1px solid var(--border);
            border-radius: 0.5rem;
            color: var(--text-primary);
            font-size: 1rem;
            transition: all 0.3s;
        }

        .form-group textarea {
            resize: vertical;
            min-height: 100px;
            font-family: inherit;
        }

        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--primary);
            background: rgba(139, 92, 246, 0.15);
            box-shadow: 0 0 0 3px rgba(139, 92, 246, 0.1);
        }

        .visibility-options {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }

        .visibility-option {
            position: relative;
        }

        .visibility-option input[type="radio"] {
            position: absolute;
            opacity: 0;
        }

        .visibility-label {
            display: block;
            padding: 1rem;
            background: rgba(139, 92, 246, 0.1);
            border: 2px solid var(--border);
            border-radius: 0.5rem;
            cursor: pointer;
            text-align: center;
            transition: all 0.3s;
        }

        .visibility-option input[type="radio"]:checked + .visibility-label {
            background: var(--primary);
            border-color: var(--primary);
            color: white;
        }

        .visibility-label:hover {
            border-color: var(--primary);
        }

        .visibility-icon {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }

        .visibility-name {
            font-weight: 600;
            margin-bottom: 0.25rem;
        }

        .visibility-desc {
            font-size: 0.75rem;
            color: var(--text-secondary);
        }

        .visibility-option input[type="radio"]:checked + .visibility-label .visibility-desc {
            color: rgba(255, 255, 255, 0.8);
        }

        .error-message {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--danger);
            color: var(--danger);
            padding: 1rem;
            border-radius: 0.5rem;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-group {
            display: flex;
            gap: 1rem;
        }

        .btn {
            flex: 1;
            padding: 0.875rem;
            border: none;
            border-radius: 0.5rem;
            cursor: pointer;
            font-size: 1rem;
            font-weight: 600;
            transition: all 0.3s;
            text-decoration: none;
            text-align: center;
            display: inline-block;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: 0 8px 16px rgba(139, 92, 246, 0.3);
        }

        .btn-secondary {
            background: var(--bg-hover);
            color: var(--text-primary);
        }

        .btn-secondary:hover {
            background: #475569;
            transform: translateY(-2px);
        }

        @media (max-width: 640px) {
            body {
                padding: 1rem;
            }

            .card {
                padding: 1.5rem;
            }

            .header h1 {
                font-size: 1.5rem;
            }

            .visibility-options {
                grid-template-columns: 1fr;
            }

            .btn-group {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <div class="header">
                <h1>✏️ Edit Folder</h1>
                <p>Update folder information</p>
            </div>

            <?php if (isset($error)): ?>
                <div class="error-message">
                    <span>⚠️</span>
                    <span><?php echo htmlspecialchars($error); ?></span>
                </div>
            <?php endif; ?>

            <form method="POST">
                <?php echo CSRF::field(); ?>

                <div class="form-group">
                    <label for="name">Folder Name *</label>
                    <input type="text" 
                           id="name" 
                           name="name" 
                           value="<?php echo htmlspecialchars($folder['name']); ?>"
                           required
                           autofocus>
                </div>

                <div class="form-group">
                    <label>Visibility *</label>
                    <div class="visibility-options">
                        <div class="visibility-option">
                            <input type="radio" 
                                   id="private" 
                                   name="visibility" 
                                   value="private" 
                                   <?php echo $folder['visibility'] === 'private' ? 'checked' : ''; ?>>
                            <label for="private" class="visibility-label">
                                <div class="visibility-icon">🔒</div>
                                <div class="visibility-name">Private</div>
                                <div class="visibility-desc">Only you can see</div>
                            </label>
                        </div>

                        <div class="visibility-option">
                            <input type="radio" 
                                   id="shared" 
                                   name="visibility" 
                                   value="shared"
                                   <?php echo $folder['visibility'] === 'shared' ? 'checked' : ''; ?>>
                            <label for="shared" class="visibility-label">
                                <div class="visibility-icon">🌐</div>
                                <div class="visibility-name">Shared</div>
                                <div class="visibility-desc">Everyone can see</div>
                            </label>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <label for="description">Description (Optional)</label>
                    <textarea id="description" 
                              name="description"><?php echo htmlspecialchars($folder['description']); ?></textarea>
                </div>

                <div class="btn-group">
                    <a href="../folders.php" class="btn btn-secondary">Cancel</a>
                    <button type="submit" class="btn btn-primary">Update Folder</button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>
