<?php
/**
 * Delete Folder Handler
 * Handles folder deletion with confirmation
 */

require_once '../../includes/config.php';
require_once '../../includes/Auth.php';
require_once '../../includes/CSRF.php';
require_once '../../includes/FolderManager.php';
require_once '../../includes/PermissionManager.php';

// Require login
Auth::requireLogin();

$username = Auth::getUsername();
$isAdmin = Auth::isAdmin();
$folderManager = new FolderManager();

// Get folder ID
$folderId = $_GET['id'] ?? '';

if (empty($folderId)) {
    header('Location: ../folders.php?error=invalid_folder');
    exit;
}

$folder = $folderManager->getFolder($folderId);

if (!$folder) {
    header('Location: ../folders.php?error=folder_not_found');
    exit;
}

// Check permission
if (!$folderManager->ownsFolder($folderId, $username) && !$isAdmin) {
    die('You do not have permission to delete this folder');
}

// Handle deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && CSRF::validate()) {
    $deleteImages = isset($_POST['delete_images']) && $_POST['delete_images'] === '1';
    
    $result = $folderManager->deleteFolder($folderId, $deleteImages);
    
    if (isset($result['success'])) {
        header('Location: ../folders.php?success=folder_deleted');
        exit;
    } else {
        $error = $result['error'] ?? 'Failed to delete folder';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Delete Folder - Galeriku ProMax</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --danger: #EF4444;
            --danger-dark: #DC2626;
            --bg-dark: #0F172A;
            --bg-card: #1E293B;
            --bg-hover: #334155;
            --text-primary: #F1F5F9;
            --text-secondary: #94A3B8;
            --border: #334155;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, var(--bg-dark) 0%, #1a1a2e 100%);
            color: var(--text-primary);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }

        .container {
            max-width: 500px;
            width: 100%;
        }

        .card {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 1rem;
            padding: 2rem;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }

        .icon-warning {
            width: 80px;
            height: 80px;
            margin: 0 auto 1.5rem;
            background: rgba(239, 68, 68, 0.1);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 3rem;
        }

        .header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .header h1 {
            font-size: 1.75rem;
            color: var(--danger);
            margin-bottom: 0.5rem;
        }

        .header p {
            color: var(--text-secondary);
        }

        .folder-info {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--danger);
            border-radius: 0.5rem;
            padding: 1rem;
            margin-bottom: 1.5rem;
        }

        .folder-info strong {
            display: block;
            margin-bottom: 0.5rem;
        }

        .folder-info p {
            color: var(--text-secondary);
            font-size: 0.875rem;
        }

        .warning-box {
            background: rgba(239, 68, 68, 0.05);
            border: 1px solid var(--danger);
            border-radius: 0.5rem;
            padding: 1rem;
            margin-bottom: 1.5rem;
        }

        .warning-box p {
            color: var(--text-secondary);
            font-size: 0.875rem;
            line-height: 1.6;
        }

        .checkbox-group {
            margin-bottom: 1.5rem;
        }

        .checkbox-label {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 1rem;
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--border);
            border-radius: 0.5rem;
            cursor: pointer;
            transition: all 0.3s;
        }

        .checkbox-label:hover {
            border-color: var(--danger);
            background: rgba(239, 68, 68, 0.15);
        }

        .checkbox-label input {
            width: 20px;
            height: 20px;
            cursor: pointer;
            accent-color: var(--danger);
        }

        .checkbox-label span {
            font-size: 0.875rem;
        }

        .error-message {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid var(--danger);
            color: var(--danger);
            padding: 1rem;
            border-radius: 0.5rem;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-group {
            display: flex;
            gap: 1rem;
        }

        .btn {
            flex: 1;
            padding: 0.875rem;
            border: none;
            border-radius: 0.5rem;
            cursor: pointer;
            font-size: 1rem;
            font-weight: 600;
            transition: all 0.3s;
            text-decoration: none;
            text-align: center;
            display: inline-block;
        }

        .btn-danger {
            background: var(--danger);
            color: white;
        }

        .btn-danger:hover {
            background: var(--danger-dark);
            transform: translateY(-2px);
            box-shadow: 0 8px 16px rgba(239, 68, 68, 0.3);
        }

        .btn-secondary {
            background: var(--bg-hover);
            color: var(--text-primary);
        }

        .btn-secondary:hover {
            background: #475569;
            transform: translateY(-2px);
        }

        @media (max-width: 640px) {
            body {
                padding: 1rem;
            }

            .card {
                padding: 1.5rem;
            }

            .btn-group {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <div class="icon-warning">⚠️</div>

            <div class="header">
                <h1>Delete Folder?</h1>
                <p>This action cannot be undone</p>
            </div>

            <?php if (isset($error)): ?>
                <div class="error-message">
                    <span>⚠️</span>
                    <span><?php echo htmlspecialchars($error); ?></span>
                </div>
            <?php endif; ?>

            <div class="folder-info">
                <strong><?php echo htmlspecialchars($folder['name']); ?></strong>
                <p>Contains <?php echo $folder['image_count']; ?> image(s)</p>
            </div>

            <div class="warning-box">
                <p>
                    <strong>⚠️ Warning:</strong> Deleting this folder will remove it permanently. 
                    You can choose to keep the images (they will be moved to your gallery) or delete them along with the folder.
                </p>
            </div>

            <form method="POST">
                <?php echo CSRF::field(); ?>

                <?php if ($folder['image_count'] > 0): ?>
                    <div class="checkbox-group">
                        <label class="checkbox-label">
                            <input type="checkbox" name="delete_images" value="1">
                            <span>Also delete all <?php echo $folder['image_count']; ?> image(s) in this folder</span>
                        </label>
                    </div>
                <?php endif; ?>

                <div class="btn-group">
                    <a href="../folders.php" class="btn btn-secondary">Cancel</a>
                    <button type="submit" class="btn btn-danger">Delete Folder</button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>
