<?php
/**
 * PermissionManager.php
 * Handles user permissions and access control
 */

class PermissionManager {
    private $dataFile;
    
    // Default permissions for new users
    private $defaultPermissions = [
        'can_upload' => true,
        'can_create_folders' => true,
        'can_delete' => true,
        'can_share' => true,
        'can_download_original' => true,
        'storage_quota' => 1073741824, // 1GB in bytes
        'max_folders' => 50,
        'max_images_per_folder' => 1000
    ];
    
    public function __construct() {
        $this->dataFile = STORAGE_PATH . '/permissions.json';
        $this->initDataFile();
    }
    
    /**
     * Initialize permissions.json
     */
    private function initDataFile() {
        if (!file_exists($this->dataFile)) {
            // Create default admin permissions
            $defaultData = [
                'admin' => array_merge($this->defaultPermissions, [
                    'storage_quota' => -1, // Unlimited for admin
                    'max_folders' => -1,
                    'max_images_per_folder' => -1
                ])
            ];
            file_put_contents($this->dataFile, json_encode($defaultData, JSON_PRETTY_PRINT));
        }
    }
    
    /**
     * Load permissions
     */
    private function loadPermissions() {
        if (!file_exists($this->dataFile)) {
            return [];
        }
        
        $data = file_get_contents($this->dataFile);
        return json_decode($data, true) ?: [];
    }
    
    /**
     * Save permissions
     */
    private function savePermissions($permissions) {
        return file_put_contents($this->dataFile, json_encode($permissions, JSON_PRETTY_PRINT));
    }
    
    /**
     * Get user permissions
     */
    public function getUserPermissions($username) {
        $permissions = $this->loadPermissions();
        
        // Return user's permissions or defaults
        return $permissions[$username] ?? $this->defaultPermissions;
    }
    
    /**
     * Set user permissions
     */
    public function setUserPermissions($username, $newPermissions) {
        $permissions = $this->loadPermissions();
        
        // Merge with existing or defaults
        $current = $permissions[$username] ?? $this->defaultPermissions;
        $permissions[$username] = array_merge($current, $newPermissions);
        
        if ($this->savePermissions($permissions)) {
            return ['success' => true, 'permissions' => $permissions[$username]];
        }
        
        return ['error' => 'Failed to save permissions'];
    }
    
    /**
     * Check specific permission
     */
    public function can($username, $permission) {
        $permissions = $this->getUserPermissions($username);
        return $permissions[$permission] ?? false;
    }
    
    /**
     * Check if user can upload
     */
    public function canUpload($username) {
        return $this->can($username, 'can_upload');
    }
    
    /**
     * Check if user can create folders
     */
    public function canCreateFolders($username) {
        return $this->can($username, 'can_create_folders');
    }
    
    /**
     * Check if user can delete
     */
    public function canDelete($username) {
        return $this->can($username, 'can_delete');
    }
    
    /**
     * Check if user can share
     */
    public function canShare($username) {
        return $this->can($username, 'can_share');
    }
    
    /**
     * Check if user can download original
     */
    public function canDownloadOriginal($username) {
        return $this->can($username, 'can_download_original');
    }
    
    /**
     * Get storage quota
     */
    public function getStorageQuota($username) {
        $permissions = $this->getUserPermissions($username);
        return $permissions['storage_quota'] ?? $this->defaultPermissions['storage_quota'];
    }
    
    /**
     * Check if user has reached storage quota
     */
    public function hasReachedStorageQuota($username, $currentUsage) {
        $quota = $this->getStorageQuota($username);
        
        // -1 means unlimited
        if ($quota === -1) {
            return false;
        }
        
        return $currentUsage >= $quota;
    }
    
    /**
     * Get max folders allowed
     */
    public function getMaxFolders($username) {
        $permissions = $this->getUserPermissions($username);
        return $permissions['max_folders'] ?? $this->defaultPermissions['max_folders'];
    }
    
    /**
     * Check if user has reached max folders
     */
    public function hasReachedMaxFolders($username, $currentCount) {
        $maxFolders = $this->getMaxFolders($username);
        
        // -1 means unlimited
        if ($maxFolders === -1) {
            return false;
        }
        
        return $currentCount >= $maxFolders;
    }
    
    /**
     * Require permission or die
     */
    public function requirePermission($username, $permission, $errorMessage = null) {
        if (!$this->can($username, $permission)) {
            if (!$errorMessage) {
                $errorMessage = "You don't have permission to perform this action";
            }
            
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                header('Content-Type: application/json');
                echo json_encode(['error' => $errorMessage]);
            } else {
                die($errorMessage);
            }
            exit;
        }
    }
    
    /**
     * Get all permissions for admin view
     */
    public function getAllPermissions() {
        return $this->loadPermissions();
    }
    
    /**
     * Reset user permissions to default
     */
    public function resetToDefault($username) {
        $permissions = $this->loadPermissions();
        $permissions[$username] = $this->defaultPermissions;
        
        if ($this->savePermissions($permissions)) {
            return ['success' => true, 'permissions' => $permissions[$username]];
        }
        
        return ['error' => 'Failed to reset permissions'];
    }
    
    /**
     * Delete user permissions
     */
    public function deleteUserPermissions($username) {
        $permissions = $this->loadPermissions();
        
        if (isset($permissions[$username])) {
            unset($permissions[$username]);
            
            if ($this->savePermissions($permissions)) {
                return ['success' => true];
            }
        }
        
        return ['error' => 'Failed to delete permissions'];
    }
    
    /**
     * Get permission presets
     */
    public function getPresets() {
        return [
            'full' => [
                'name' => 'Full Access',
                'permissions' => [
                    'can_upload' => true,
                    'can_create_folders' => true,
                    'can_delete' => true,
                    'can_share' => true,
                    'can_download_original' => true,
                    'storage_quota' => -1,
                    'max_folders' => -1,
                    'max_images_per_folder' => -1
                ]
            ],
            'standard' => [
                'name' => 'Standard User',
                'permissions' => $this->defaultPermissions
            ],
            'limited' => [
                'name' => 'Limited Access',
                'permissions' => [
                    'can_upload' => true,
                    'can_create_folders' => false,
                    'can_delete' => false,
                    'can_share' => false,
                    'can_download_original' => false,
                    'storage_quota' => 536870912, // 512MB
                    'max_folders' => 10,
                    'max_images_per_folder' => 100
                ]
            ],
            'readonly' => [
                'name' => 'Read Only',
                'permissions' => [
                    'can_upload' => false,
                    'can_create_folders' => false,
                    'can_delete' => false,
                    'can_share' => false,
                    'can_download_original' => false,
                    'storage_quota' => 0,
                    'max_folders' => 0,
                    'max_images_per_folder' => 0
                ]
            ]
        ];
    }
    
    /**
     * Apply preset to user
     */
    public function applyPreset($username, $presetName) {
        $presets = $this->getPresets();
        
        if (!isset($presets[$presetName])) {
            return ['error' => 'Invalid preset'];
        }
        
        return $this->setUserPermissions($username, $presets[$presetName]['permissions']);
    }
}
