<?php
/**
 * ImageManager.php
 * Handles image metadata, visibility, and folder organization
 */

class ImageManager {
    private $dataFile;
    
    public function __construct() {
        $this->dataFile = STORAGE_PATH . '/images.json';
        $this->initDataFile();
    }
    
    /**
     * Initialize images.json
     */
    private function initDataFile() {
        if (!file_exists($this->dataFile)) {
            file_put_contents($this->dataFile, json_encode([], JSON_PRETTY_PRINT));
        }
    }
    
    /**
     * Load images metadata
     */
    private function loadImages() {
        if (!file_exists($this->dataFile)) {
            return [];
        }
        
        $data = file_get_contents($this->dataFile);
        return json_decode($data, true) ?: [];
    }
    
    /**
     * Save images metadata
     */
    private function saveImages($images) {
        return file_put_contents($this->dataFile, json_encode($images, JSON_PRETTY_PRINT));
    }
    
    /**
     * Generate unique image ID
     */
    private function generateImageId() {
        return 'img_' . bin2hex(random_bytes(8)) . '_' . time();
    }
    
    /**
     * Add image metadata
     */
    public function addImage($filename, $owner, $folderId = null, $visibility = 'private', $size = 0) {
        $images = $this->loadImages();
        
        $imageId = $this->generateImageId();
        
        $images[$imageId] = [
            'id' => $imageId,
            'filename' => $filename,
            'owner' => $owner,
            'folder_id' => $folderId,
            'visibility' => in_array($visibility, ['private', 'shared']) ? $visibility : 'private',
            'uploaded' => time(),
            'size' => $size,
            'tags' => [],
            'downloads' => 0,
            'views' => 0
        ];
        
        if ($this->saveImages($images)) {
            // Update folder count if in folder
            if ($folderId) {
                $folderManager = new FolderManager();
                $folderManager->incrementImageCount($folderId);
                
                // Set folder thumbnail if first image
                $folder = $folderManager->getFolder($folderId);
                if ($folder && !$folder['thumbnail']) {
                    $folderManager->updateFolder($folderId, ['thumbnail' => $filename]);
                }
            }
            
            return ['success' => true, 'image_id' => $imageId, 'image' => $images[$imageId]];
        }
        
        return ['error' => 'Failed to add image metadata'];
    }
    
    /**
     * Get image by ID
     */
    public function getImage($imageId) {
        $images = $this->loadImages();
        return $images[$imageId] ?? null;
    }
    
    /**
     * Get image by filename and owner
     */
    public function getImageByFilename($filename, $owner) {
        $images = $this->loadImages();
        
        foreach ($images as $image) {
            if ($image['filename'] === $filename && $image['owner'] === $owner) {
                return $image;
            }
        }
        
        return null;
    }
    
    /**
     * Update image metadata
     */
    public function updateImage($imageId, $data) {
        $images = $this->loadImages();
        
        if (!isset($images[$imageId])) {
            return ['error' => 'Image not found'];
        }
        
        $image = $images[$imageId];
        
        // Update allowed fields
        if (isset($data['filename'])) {
            $image['filename'] = $data['filename'];
        }
        
        if (isset($data['folder_id'])) {
            $oldFolderId = $image['folder_id'];
            $image['folder_id'] = $data['folder_id'];
            
            // Update folder counts
            if ($oldFolderId !== $data['folder_id']) {
                $folderManager = new FolderManager();
                
                if ($oldFolderId) {
                    $folderManager->decrementImageCount($oldFolderId);
                }
                
                if ($data['folder_id']) {
                    $folderManager->incrementImageCount($data['folder_id']);
                }
            }
        }
        
        if (isset($data['visibility']) && in_array($data['visibility'], ['private', 'shared'])) {
            $image['visibility'] = $data['visibility'];
        }
        
        if (isset($data['tags']) && is_array($data['tags'])) {
            $image['tags'] = $data['tags'];
        }
        
        $images[$imageId] = $image;
        
        if ($this->saveImages($images)) {
            return ['success' => true, 'image' => $image];
        }
        
        return ['error' => 'Failed to update image'];
    }
    
    /**
     * Delete image metadata
     */
    public function deleteImage($imageId) {
        $images = $this->loadImages();
        
        if (!isset($images[$imageId])) {
            return ['error' => 'Image not found'];
        }
        
        $image = $images[$imageId];
        
        // Update folder count
        if ($image['folder_id']) {
            $folderManager = new FolderManager();
            $folderManager->decrementImageCount($image['folder_id']);
        }
        
        unset($images[$imageId]);
        
        if ($this->saveImages($images)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to delete image metadata'];
    }
    
    /**
     * Get user's images
     */
    public function getUserImages($username, $visibility = null, $folderId = null) {
        $images = $this->loadImages();
        $userImages = [];
        
        foreach ($images as $image) {
            // Filter by owner
            if ($image['owner'] !== $username) {
                continue;
            }
            
            // Filter by visibility
            if ($visibility && $image['visibility'] !== $visibility) {
                continue;
            }
            
            // Filter by folder
            if ($folderId !== null) {
                if ($folderId === 'root' && $image['folder_id'] !== null) {
                    continue;
                } elseif ($folderId !== 'root' && $image['folder_id'] !== $folderId) {
                    continue;
                }
            }
            
            $userImages[] = $image;
        }
        
        // Sort by uploaded date (newest first)
        usort($userImages, function($a, $b) {
            return $b['uploaded'] - $a['uploaded'];
        });
        
        return $userImages;
    }
    
    /**
     * Get shared images (public gallery)
     */
    public function getSharedImages($folderId = null) {
        $images = $this->loadImages();
        $sharedImages = [];
        
        foreach ($images as $image) {
            if ($image['visibility'] !== 'shared') {
                continue;
            }
            
            // Filter by folder
            if ($folderId !== null) {
                if ($folderId === 'root' && $image['folder_id'] !== null) {
                    continue;
                } elseif ($folderId !== 'root' && $image['folder_id'] !== $folderId) {
                    continue;
                }
            }
            
            $sharedImages[] = $image;
        }
        
        // Sort by uploaded date (newest first)
        usort($sharedImages, function($a, $b) {
            return $b['uploaded'] - $a['uploaded'];
        });
        
        return $sharedImages;
    }
    
    /**
     * Get images in folder
     */
    public function getImagesByFolder($folderId) {
        $images = $this->loadImages();
        $folderImages = [];
        
        foreach ($images as $image) {
            if ($image['folder_id'] === $folderId) {
                $folderImages[] = $image;
            }
        }
        
        return $folderImages;
    }
    
    /**
     * Move images to root (no folder)
     */
    public function moveImagesToRoot($folderId) {
        $images = $this->loadImages();
        $updated = false;
        
        foreach ($images as $imageId => $image) {
            if ($image['folder_id'] === $folderId) {
                $images[$imageId]['folder_id'] = null;
                $updated = true;
            }
        }
        
        if ($updated) {
            $this->saveImages($images);
        }
        
        return $updated;
    }
    
    /**
     * Search images
     */
    public function searchImages($query, $username = null, $isAdmin = false) {
        $images = $this->loadImages();
        $results = [];
        $query = strtolower(trim($query));
        
        if (empty($query)) {
            return [];
        }
        
        foreach ($images as $image) {
            // Check access
            if (!$isAdmin) {
                if ($image['owner'] !== $username && $image['visibility'] !== 'shared') {
                    continue;
                }
            }
            
            // Search in filename
            if (strpos(strtolower($image['filename']), $query) !== false) {
                $results[] = $image;
            }
        }
        
        return $results;
    }
    
    /**
     * Increment view count
     */
    public function incrementViews($imageId) {
        $images = $this->loadImages();
        
        if (isset($images[$imageId])) {
            $images[$imageId]['views'] = ($images[$imageId]['views'] ?? 0) + 1;
            $this->saveImages($images);
        }
    }
    
    /**
     * Increment download count
     */
    public function incrementDownloads($imageId) {
        $images = $this->loadImages();
        
        if (isset($images[$imageId])) {
            $images[$imageId]['downloads'] = ($images[$imageId]['downloads'] ?? 0) + 1;
            $this->saveImages($images);
        }
    }
}
