<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/ImageManager.php';
require_once __DIR__ . '/../includes/PermissionManager.php';

Auth::requireLogin();

$username = Auth::getUsername();
$isAdmin = Auth::isAdmin();
$currentUser = Auth::getCurrentUser();

$imageManager = new ImageManager();
$permManager = new PermissionManager();

// Get view mode and pagination
$view = $_GET['view'] ?? 'my';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = ITEMS_PER_PAGE;

// Get images based on view mode
if ($view === 'shared') {
    $allImages = $imageManager->getSharedImages();
} else {
    $allImages = $imageManager->getUserImages($username);
}

// Manual pagination
$totalImages = count($allImages);
$totalPages = ceil($totalImages / $perPage);
$offset = ($page - 1) * $perPage;
$images = array_slice($allImages, $offset, $perPage);

// User stats
$storageUsed = $currentUser['storage_used'] ?? 0;
$totalUploads = $currentUser['total_uploads'] ?? 0;

// Page config
$pageTitle = 'Gallery - Galeriku ProMax';
$pageHeader = 'Gallery';
$additionalCSS = ['assets/css/gallery.css'];

// Include header
include __DIR__ . '/components/header.php';
?>

<!-- Stats Cards -->
<div class="grid grid-4" style="margin-bottom: var(--spacing-xl);">
    <div class="card">
        <div class="flex items-center gap-md">
            <div style="width: 48px; height: 48px; border-radius: var(--radius-lg); background: linear-gradient(135deg, rgba(139, 92, 246, 0.15), rgba(236, 72, 153, 0.15)); display: flex; align-items: center; justify-content: center;">
                <?php echo UIIcons::image('icon-lg'); ?>
            </div>
            <div>
                <div style="font-size: 0.75rem; color: var(--text-secondary); font-weight: 600; text-transform: uppercase;">Total Images</div>
                <div style="font-size: 1.5rem; font-weight: 700; color: var(--primary);"><?php echo $totalUploads; ?></div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="flex items-center gap-md">
            <div style="width: 48px; height: 48px; border-radius: var(--radius-lg); background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(6, 182, 212, 0.15)); display: flex; align-items: center; justify-content: center;">
                <?php echo UIIcons::folder('icon-lg'); ?>
            </div>
            <div>
                <div style="font-size: 0.75rem; color: var(--text-secondary); font-weight: 600; text-transform: uppercase;">Storage Used</div>
                <div style="font-size: 1.5rem; font-weight: 700; color: var(--success);"><?php echo formatBytes($storageUsed); ?></div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="flex items-center gap-md">
            <div style="width: 48px; height: 48px; border-radius: var(--radius-lg); background: linear-gradient(135deg, rgba(59, 130, 246, 0.15), rgba(139, 92, 246, 0.15)); display: flex; align-items: center; justify-content: center;">
                <?php echo UIIcons::eye('icon-lg'); ?>
            </div>
            <div>
                <div style="font-size: 0.75rem; color: var(--text-secondary); font-weight: 600; text-transform: uppercase;">Current View</div>
                <div style="font-size: 1.125rem; font-weight: 700; color: var(--info);"><?php echo $view === 'shared' ? 'Shared' : 'My Photos'; ?></div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="flex items-center gap-md">
            <div style="width: 48px; height: 48px; border-radius: var(--radius-lg); background: linear-gradient(135deg, rgba(245, 158, 11, 0.15), rgba(239, 68, 68, 0.15)); display: flex; align-items: center; justify-content: center;">
                <?php echo UIIcons::search('icon-lg'); ?>
            </div>
            <div>
                <div style="font-size: 0.75rem; color: var(--text-secondary); font-weight: 600; text-transform: uppercase;">Page</div>
                <div style="font-size: 1.5rem; font-weight: 700; color: var(--warning);"><?php echo $page; ?> / <?php echo max(1, $totalPages); ?></div>
            </div>
        </div>
    </div>
</div>

<!-- View Toggle -->
<div class="card" style="margin-bottom: var(--spacing-xl);">
    <div class="flex items-center justify-between">
        <h2 style="font-size: 1.125rem; font-weight: 600;">View Mode</h2>
        <div class="flex gap-sm">
            <a href="?view=my" class="btn <?php echo $view === 'my' ? 'btn-primary' : 'btn-secondary'; ?> btn-sm">
                <?php echo UIIcons::lock('icon-sm'); ?>
                <span>My Photos</span>
            </a>
            <a href="?view=shared" class="btn <?php echo $view === 'shared' ? 'btn-primary' : 'btn-secondary'; ?> btn-sm">
                <?php echo UIIcons::globe('icon-sm'); ?>
                <span>Shared Gallery</span>
            </a>
        </div>
    </div>
</div>

<!-- Gallery Grid -->
<?php if (!empty($images)): ?>
    <div class="gallery-grid">
        <?php foreach ($images as $image): ?>
            <div class="gallery-item" onclick="openLightbox('<?php echo htmlspecialchars($image['filename']); ?>', '<?php echo htmlspecialchars($image['owner']); ?>')">
                <div class="gallery-image">
                    <img src="image.php?file=<?php echo urlencode($image['filename']); ?>&username=<?php echo urlencode($image['owner']); ?>" 
                         alt="<?php echo htmlspecialchars($image['filename']); ?>" 
                         loading="lazy">
                </div>
                <div class="gallery-info">
                    <div class="gallery-title"><?php echo htmlspecialchars($image['filename']); ?></div>
                    <div class="gallery-meta">
                        <span class="badge badge-<?php echo $image['visibility'] === 'private' ? 'danger' : 'success'; ?>">
                            <?php echo $image['visibility'] === 'private' ? UIIcons::lock('icon-sm') : UIIcons::globe('icon-sm'); ?>
                            <?php echo ucfirst($image['visibility']); ?>
                        </span>
                        <div class="flex gap-xs">
                            <button class="btn-icon-sm" onclick="event.stopPropagation(); downloadImage('<?php echo htmlspecialchars($image['filename']); ?>', '<?php echo htmlspecialchars($image['owner']); ?>');" title="Download">
                                <?php echo UIIcons::download('icon-sm'); ?>
                            </button>
                            <?php if ($image['owner'] === $username || $isAdmin): ?>
                                <button class="btn-icon-sm" onclick="event.stopPropagation(); deleteImage('<?php echo htmlspecialchars($image['filename']); ?>');" title="Delete">
                                    <?php echo UIIcons::trash('icon-sm'); ?>
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
    
    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=1&view=<?php echo $view; ?>" class="pagination-item">
                    <?php echo UIIcons::chevronLeft('icon-sm'); ?>
                    <?php echo UIIcons::chevronLeft('icon-sm'); ?>
                </a>
                <a href="?page=<?php echo $page - 1; ?>&view=<?php echo $view; ?>" class="pagination-item">
                    <?php echo UIIcons::chevronLeft('icon-sm'); ?>
                </a>
            <?php endif; ?>
            
            <?php
            $start = max(1, $page - 2);
            $end = min($totalPages, $page + 2);
            for ($i = $start; $i <= $end; $i++):
            ?>
                <a href="?page=<?php echo $i; ?>&view=<?php echo $view; ?>" 
                   class="pagination-item <?php echo $i === $page ? 'active' : ''; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
                <a href="?page=<?php echo $page + 1; ?>&view=<?php echo $view; ?>" class="pagination-item">
                    <?php echo UIIcons::chevronRight('icon-sm'); ?>
                </a>
                <a href="?page=<?php echo $totalPages; ?>&view=<?php echo $view; ?>" class="pagination-item">
                    <?php echo UIIcons::chevronRight('icon-sm'); ?>
                    <?php echo UIIcons::chevronRight('icon-sm'); ?>
                </a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
    
<?php else: ?>
    <div class="empty-state">
        <?php echo UIIcons::image('empty-icon'); ?>
        <h3 class="empty-title">No images found</h3>
        <p class="empty-text">
            <?php if ($view === 'shared'): ?>
                There are no shared images yet. Switch to "My Photos" or upload some images!
            <?php else: ?>
                Start by uploading your first image!
            <?php endif; ?>
        </p>
        <a href="admin/upload.php" class="btn btn-primary">
            <?php echo UIIcons::upload(); ?>
            <span>Upload Image</span>
        </a>
    </div>
<?php endif; ?>

<!-- Lightbox -->
<div id="lightbox" class="lightbox" onclick="closeLightbox()">
    <button class="lightbox-close" onclick="closeLightbox()">
        <?php echo UIIcons::x(); ?>
    </button>
    <img id="lightbox-img" src="" alt="">
</div>

<style>
.gallery-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: var(--spacing-lg);
}

.gallery-item {
    background: var(--bg-card);
    border: 1px solid var(--border);
    border-radius: var(--radius-xl);
    overflow: hidden;
    cursor: pointer;
    transition: all var(--transition-base);
}

.gallery-item:hover {
    border-color: var(--primary);
    transform: translateY(-4px);
    box-shadow: var(--shadow-lg);
}

.gallery-image {
    width: 100%;
    height: 240px;
    background: repeating-conic-gradient(var(--bg-secondary) 0% 25%, var(--bg-tertiary) 0% 50%) 50% / 20px 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    overflow: hidden;
}

.gallery-image img {
    max-width: 100%;
    max-height: 100%;
    object-fit: contain;
}

.gallery-info {
    padding: var(--spacing-md);
}

.gallery-title {
    font-weight: 600;
    margin-bottom: var(--spacing-sm);
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.gallery-meta {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.btn-icon-sm {
    width: 32px;
    height: 32px;
    border: 1px solid var(--border);
    background: var(--bg-secondary);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all var(--transition-fast);
    color: var(--text-secondary);
}

.btn-icon-sm:hover {
    background: var(--primary);
    border-color: var(--primary);
    color: white;
}

.lightbox {
    position: fixed;
    inset: 0;
    background: rgba(0, 0, 0, 0.95);
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}

.lightbox.active {
    display: flex;
}

.lightbox img {
    max-width: 90%;
    max-height: 90%;
    object-fit: contain;
}

.lightbox-close {
    position: absolute;
    top: var(--spacing-xl);
    right: var(--spacing-xl);
    width: 48px;
    height: 48px;
    background: var(--primary);
    border: none;
    border-radius: 50%;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all var(--transition-base);
}

.lightbox-close:hover {
    background: var(--primary-dark);
    transform: scale(1.1);
}

@media (max-width: 768px) {
    .gallery-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
function openLightbox(filename, owner) {
    const lightbox = document.getElementById('lightbox');
    const img = document.getElementById('lightbox-img');
    img.src = 'image.php?file=' + encodeURIComponent(filename) + '&username=' + encodeURIComponent(owner);
    lightbox.classList.add('active');
}

function closeLightbox() {
    document.getElementById('lightbox').classList.remove('active');
}

function downloadImage(filename, owner) {
    window.location.href = 'image.php?file=' + encodeURIComponent(filename) + '&username=' + encodeURIComponent(owner) + '&download=1';
}

function deleteImage(filename) {
    if (confirm('Are you sure you want to delete this image?')) {
        window.location.href = 'admin/delete.php?file=' + encodeURIComponent(filename);
    }
}

document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') closeLightbox();
});
</script>

<?php include __DIR__ . '/components/footer.php'; ?>
