<?php
/**
 * PNG Handler Class - ProMax Version 2.0
 * Enhanced with per-user storage, better optimization, and fixed transparency issues
 */

class PngHandler
{
    private $errors = [];
    private $userId = null;
    private $username = null;
    
    public function __construct($userId = null, $username = null)
    {
        $this->userId = $userId;
        $this->username = $username;
    }
    
    /**
     * Set user context
     */
    public function setUser($userId, $username)
    {
        $this->userId = $userId;
        $this->username = $username;
    }
    
    /**
     * Get user upload path
     */
    private function getUserPath($type = 'originals')
    {
        if (!$this->username) {
            return constant(strtoupper($type) . '_PATH');
        }
        
        $userDir = USERS_PATH . '/' . $this->username . '/' . $type;
        if (!is_dir($userDir)) {
            mkdir($userDir, 0755, true);
        }
        
        return $userDir;
    }
    
    /**
     * Validate uploaded PNG file
     */
    public function validate($file)
    {
        $this->errors = [];
        
        // Check if file exists
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            $this->errors[] = 'Invalid file upload';
            return false;
        }
        
        // Check file size
        if ($file['size'] > MAX_FILE_SIZE) {
            $this->errors[] = 'File too large (max ' . formatBytes(MAX_FILE_SIZE) . ')';
            return false;
        }
        
        // Check MIME type
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if ($mimeType !== ALLOWED_MIME) {
            $this->errors[] = 'Only PNG files allowed (detected: ' . $mimeType . ')';
            return false;
        }
        
        // Check extension
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if ($ext !== 'png') {
            $this->errors[] = 'File must have .png extension';
            return false;
        }
        
        // Validate PNG integrity
        $img = @imagecreatefrompng($file['tmp_name']);
        if (!$img) {
            $this->errors[] = 'Corrupt or invalid PNG file';
            return false;
        }
        imagedestroy($img);
        
        return true;
    }
    
    /**
     * Process uploaded PNG
     */
    public function process($file)
    {
        if (!$this->validate($file)) {
            return false;
        }
        
        // Generate unique filename
        $filename = $this->generateFilename($file['name']);
        
        $paths = [
            'original' => $this->getUserPath('originals') . '/' . $filename,
            'optimized' => $this->getUserPath('optimized') . '/' . $filename,
            'thumb' => $this->getUserPath('thumbs') . '/' . $filename
        ];
        
        try {
            // 1. Save original
            if (!move_uploaded_file($file['tmp_name'], $paths['original'])) {
                throw new Exception('Failed to save original file');
            }
            
            // 2. Create optimized version
            $this->optimize($paths['original'], $paths['optimized']);
            
            // 3. Create thumbnail
            $this->createThumbnail($paths['original'], $paths['thumb']);
            
            // 4. Update user storage if user context is set
            if ($this->username) {
                $userManager = new UserManager();
                $userManager->incrementUploads($this->username);
                $userManager->updateStorageUsed($this->username);
                $userManager->logActivity($this->username, 'file_uploaded', 'Uploaded: ' . $filename);
            }
            
            return [
                'success' => true,
                'filename' => $filename,
                'original' => $paths['original'],
                'optimized' => $paths['optimized'],
                'thumb' => $paths['thumb'],
                'size' => filesize($paths['original']),
                'uploaded_at' => time()
            ];
            
        } catch (Exception $e) {
            // Cleanup on error
            foreach ($paths as $path) {
                if (file_exists($path)) {
                    @unlink($path);
                }
            }
            
            $this->errors[] = $e->getMessage();
            return false;
        }
    }
    
    /**
     * Optimize PNG with compression while preserving transparency
     */
    public function optimize($sourcePath, $destPath)
    {
        if (!file_exists($sourcePath)) {
            throw new Exception('Source file not found');
        }
        
        // Load PNG
        $img = @imagecreatefrompng($sourcePath);
        if (!$img) {
            throw new Exception('Failed to load PNG');
        }
        
        // === CRITICAL: PRESERVE ALPHA CHANNEL ===
        imagealphablending($img, false);
        imagesavealpha($img, true);
        
        // Save with compression
        $success = imagepng($img, $destPath, PNG_COMPRESSION);
        
        imagedestroy($img);
        
        if (!$success) {
            throw new Exception('Failed to save optimized PNG');
        }
        
        return true;
    }
    
    /**
     * Create thumbnail with preserved transparency
     */
    public function createThumbnail($sourcePath, $destPath, $size = THUMBNAIL_SIZE)
    {
        if (!file_exists($sourcePath)) {
            throw new Exception('Source file not found');
        }
        
        // Load original
        $source = @imagecreatefrompng($sourcePath);
        if (!$source) {
            throw new Exception('Failed to load source PNG');
        }
        
        $srcWidth = imagesx($source);
        $srcHeight = imagesy($source);
        
        // Calculate thumbnail dimensions (maintain aspect ratio)
        if ($srcWidth > $srcHeight) {
            $thumbWidth = $size;
            $thumbHeight = intval(($srcHeight / $srcWidth) * $size);
        } else {
            $thumbHeight = $size;
            $thumbWidth = intval(($srcWidth / $srcHeight) * $size);
        }
        
        // Create thumbnail canvas
        $thumb = imagecreatetruecolor($thumbWidth, $thumbHeight);
        
        // === PRESERVE TRANSPARENCY ===
        imagealphablending($thumb, false);
        $transparent = imagecolorallocatealpha($thumb, 0, 0, 0, 127);
        imagefill($thumb, 0, 0, $transparent);
        imagesavealpha($thumb, true);
        
        // Resize
        imagecopyresampled(
            $thumb, $source,
            0, 0, 0, 0,
            $thumbWidth, $thumbHeight,
            $srcWidth, $srcHeight
        );
        
        // Save
        $success = imagepng($thumb, $destPath, PNG_COMPRESSION);
        
        imagedestroy($source);
        imagedestroy($thumb);
        
        if (!$success) {
            throw new Exception('Failed to save thumbnail');
        }
        
        return true;
    }
    
    /**
     * Output PNG to browser
     */
    public function output($path, $download = false)
    {
        if (!file_exists($path)) {
            http_response_code(404);
            die('File not found');
        }
        
        // Clear output buffer
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        // Set headers
        header('Content-Type: image/png');
        header('Content-Length: ' . filesize($path));
        
        if ($download) {
            $filename = basename($path);
            header('Content-Disposition: attachment; filename="' . $filename . '"');
        } else {
            header('Content-Disposition: inline');
        }
        
        // Cache headers
        header('Cache-Control: public, max-age=86400');
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 86400) . ' GMT');
        
        // Output file
        readfile($path);
        exit;
    }
    
    /**
     * Delete PNG and all versions
     */
    public function delete($filename, $username = null)
    {
        $username = $username ?: $this->username;
        
        if (!$username) {
            return false;
        }
        
        $paths = [
            USERS_PATH . '/' . $username . '/originals/' . $filename,
            USERS_PATH . '/' . $username . '/optimized/' . $filename,
            USERS_PATH . '/' . $username . '/thumbs/' . $filename
        ];
        
        $deleted = 0;
        foreach ($paths as $path) {
            if (file_exists($path) && unlink($path)) {
                $deleted++;
            }
        }
        
        // Update user storage
        if ($deleted > 0) {
            $userManager = new UserManager();
            $userManager->updateStorageUsed($username);
            $userManager->logActivity($username, 'file_deleted', 'Deleted: ' . $filename);
        }
        
        return $deleted > 0;
    }
    
    /**
     * Rename PNG file
     */
    public function rename($oldFilename, $newFilename, $username = null)
    {
        $username = $username ?: $this->username;
        
        if (!$username) {
            return ['error' => 'User context required'];
        }
        
        // Validate filenames
        $oldFilename = basename($oldFilename);
        if (!preg_match('/\.png$/i', $oldFilename)) {
            return ['error' => 'Invalid old filename'];
        }
        
        // Sanitize new filename
        $newFilename = sanitizeFilename($newFilename);
        if (empty($newFilename)) {
            return ['error' => 'Invalid new filename'];
        }
        
        $newFilename = $newFilename . '.png';
        
        // Check if new filename exists
        $newPath = USERS_PATH . '/' . $username . '/originals/' . $newFilename;
        if (file_exists($newPath)) {
            return ['error' => 'Filename already exists'];
        }
        
        // Rename all versions
        $types = ['originals', 'optimized', 'thumbs'];
        $oldPaths = [];
        $newPaths = [];
        
        foreach ($types as $type) {
            $oldPaths[$type] = USERS_PATH . '/' . $username . '/' . $type . '/' . $oldFilename;
            $newPaths[$type] = USERS_PATH . '/' . $username . '/' . $type . '/' . $newFilename;
            
            if (!file_exists($oldPaths[$type])) {
                return ['error' => ucfirst($type) . ' file not found'];
            }
        }
        
        // Rename all files
        $renamed = 0;
        foreach ($types as $type) {
            if (rename($oldPaths[$type], $newPaths[$type])) {
                $renamed++;
            } else {
                // Rollback on failure
                foreach ($types as $rollbackType) {
                    if (file_exists($newPaths[$rollbackType])) {
                        rename($newPaths[$rollbackType], $oldPaths[$rollbackType]);
                    }
                }
                return ['error' => 'Failed to rename ' . $type];
            }
        }
        
        // Log activity
        $userManager = new UserManager();
        $userManager->logActivity($username, 'file_renamed', "Renamed: $oldFilename → $newFilename");
        
        return ['success' => true, 'new_filename' => $newFilename];
    }
    
    /**
     * Get all images for a user
     */
    public function getAllImages($username = null)
    {
        $username = $username ?: $this->username;
        
        if (!$username) {
            return [];
        }
        
        $originalsPath = USERS_PATH . '/' . $username . '/originals';
        
        if (!is_dir($originalsPath)) {
            return [];
        }
        
        $files = glob($originalsPath . '/*.png');
        $images = [];
        
        foreach ($files as $file) {
            $filename = basename($file);
            $images[] = [
                'filename' => $filename,
                'original' => $file,
                'optimized' => USERS_PATH . '/' . $username . '/optimized/' . $filename,
                'thumb' => USERS_PATH . '/' . $username . '/thumbs/' . $filename,
                'size' => filesize($file),
                'uploaded' => filemtime($file),
                'has_alpha' => $this->hasAlpha($file)
            ];
        }
        
        // Sort by upload time (newest first)
        usort($images, function($a, $b) {
            return $b['uploaded'] - $a['uploaded'];
        });
        
        return $images;
    }
    
    /**
     * Get paginated images
     */
    public function getPaginatedImages($page = 1, $perPage = ITEMS_PER_PAGE, $username = null)
    {
        $allImages = $this->getAllImages($username);
        $total = count($allImages);
        $pages = ceil($total / $perPage);
        
        // Validate page
        $page = max(1, min($page, $pages ?: 1));
        
        // Get slice
        $offset = ($page - 1) * $perPage;
        $images = array_slice($allImages, $offset, $perPage);
        
        return [
            'images' => $images,
            'total' => $total,
            'pages' => $pages,
            'current_page' => $page,
            'per_page' => $perPage,
            'has_prev' => $page > 1,
            'has_next' => $page < $pages
        ];
    }
    
    /**
     * Check if PNG has transparency
     */
    public function hasAlpha($path)
    {
        if (!file_exists($path)) {
            return false;
        }
        
        $img = @imagecreatefrompng($path);
        if (!$img) {
            return false;
        }
        
        // Check for alpha channel
        $width = imagesx($img);
        $height = imagesy($img);
        
        // Sample pixels to detect alpha
        for ($x = 0; $x < $width; $x += max(1, floor($width / 10))) {
            for ($y = 0; $y < $height; $y += max(1, floor($height / 10))) {
                $color = imagecolorat($img, $x, $y);
                $alpha = ($color >> 24) & 0x7F;
                
                if ($alpha > 0) {
                    imagedestroy($img);
                    return true;
                }
            }
        }
        
        imagedestroy($img);
        return false;
    }
    
    /**
     * Get image info
     */
    public function getImageInfo($filename, $username = null)
    {
        $username = $username ?: $this->username;
        
        if (!$username) {
            return null;
        }
        
        $originalPath = USERS_PATH . '/' . $username . '/originals/' . $filename;
        
        if (!file_exists($originalPath)) {
            return null;
        }
        
        $img = @imagecreatefrompng($originalPath);
        if (!$img) {
            return null;
        }
        
        $info = [
            'filename' => $filename,
            'size' => filesize($originalPath),
            'width' => imagesx($img),
            'height' => imagesy($img),
            'has_alpha' => $this->hasAlpha($originalPath),
            'uploaded' => filemtime($originalPath)
        ];
        
        imagedestroy($img);
        
        return $info;
    }
    
    /**
     * Generate unique filename
     */
    private function generateFilename($originalName)
    {
        $ext = pathinfo($originalName, PATHINFO_EXTENSION);
        $name = pathinfo($originalName, PATHINFO_FILENAME);
        
        // Sanitize
        $name = sanitizeFilename($name);
        
        // Add timestamp for uniqueness
        return $name . '_' . time() . '.' . $ext;
    }
    
    /**
     * Get errors
     */
    public function getErrors()
    {
        return $this->errors;
    }
    
    /**
     * Get last error
     */
    public function getLastError()
    {
        return end($this->errors) ?: 'Unknown error';
    }
}
