<?php
/**
 * Authentication Class - Multi-User Support
 */

require_once __DIR__ . '/UserManager.php';

class Auth
{
    private static $userManager;
    
    /**
     * Get UserManager instance
     */
    private static function getUserManager()
    {
        if (!self::$userManager) {
            self::$userManager = new UserManager();
        }
        return self::$userManager;
    }
    
    /**
     * Login user (with IP and User Agent tracking)
     */
    public static function login($username, $password)
    {
        $userManager = self::getUserManager();
        
        if ($userManager->verifyCredentials($username, $password)) {
            session_regenerate_id(true);
            
            $user = $userManager->getUser($username);
            $_SESSION['logged_in'] = true;
            $_SESSION['username'] = $username;
            $_SESSION['display_name'] = $user['display_name'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['login_time'] = time();
            
            // Update last login with IP tracking
            $userManager->updateLastLoginWithTracking($username);
            
            // Log activity
            $userManager->logActivity($username, 'login', 'User logged in');
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Logout user
     */
    public static function logout()
    {
        $_SESSION = [];
        
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time() - 3600, '/');
        }
        
        session_destroy();
    }
    
    /**
     * Check if user is logged in
     */
    public static function check()
    {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }
    
    /**
     * Require login (redirect if not logged in)
     */
    public static function requireLogin()
    {
        if (!self::check()) {
            header('Location: ' . BASE_URL . '/login.php');
            exit;
        }
    }
    
    /**
     * Redirect if already logged in
     */
    public static function redirectIfLoggedIn()
    {
        if (self::check()) {
            header('Location: ' . BASE_URL . '/index.php');
            exit;
        }
    }
    
    /**
     * Check if current user is admin
     */
    public static function isAdmin()
    {
        return self::check() && $_SESSION['role'] === UserManager::ROLE_ADMIN;
    }
    
    /**
     * Require admin role
     */
    public static function requireAdmin()
    {
        self::requireLogin();
        
        if (!self::isAdmin()) {
            http_response_code(403);
            die('Access denied: Admin privileges required');
        }
    }
    
    /**
     * Get current username
     */
    public static function getUsername()
    {
        return $_SESSION['username'] ?? null;
    }
    
    /**
     * Get current user data
     */
    public static function getCurrentUser()
    {
        if (!self::check()) {
            return null;
        }
        
        $userManager = self::getUserManager();
        return $userManager->getUser(self::getUsername());
    }
}
