<?php
/**
 * User Management Class - PRO VERSION
 * Multi-user system dengan role-based access, IP tracking, login history, dan activity log
 */

class UserManager
{
    private $usersFile;
    private $storageDir;
    private $loginHistoryFile;
    private $activityLogFile;
    
    const ROLE_ADMIN = 'admin';
    const ROLE_USER = 'user';
    
    const MAX_LOGIN_HISTORY = 50; // Max login records per user
    const MAX_ACTIVITY_LOG = 100; // Max activity records per user
    
    public function __construct()
    {
        $this->usersFile = BASE_PATH . '/storage/users.json';
        $this->storageDir = STORAGE_PATH . '/users';
        $this->loginHistoryFile = BASE_PATH . '/storage/login_history.json';
        $this->activityLogFile = BASE_PATH . '/storage/activity_log.json';
        
        // Create users directory if not exists
        if (!is_dir($this->storageDir)) {
            mkdir($this->storageDir, 0700, true);
        }
        
        // Initialize files
        if (!file_exists($this->loginHistoryFile)) {
            file_put_contents($this->loginHistoryFile, json_encode([], JSON_PRETTY_PRINT));
        }
        
        if (!file_exists($this->activityLogFile)) {
            file_put_contents($this->activityLogFile, json_encode([], JSON_PRETTY_PRINT));
        }
        
        // Initialize with default admin if empty
        if (!file_exists($this->usersFile)) {
            $this->initializeDefaultUser();
        }
    }
    
    /**
     * Initialize default admin user
     */
    private function initializeDefaultUser()
    {
        $users = [
            'admin' => [
                'username' => 'admin',
                'password_hash' => ADMIN_PASSWORD_HASH,
                'display_name' => 'Administrator',
                'role' => self::ROLE_ADMIN,
                'email' => 'admin@example.com',
                'bio' => 'System Administrator',
                'has_avatar' => false,
                'created_at' => time(),
                'last_login' => null,
                'last_login_ip' => null,
                'last_login_user_agent' => null,
                'total_uploads' => 0,
                'total_logins' => 0,
                'status' => 'active'
            ]
        ];
        
        $this->saveUsers($users);
    }
    
    /**
     * Get all users
     */
    public function getAllUsers()
    {
        if (!file_exists($this->usersFile)) {
            return [];
        }
        
        $data = file_get_contents($this->usersFile);
        return json_decode($data, true) ?: [];
    }
    
    /**
     * Get user by username
     */
    public function getUser($username)
    {
        $users = $this->getAllUsers();
        return $users[$username] ?? null;
    }
    
    /**
     * Save users to file
     */
    private function saveUsers($users)
    {
        return file_put_contents($this->usersFile, json_encode($users, JSON_PRETTY_PRINT));
    }
    
    /**
     * Update user (enhanced with activity logging)
     */
    public function updateUser($username, $data)
    {
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        // Update allowed fields
        $allowedFields = ['display_name', 'email', 'bio', 'role', 'status'];
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $users[$username][$field] = $data[$field];
            }
        }
        
        if ($this->saveUsers($users)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to update user'];
    }
    
    /**
     * Change user password
     */
    public function changePassword($username, $newPassword)
    {
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        if (strlen($newPassword) < 6) {
            return ['error' => 'Password must be at least 6 characters'];
        }
        
        $users[$username]['password_hash'] = password_hash($newPassword, PASSWORD_DEFAULT);
        
        if ($this->saveUsers($users)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to change password'];
    }
    
    /**
     * Delete user
     */
    public function deleteUser($username)
    {
        // Can't delete yourself or default admin
        if ($username === $_SESSION['username'] || $username === 'admin') {
            return ['error' => 'Cannot delete this user'];
        }
        
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        // Delete user data
        unset($users[$username]);
        
        // Delete user directory
        $userDir = $this->storageDir . '/' . $username;
        if (is_dir($userDir)) {
            $this->deleteDirectory($userDir);
        }
        
        if ($this->saveUsers($users)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to delete user'];
    }
    
    /**
     * Verify user credentials
     */
    public function verifyCredentials($username, $password)
    {
        $user = $this->getUser($username);
        
        if (!$user) {
            return false;
        }
        
        if ($user['status'] !== 'active') {
            return false;
        }
        
        return password_verify($password, $user['password_hash']);
    }
    
    /**
     * Update last login
     */
    public function updateLastLogin($username)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            $users[$username]['last_login'] = time();
            $this->saveUsers($users);
        }
    }
    
    /**
     * Increment upload count
     */
    public function incrementUploads($username)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            $users[$username]['total_uploads'] = ($users[$username]['total_uploads'] ?? 0) + 1;
            $this->saveUsers($users);
        }
    }
    
    /**
     * Upload user avatar
     */
    public function uploadAvatar($username, $file)
    {
        // Validate
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            return ['error' => 'Invalid file'];
        }
        
        // Check size (max 2MB)
        if ($file['size'] > 2 * 1024 * 1024) {
            return ['error' => 'File too large (max 2MB)'];
        }
        
        // Check MIME
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, ['image/png', 'image/jpeg', 'image/jpg'])) {
            return ['error' => 'Only PNG/JPG allowed'];
        }
        
        // Load image
        if ($mimeType === 'image/png') {
            $img = imagecreatefrompng($file['tmp_name']);
        } else {
            $img = imagecreatefromjpeg($file['tmp_name']);
        }
        
        if (!$img) {
            return ['error' => 'Failed to load image'];
        }
        
        // Resize to 200x200
        $size = 200;
        $thumb = imagecreatetruecolor($size, $size);
        
        // Preserve transparency
        imagealphablending($thumb, false);
        imagesavealpha($thumb, true);
        $transparent = imagecolorallocatealpha($thumb, 0, 0, 0, 127);
        imagefill($thumb, 0, 0, $transparent);
        
        // Get dimensions
        $width = imagesx($img);
        $height = imagesy($img);
        
        // Calculate crop
        $cropSize = min($width, $height);
        $x = (int) round(($width - $cropSize) / 2);
$y = (int) round(($height - $cropSize) / 2);

        
        // Resize
        imagecopyresampled($thumb, $img, 0, 0, $x, $y, $size, $size, $cropSize, $cropSize);
        
        // Save
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        $userDir = dirname($avatarPath);
        
        if (!is_dir($userDir)) {
            mkdir($userDir, 0700, true);
        }
        
        $success = imagepng($thumb, $avatarPath, 9);
        
        imagedestroy($img);
        imagedestroy($thumb);
        
        if ($success) {
            // Update user
            $users = $this->getAllUsers();
            if (isset($users[$username])) {
                $users[$username]['has_avatar'] = true;
                $this->saveUsers($users);
            }
            
            return ['success' => true];
        }
        
        return ['error' => 'Failed to save avatar'];
    }
    
    /**
     * Get avatar URL
     */
    public function getAvatarUrl($username)
    {
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        
        if (file_exists($avatarPath)) {
            return BASE_URL . '/user-avatar.php?user=' . urlencode($username) . '&t=' . filemtime($avatarPath);
        }
        
        return null;
    }
    
    /**
     * Delete avatar
     */
    public function deleteAvatar($username)
    {
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        
        if (file_exists($avatarPath)) {
            if (unlink($avatarPath)) {
                $users = $this->getAllUsers();
                if (isset($users[$username])) {
                    $users[$username]['has_avatar'] = false;
                    $this->saveUsers($users);
                }
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check if user is admin
     */
    public function isAdmin($username)
    {
        $user = $this->getUser($username);
        return $user && $user['role'] === self::ROLE_ADMIN;
    }
    
    /**
     * Get user stats
     */
    public function getUserStats()
    {
        $users = $this->getAllUsers();
        
        return [
            'total' => count($users),
            'admins' => count(array_filter($users, fn($u) => $u['role'] === self::ROLE_ADMIN)),
            'users' => count(array_filter($users, fn($u) => $u['role'] === self::ROLE_USER)),
            'active' => count(array_filter($users, fn($u) => $u['status'] === 'active'))
        ];
    }
    
    /**
     * ===== NEW FEATURES - IP TRACKING & LOGIN HISTORY =====
     */
    
    /**
     * Update last login with IP and User Agent tracking
     */
    public function updateLastLoginWithTracking($username, $ip = null, $userAgent = null)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            $users[$username]['last_login'] = time();
            $users[$username]['last_login_ip'] = $ip ?: $this->getClientIP();
            $users[$username]['last_login_user_agent'] = $userAgent ?: $this->getUserAgent();
            $users[$username]['total_logins'] = ($users[$username]['total_logins'] ?? 0) + 1;
            $this->saveUsers($users);
            
            // Add to login history
            $this->addLoginHistory($username, $ip, $userAgent);
        }
    }
    
    /**
     * Get client IP address
     */
    private function getClientIP()
    {
        $ip = 'Unknown';
        
        if (isset($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            $ip = $_SERVER['HTTP_CF_CONNECTING_IP']; // CloudFlare
        } elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } elseif (isset($_SERVER['HTTP_X_REAL_IP'])) {
            $ip = $_SERVER['HTTP_X_REAL_IP'];
        } elseif (isset($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        
        return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : 'Unknown';
    }
    
    /**
     * Get user agent
     */
    private function getUserAgent()
    {
        return $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    }
    
    /**
     * Add login to history
     */
    private function addLoginHistory($username, $ip = null, $userAgent = null)
    {
        $history = $this->getLoginHistory();
        
        if (!isset($history[$username])) {
            $history[$username] = [];
        }
        
        // Add new login
        array_unshift($history[$username], [
            'timestamp' => time(),
            'ip' => $ip ?: $this->getClientIP(),
            'user_agent' => $userAgent ?: $this->getUserAgent(),
            'browser' => $this->parseBrowser($userAgent ?: $this->getUserAgent()),
            'os' => $this->parseOS($userAgent ?: $this->getUserAgent())
        ]);
        
        // Keep only last N logins
        $history[$username] = array_slice($history[$username], 0, self::MAX_LOGIN_HISTORY);
        
        file_put_contents($this->loginHistoryFile, json_encode($history, JSON_PRETTY_PRINT));
    }
    
    /**
     * Get login history for a user
     */
    public function getLoginHistory($username = null, $limit = 10)
    {
        if (!file_exists($this->loginHistoryFile)) {
            return $username ? [] : [];
        }
        
        $history = json_decode(file_get_contents($this->loginHistoryFile), true) ?: [];
        
        if ($username) {
            $userHistory = $history[$username] ?? [];
            return array_slice($userHistory, 0, $limit);
        }
        
        return $history;
    }
    
    /**
     * Parse browser from user agent
     */
    private function parseBrowser($userAgent)
    {
        if (preg_match('/MSIE|Trident/i', $userAgent)) return 'Internet Explorer';
        if (preg_match('/Edge/i', $userAgent)) return 'Microsoft Edge';
        if (preg_match('/Chrome/i', $userAgent)) return 'Google Chrome';
        if (preg_match('/Safari/i', $userAgent)) return 'Safari';
        if (preg_match('/Firefox/i', $userAgent)) return 'Mozilla Firefox';
        if (preg_match('/Opera/i', $userAgent)) return 'Opera';
        return 'Unknown Browser';
    }
    
    /**
     * Parse OS from user agent
     */
    private function parseOS($userAgent)
    {
        if (preg_match('/Windows NT 10/i', $userAgent)) return 'Windows 10';
        if (preg_match('/Windows NT 11/i', $userAgent)) return 'Windows 11';
        if (preg_match('/Windows/i', $userAgent)) return 'Windows';
        if (preg_match('/Mac OS X/i', $userAgent)) return 'macOS';
        if (preg_match('/Linux/i', $userAgent)) return 'Linux';
        if (preg_match('/Android/i', $userAgent)) return 'Android';
        if (preg_match('/iOS|iPhone|iPad/i', $userAgent)) return 'iOS';
        return 'Unknown OS';
    }
    
    /**
     * ===== CHANGE USERNAME FEATURE =====
     */
    
    /**
     * Change username (with data migration)
     */
    public function changeUsername($oldUsername, $newUsername)
    {
        // Validate new username
        if (empty($newUsername)) {
            return ['error' => 'Username cannot be empty'];
        }
        
        if (!preg_match('/^[a-zA-Z0-9_-]{3,20}$/', $newUsername)) {
            return ['error' => 'Username: 3-20 chars, letters/numbers/underscore/dash only'];
        }
        
        // Cannot change admin username
        if ($oldUsername === 'admin') {
            return ['error' => 'Cannot change admin username'];
        }
        
        // Check if old user exists
        $users = $this->getAllUsers();
        if (!isset($users[$oldUsername])) {
            return ['error' => 'User not found'];
        }
        
        // Check if new username already exists
        if (isset($users[$newUsername])) {
            return ['error' => 'Username already taken'];
        }
        
        // Get user data
        $userData = $users[$oldUsername];
        $userData['username'] = $newUsername;
        
        // Remove old username, add new
        unset($users[$oldUsername]);
        $users[$newUsername] = $userData;
        
        // Save users
        if (!$this->saveUsers($users)) {
            return ['error' => 'Failed to update username'];
        }
        
        // Rename user directory
        $oldDir = $this->storageDir . '/' . $oldUsername;
        $newDir = $this->storageDir . '/' . $newUsername;
        
        if (is_dir($oldDir)) {
            if (!rename($oldDir, $newDir)) {
                return ['error' => 'Failed to rename user directory'];
            }
        }
        
        // Migrate login history
        $this->migrateLoginHistory($oldUsername, $newUsername);
        
        // Migrate activity log
        $this->migrateActivityLog($oldUsername, $newUsername);
        
        // Log activity
        $this->logActivity($newUsername, 'username_changed', "Changed username from $oldUsername to $newUsername");
        
        return ['success' => true, 'new_username' => $newUsername];
    }
    
    /**
     * Migrate login history after username change
     */
    private function migrateLoginHistory($oldUsername, $newUsername)
    {
        if (!file_exists($this->loginHistoryFile)) return;
        
        $history = json_decode(file_get_contents($this->loginHistoryFile), true) ?: [];
        
        if (isset($history[$oldUsername])) {
            $history[$newUsername] = $history[$oldUsername];
            unset($history[$oldUsername]);
            file_put_contents($this->loginHistoryFile, json_encode($history, JSON_PRETTY_PRINT));
        }
    }
    
    /**
     * Migrate activity log after username change
     */
    private function migrateActivityLog($oldUsername, $newUsername)
    {
        if (!file_exists($this->activityLogFile)) return;
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        
        if (isset($log[$oldUsername])) {
            $log[$newUsername] = $log[$oldUsername];
            unset($log[$oldUsername]);
            file_put_contents($this->activityLogFile, json_encode($log, JSON_PRETTY_PRINT));
        }
    }
    
    /**
     * ===== ACTIVITY LOG FEATURE =====
     */
    
    /**
     * Log user activity
     */
    public function logActivity($username, $action, $description = '', $metadata = [])
    {
        if (!file_exists($this->activityLogFile)) {
            file_put_contents($this->activityLogFile, json_encode([], JSON_PRETTY_PRINT));
        }
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        
        if (!isset($log[$username])) {
            $log[$username] = [];
        }
        
        // Add activity
        array_unshift($log[$username], [
            'timestamp' => time(),
            'action' => $action,
            'description' => $description,
            'ip' => $this->getClientIP(),
            'metadata' => $metadata
        ]);
        
        // Keep only last N activities
        $log[$username] = array_slice($log[$username], 0, self::MAX_ACTIVITY_LOG);
        
        file_put_contents($this->activityLogFile, json_encode($log, JSON_PRETTY_PRINT));
    }
    
    /**
     * Get activity log for a user
     */
    public function getActivityLog($username, $limit = 20)
    {
        if (!file_exists($this->activityLogFile)) {
            return [];
        }
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        $userLog = $log[$username] ?? [];
        
        return array_slice($userLog, 0, $limit);
    }
    
    /**
     * Get all activity logs (admin only)
     */
    public function getAllActivityLogs($limit = 50)
    {
        if (!file_exists($this->activityLogFile)) {
            return [];
        }
        
        $log = json_decode(file_get_contents($this->activityLogFile), true) ?: [];
        
        // Flatten all logs and sort by timestamp
        $allLogs = [];
        foreach ($log as $username => $userLogs) {
            foreach ($userLogs as $activity) {
                $activity['username'] = $username;
                $allLogs[] = $activity;
            }
        }
        
        // Sort by timestamp desc
        usort($allLogs, function($a, $b) {
            return $b['timestamp'] - $a['timestamp'];
        });
        
        return array_slice($allLogs, 0, $limit);
    }
    
    /**
     * ===== ENHANCED USER CREATION =====
     */
    
    /**
     * Create new user (enhanced with tracking)
     */
    public function createUser($username, $password, $displayName, $email, $role = self::ROLE_USER)
    {
        // Validate
        if (empty($username) || empty($password)) {
            return ['error' => 'Username and password required'];
        }
        
        // Check username format
        if (!preg_match('/^[a-zA-Z0-9_-]{3,20}$/', $username)) {
            return ['error' => 'Username: 3-20 chars, letters/numbers/underscore/dash only'];
        }
        
        // Check if exists
        if ($this->getUser($username)) {
            return ['error' => 'Username already exists'];
        }
        
        // Validate email
        if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['error' => 'Invalid email address'];
        }
        
        // Create user
        $users = $this->getAllUsers();
        $users[$username] = [
            'username' => $username,
            'password_hash' => password_hash($password, PASSWORD_DEFAULT),
            'display_name' => $displayName ?: $username,
            'role' => $role,
            'email' => $email,
            'bio' => '',
            'has_avatar' => false,
            'created_at' => time(),
            'last_login' => null,
            'last_login_ip' => null,
            'last_login_user_agent' => null,
            'total_uploads' => 0,
            'total_logins' => 0,
            'status' => 'active'
        ];
        
        // Create user directory
        $userDir = $this->storageDir . '/' . $username;
        if (!is_dir($userDir)) {
            mkdir($userDir, 0700, true);
        }
        
        if ($this->saveUsers($users)) {
            // Log activity
            $this->logActivity($username, 'user_created', "User account created");
            
            return ['success' => true, 'username' => $username];
        }
        
        return ['error' => 'Failed to create user'];
    }
    
    /**
     * Delete directory recursively
     */
    private function deleteDirectory($dir)
    {
        if (!is_dir($dir)) {
            return false;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->deleteDirectory($path) : unlink($path);
        }
        
        return rmdir($dir);
    }
}
