<?php
/**
 * Profile Management Class - UPGRADED
 * Fully integrated dengan UserManager
 */

require_once __DIR__ . '/UserManager.php';

class Profile
{
    private $userManager;
    private $username;
    
    public function __construct()
    {
        $this->userManager = new UserManager();
        $this->username = $_SESSION['username'] ?? null;
    }
    
    /**
     * Get current user profile data
     */
    public function getData()
    {
        if (!$this->username) {
            return null;
        }
        
        return $this->userManager->getUser($this->username);
    }
    
    /**
     * Update display name
     */
    public function updateDisplayName($name)
    {
        if (!$this->username) {
            return false;
        }
        
        $result = $this->userManager->updateUser($this->username, [
            'display_name' => trim($name)
        ]);
        
        if (isset($result['success'])) {
            $_SESSION['display_name'] = trim($name);
            $this->userManager->logActivity($this->username, 'profile_updated', 'Updated display name');
            return true;
        }
        
        return false;
    }
    
    /**
     * Update bio
     */
    public function updateBio($bio)
    {
        if (!$this->username) {
            return false;
        }
        
        $result = $this->userManager->updateUser($this->username, [
            'bio' => trim($bio)
        ]);
        
        if (isset($result['success'])) {
            $this->userManager->logActivity($this->username, 'profile_updated', 'Updated bio');
            return true;
        }
        
        return false;
    }
    
    /**
     * Update email
     */
    public function updateEmail($email)
    {
        if (!$this->username) {
            return ['error' => 'User not logged in'];
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['error' => 'Invalid email address'];
        }
        
        $result = $this->userManager->updateUser($this->username, [
            'email' => trim($email)
        ]);
        
        if (isset($result['success'])) {
            $this->userManager->logActivity($this->username, 'profile_updated', 'Updated email');
            return ['success' => true];
        }
        
        return ['error' => 'Failed to update email'];
    }
    
    /**
     * Change password
     */
    public function changePassword($oldPassword, $newPassword)
    {
        if (!$this->username) {
            return ['error' => 'User not logged in'];
        }
        
        // Verify old password
        if (!$this->userManager->verifyCredentials($this->username, $oldPassword)) {
            return ['error' => 'Current password is incorrect'];
        }
        
        // Validate new password
        if (strlen($newPassword) < 6) {
            return ['error' => 'Password must be at least 6 characters'];
        }
        
        // Change password
        $result = $this->userManager->changePassword($this->username, $newPassword);
        
        if (isset($result['success'])) {
            $this->userManager->logActivity($this->username, 'password_changed', 'Password changed successfully');
            return ['success' => true];
        }
        
        return $result;
    }
    
    /**
     * Upload avatar
     */
    public function uploadAvatar($file)
    {
        if (!$this->username) {
            return ['error' => 'User not logged in'];
        }
        
        $result = $this->userManager->uploadAvatar($this->username, $file);
        
        if (isset($result['success'])) {
            $this->userManager->logActivity($this->username, 'avatar_uploaded', 'Profile picture updated');
        }
        
        return $result;
    }
    
    /**
     * Get avatar URL
     */
    public function getAvatarUrl()
    {
        if (!$this->username) {
            return null;
        }
        
        return $this->userManager->getAvatarUrl($this->username);
    }
    
    /**
     * Delete avatar
     */
    public function deleteAvatar()
    {
        if (!$this->username) {
            return false;
        }
        
        $result = $this->userManager->deleteAvatar($this->username);
        
        if ($result) {
            $this->userManager->logActivity($this->username, 'avatar_deleted', 'Profile picture removed');
        }
        
        return $result;
    }
}
