<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/csrf.php';
require_once __DIR__ . '/../includes/Profile.php';

Auth::requireLogin();

$profile = new Profile();
$profileData = $profile->getData();
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    CSRF::verifyOrDie();
    
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'update_profile':
            $profile->updateDisplayName($_POST['display_name'] ?? '');
            $profile->updateBio($_POST['bio'] ?? '');
            $message = 'Profile updated successfully!';
            $profileData = $profile->getData();
            break;
            
        case 'change_password':
            $result = $profile->changePassword($_POST['old_password'] ?? '', $_POST['new_password'] ?? '');
            if (isset($result['success'])) {
                $message = 'Password changed successfully!';
            } else {
                $error = $result['error'];
            }
            break;
            
        case 'upload_avatar':
            if (isset($_FILES['avatar'])) {
                $result = $profile->uploadAvatar($_FILES['avatar']);
                if (isset($result['success'])) {
                    $message = 'Avatar uploaded successfully!';
                    $profileData = $profile->getData();
                } else {
                    $error = $result['error'];
                }
            }
            break;
            
        case 'delete_avatar':
            $profile->deleteAvatar();
            $message = 'Avatar deleted successfully!';
            $profileData = $profile->getData();
            break;
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - PNG Gallery Ultra</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif;
            background: linear-gradient(180deg, #FAF5FF 0%, #F5F3FF 100%);
            min-height: 100vh;
            padding: 2rem;
        }
        
        [data-theme="dark"] body {
            background: linear-gradient(180deg, #0F0B1F 0%, #1E1B4B 100%);
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
        }
        
        .back-btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem 1.5rem;
            background: white;
            border-radius: 12px;
            text-decoration: none;
            color: #374151;
            font-weight: 600;
            margin-bottom: 2rem;
            transition: all 0.3s ease;
            border: 1px solid #E5E7EB;
        }
        
        .back-btn:hover {
            transform: translateX(-4px);
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        
        .card {
            background: white;
            border-radius: 20px;
            padding: 2rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            border: 1px solid #E5E7EB;
        }
        
        [data-theme="dark"] .card {
            background: rgba(49, 46, 129, 0.5);
            border-color: #4C1D95;
        }
        
        h1 {
            font-size: 2rem;
            margin-bottom: 0.5rem;
            background: linear-gradient(135deg, #8B5CF6 0%, #6D28D9 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        h2 {
            font-size: 1.25rem;
            margin-bottom: 1.5rem;
            color: #374151;
        }
        
        .avatar-section {
            display: flex;
            align-items: center;
            gap: 2rem;
            margin-bottom: 2rem;
        }
        
        .avatar {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            object-fit: cover;
            border: 4px solid #8B5CF6;
            box-shadow: 0 4px 6px rgba(139, 92, 246, 0.3);
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #374151;
        }
        
        input[type="text"],
        input[type="password"],
        textarea {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 2px solid #E5E7EB;
            border-radius: 12px;
            font-size: 1rem;
            transition: border-color 0.3s ease;
        }
        
        input:focus,
        textarea:focus {
            outline: none;
            border-color: #8B5CF6;
            box-shadow: 0 0 0 4px rgba(139, 92, 246, 0.1);
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 12px;
            font-weight: 600;
            cursor: pointer;
            border: none;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #8B5CF6 0%, #6D28D9 100%);
            color: white;
        }
        
        .btn-danger {
            background: #EF4444;
            color: white;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px rgba(0,0,0,0.2);
        }
        
        .alert {
            padding: 1rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
        }
        
        .alert-success {
            background: #D1FAE5;
            color: #065F46;
        }
        
        .alert-error {
            background: #FEE2E2;
            color: #991B1B;
        }
    </style>
</head>
<body>
    <div class="container">
        <a href="index.php" class="back-btn">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M19 12H5M12 19l-7-7 7-7"/>
            </svg>
            Back to Gallery
        </a>
        
        <h1>⚙️ Settings</h1>
        <p style="color: #6B7280; margin-bottom: 2rem;">Manage your profile and account settings</p>
        
        <?php if ($message): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="card">
            <h2>Profile Picture</h2>
            <div class="avatar-section">
                <img src="profile-avatar.php" alt="Avatar" class="avatar" id="avatar-preview">
                <div>
                    <form method="POST" enctype="multipart/form-data" id="avatar-form">
                        <?php echo CSRF::field(); ?>
                        <input type="hidden" name="action" value="upload_avatar">
                        <input type="file" name="avatar" id="avatar-input" accept="image/png,image/jpeg" style="display: none;">
                        <button type="button" onclick="document.getElementById('avatar-input').click()" class="btn btn-primary">Upload Photo</button>
                        <?php if ($profileData['has_avatar']): ?>
                            <button type="button" onclick="deleteAvatar()" class="btn btn-danger">Delete</button>
                        <?php endif; ?>
                    </form>
                    <p style="color: #6B7280; font-size: 0.875rem; margin-top: 0.5rem;">PNG or JPG, max 2MB</p>
                </div>
            </div>
        </div>
        
        <div class="card">
            <h2>Profile Information</h2>
            <form method="POST">
                <?php echo CSRF::field(); ?>
                <input type="hidden" name="action" value="update_profile">
                
                <div class="form-group">
                    <label>Display Name</label>
                    <input type="text" name="display_name" value="<?php echo htmlspecialchars($profileData['display_name']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label>Bio</label>
                    <textarea name="bio" rows="3"><?php echo htmlspecialchars($profileData['bio']); ?></textarea>
                </div>
                
                <button type="submit" class="btn btn-primary">Save Changes</button>
            </form>
        </div>
        
        <div class="card">
            <h2>Change Password</h2>
            <form method="POST">
                <?php echo CSRF::field(); ?>
                <input type="hidden" name="action" value="change_password">
                
                <div class="form-group">
                    <label>Current Password</label>
                    <input type="password" name="old_password" required>
                </div>
                
                <div class="form-group">
                    <label>New Password</label>
                    <input type="password" name="new_password" minlength="6" required>
                </div>
                
                <button type="submit" class="btn btn-primary">Change Password</button>
            </form>
        </div>
    </div>
    
    <script>
        document.getElementById('avatar-input').addEventListener('change', function() {
            if (this.files.length > 0) {
                document.getElementById('avatar-form').submit();
            }
        });
        
        function deleteAvatar() {
            Swal.fire({
                title: 'Delete avatar?',
                text: 'This action cannot be undone',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#EF4444',
                confirmButtonText: 'Delete'
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = '<?php echo CSRF::field(); ?><input type="hidden" name="action" value="delete_avatar">';
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }
        
        // Theme
        const savedTheme = localStorage.getItem('theme') || 'light';
        document.documentElement.setAttribute('data-theme', savedTheme);
    </script>
</body>
</html>
