<?php
/**
 * User Management Class
 * Multi-user system dengan role-based access
 */

class UserManager
{
    private $usersFile;
    private $storageDir;
    
    const ROLE_ADMIN = 'admin';
    const ROLE_USER = 'user';
    
    public function __construct()
    {
        $this->usersFile = BASE_PATH . '/storage/users.json';
        $this->storageDir = STORAGE_PATH . '/users';
        
        // Create users directory if not exists
        if (!is_dir($this->storageDir)) {
            mkdir($this->storageDir, 0700, true);
        }
        
        // Initialize with default admin if empty
        if (!file_exists($this->usersFile)) {
            $this->initializeDefaultUser();
        }
    }
    
    /**
     * Initialize default admin user
     */
    private function initializeDefaultUser()
    {
        $users = [
            'admin' => [
                'username' => 'admin',
                'password_hash' => ADMIN_PASSWORD_HASH,
                'display_name' => 'Administrator',
                'role' => self::ROLE_ADMIN,
                'email' => 'admin@example.com',
                'bio' => 'System Administrator',
                'has_avatar' => false,
                'created_at' => time(),
                'last_login' => null,
                'total_uploads' => 0,
                'status' => 'active'
            ]
        ];
        
        $this->saveUsers($users);
    }
    
    /**
     * Get all users
     */
    public function getAllUsers()
    {
        if (!file_exists($this->usersFile)) {
            return [];
        }
        
        $data = file_get_contents($this->usersFile);
        return json_decode($data, true) ?: [];
    }
    
    /**
     * Get user by username
     */
    public function getUser($username)
    {
        $users = $this->getAllUsers();
        return $users[$username] ?? null;
    }
    
    /**
     * Save users to file
     */
    private function saveUsers($users)
    {
        return file_put_contents($this->usersFile, json_encode($users, JSON_PRETTY_PRINT));
    }
    
    /**
     * Create new user
     */
    public function createUser($username, $password, $displayName, $email, $role = self::ROLE_USER)
    {
        // Validate
        if (empty($username) || empty($password)) {
            return ['error' => 'Username and password required'];
        }
        
        // Check username format
        if (!preg_match('/^[a-zA-Z0-9_-]{3,20}$/', $username)) {
            return ['error' => 'Username: 3-20 chars, letters/numbers/underscore/dash only'];
        }
        
        // Check if exists
        if ($this->getUser($username)) {
            return ['error' => 'Username already exists'];
        }
        
        // Validate email
        if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['error' => 'Invalid email address'];
        }
        
        // Create user
        $users = $this->getAllUsers();
        $users[$username] = [
            'username' => $username,
            'password_hash' => password_hash($password, PASSWORD_DEFAULT),
            'display_name' => $displayName ?: $username,
            'role' => $role,
            'email' => $email,
            'bio' => '',
            'has_avatar' => false,
            'created_at' => time(),
            'last_login' => null,
            'total_uploads' => 0,
            'status' => 'active'
        ];
        
        // Create user directory
        $userDir = $this->storageDir . '/' . $username;
        if (!is_dir($userDir)) {
            mkdir($userDir, 0700, true);
        }
        
        if ($this->saveUsers($users)) {
            return ['success' => true, 'username' => $username];
        }
        
        return ['error' => 'Failed to create user'];
    }
    
    /**
     * Update user
     */
    public function updateUser($username, $data)
    {
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        // Update allowed fields
        $allowedFields = ['display_name', 'email', 'bio', 'role', 'status'];
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $users[$username][$field] = $data[$field];
            }
        }
        
        if ($this->saveUsers($users)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to update user'];
    }
    
    /**
     * Change user password
     */
    public function changePassword($username, $newPassword)
    {
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        if (strlen($newPassword) < 6) {
            return ['error' => 'Password must be at least 6 characters'];
        }
        
        $users[$username]['password_hash'] = password_hash($newPassword, PASSWORD_DEFAULT);
        
        if ($this->saveUsers($users)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to change password'];
    }
    
    /**
     * Delete user
     */
    public function deleteUser($username)
    {
        // Can't delete yourself or default admin
        if ($username === $_SESSION['username'] || $username === 'admin') {
            return ['error' => 'Cannot delete this user'];
        }
        
        $users = $this->getAllUsers();
        
        if (!isset($users[$username])) {
            return ['error' => 'User not found'];
        }
        
        // Delete user data
        unset($users[$username]);
        
        // Delete user directory
        $userDir = $this->storageDir . '/' . $username;
        if (is_dir($userDir)) {
            $this->deleteDirectory($userDir);
        }
        
        if ($this->saveUsers($users)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to delete user'];
    }
    
    /**
     * Verify user credentials
     */
    public function verifyCredentials($username, $password)
    {
        $user = $this->getUser($username);
        
        if (!$user) {
            return false;
        }
        
        if ($user['status'] !== 'active') {
            return false;
        }
        
        return password_verify($password, $user['password_hash']);
    }
    
    /**
     * Update last login
     */
    public function updateLastLogin($username)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            $users[$username]['last_login'] = time();
            $this->saveUsers($users);
        }
    }
    
    /**
     * Increment upload count
     */
    public function incrementUploads($username)
    {
        $users = $this->getAllUsers();
        
        if (isset($users[$username])) {
            $users[$username]['total_uploads'] = ($users[$username]['total_uploads'] ?? 0) + 1;
            $this->saveUsers($users);
        }
    }
    
    /**
     * Upload user avatar
     */
    public function uploadAvatar($username, $file)
    {
        // Validate
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            return ['error' => 'Invalid file'];
        }
        
        // Check size (max 2MB)
        if ($file['size'] > 2 * 1024 * 1024) {
            return ['error' => 'File too large (max 2MB)'];
        }
        
        // Check MIME
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, ['image/png', 'image/jpeg', 'image/jpg'])) {
            return ['error' => 'Only PNG/JPG allowed'];
        }
        
        // Load image
        if ($mimeType === 'image/png') {
            $img = imagecreatefrompng($file['tmp_name']);
        } else {
            $img = imagecreatefromjpeg($file['tmp_name']);
        }
        
        if (!$img) {
            return ['error' => 'Failed to load image'];
        }
        
        // Resize to 200x200
        $size = 200;
        $thumb = imagecreatetruecolor($size, $size);
        
        // Preserve transparency
        imagealphablending($thumb, false);
        imagesavealpha($thumb, true);
        $transparent = imagecolorallocatealpha($thumb, 0, 0, 0, 127);
        imagefill($thumb, 0, 0, $transparent);
        
        // Get dimensions
        $width = imagesx($img);
        $height = imagesy($img);
        
        // Calculate crop
        $cropSize = min($width, $height);
        $x = ($width - $cropSize) / 2;
        $y = ($height - $cropSize) / 2;
        
        // Resize
        imagecopyresampled($thumb, $img, 0, 0, $x, $y, $size, $size, $cropSize, $cropSize);
        
        // Save
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        $userDir = dirname($avatarPath);
        
        if (!is_dir($userDir)) {
            mkdir($userDir, 0700, true);
        }
        
        $success = imagepng($thumb, $avatarPath, 9);
        
        imagedestroy($img);
        imagedestroy($thumb);
        
        if ($success) {
            // Update user
            $users = $this->getAllUsers();
            if (isset($users[$username])) {
                $users[$username]['has_avatar'] = true;
                $this->saveUsers($users);
            }
            
            return ['success' => true];
        }
        
        return ['error' => 'Failed to save avatar'];
    }
    
    /**
     * Get avatar URL
     */
    public function getAvatarUrl($username)
    {
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        
        if (file_exists($avatarPath)) {
            return BASE_URL . '/user-avatar.php?user=' . urlencode($username) . '&t=' . filemtime($avatarPath);
        }
        
        return null;
    }
    
    /**
     * Delete avatar
     */
    public function deleteAvatar($username)
    {
        $avatarPath = $this->storageDir . '/' . $username . '/avatar.png';
        
        if (file_exists($avatarPath)) {
            if (unlink($avatarPath)) {
                $users = $this->getAllUsers();
                if (isset($users[$username])) {
                    $users[$username]['has_avatar'] = false;
                    $this->saveUsers($users);
                }
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check if user is admin
     */
    public function isAdmin($username)
    {
        $user = $this->getUser($username);
        return $user && $user['role'] === self::ROLE_ADMIN;
    }
    
    /**
     * Get user stats
     */
    public function getUserStats()
    {
        $users = $this->getAllUsers();
        
        return [
            'total' => count($users),
            'admins' => count(array_filter($users, fn($u) => $u['role'] === self::ROLE_ADMIN)),
            'users' => count(array_filter($users, fn($u) => $u['role'] === self::ROLE_USER)),
            'active' => count(array_filter($users, fn($u) => $u['status'] === 'active'))
        ];
    }
    
    /**
     * Delete directory recursively
     */
    private function deleteDirectory($dir)
    {
        if (!is_dir($dir)) {
            return false;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->deleteDirectory($path) : unlink($path);
        }
        
        return rmdir($dir);
    }
}
