<?php
/**
 * Profile Management Class
 * Handles user profile, avatar upload, username & password changes
 */

class Profile
{
    private $profileFile;
    private $avatarPath;
    
    public function __construct()
    {
        $this->profileFile = BASE_PATH . '/storage/profile.json';
        $this->avatarPath = STORAGE_PATH . '/avatar.png';
    }
    
    /**
     * Get user profile data
     */
    public function getData()
    {
        if (file_exists($this->profileFile)) {
            $data = json_decode(file_get_contents($this->profileFile), true);
            return $data ?: $this->getDefaultData();
        }
        return $this->getDefaultData();
    }
    
    /**
     * Get default profile data
     */
    private function getDefaultData()
    {
        return [
            'username' => ADMIN_USERNAME,
            'display_name' => 'Administrator',
            'bio' => 'PNG Gallery Manager',
            'has_avatar' => file_exists($this->avatarPath),
            'joined' => time(),
            'total_uploads' => 0,
            'last_login' => time()
        ];
    }
    
    /**
     * Save profile data
     */
    public function save($data)
    {
        return file_put_contents($this->profileFile, json_encode($data, JSON_PRETTY_PRINT));
    }
    
    /**
     * Update display name
     */
    public function updateDisplayName($name)
    {
        $data = $this->getData();
        $data['display_name'] = trim($name);
        return $this->save($data);
    }
    
    /**
     * Update bio
     */
    public function updateBio($bio)
    {
        $data = $this->getData();
        $data['bio'] = trim($bio);
        return $this->save($data);
    }
    
    /**
     * Change password
     */
    public function changePassword($oldPassword, $newPassword)
    {
        // Verify old password
        if (!password_verify($oldPassword, ADMIN_PASSWORD_HASH)) {
            return ['error' => 'Current password is incorrect'];
        }
        
        // Generate new hash
        $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
        
        // Update config file
        $configPath = BASE_PATH . '/includes/config.php';
        $config = file_get_contents($configPath);
        
        // Replace password hash
        $config = preg_replace(
            "/define\('ADMIN_PASSWORD_HASH', '.*?'\);/",
            "define('ADMIN_PASSWORD_HASH', '$newHash');",
            $config
        );
        
        if (file_put_contents($configPath, $config)) {
            return ['success' => true];
        }
        
        return ['error' => 'Failed to update password'];
    }
    
    /**
     * Upload avatar
     */
    public function uploadAvatar($file)
    {
        // Validate
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            return ['error' => 'Invalid file'];
        }
        
        // Check size (max 2MB)
        if ($file['size'] > 2 * 1024 * 1024) {
            return ['error' => 'File too large (max 2MB)'];
        }
        
        // Check MIME
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, ['image/png', 'image/jpeg', 'image/jpg'])) {
            return ['error' => 'Only PNG/JPG allowed'];
        }
        
        // Load image
        if ($mimeType === 'image/png') {
            $img = imagecreatefrompng($file['tmp_name']);
        } else {
            $img = imagecreatefromjpeg($file['tmp_name']);
        }
        
        if (!$img) {
            return ['error' => 'Failed to load image'];
        }
        
        // Resize to 200x200
        $size = 200;
        $thumb = imagecreatetruecolor($size, $size);
        
        // Preserve transparency
        imagealphablending($thumb, false);
        imagesavealpha($thumb, true);
        $transparent = imagecolorallocatealpha($thumb, 0, 0, 0, 127);
        imagefill($thumb, 0, 0, $transparent);
        
        // Get original dimensions
        $width = imagesx($img);
        $height = imagesy($img);
        
        // Calculate crop to square
        $cropSize = min($width, $height);
        $x = ($width - $cropSize) / 2;
        $y = ($height - $cropSize) / 2;
        
        // Resize
        imagecopyresampled(
            $thumb, $img,
            0, 0, $x, $y,
            $size, $size,
            $cropSize, $cropSize
        );
        
        // Save as PNG
        $success = imagepng($thumb, $this->avatarPath, 9);
        
        imagedestroy($img);
        imagedestroy($thumb);
        
        if ($success) {
            // Update profile
            $data = $this->getData();
            $data['has_avatar'] = true;
            $this->save($data);
            
            return ['success' => true];
        }
        
        return ['error' => 'Failed to save avatar'];
    }
    
    /**
     * Get avatar URL
     */
    public function getAvatarUrl()
    {
        if (file_exists($this->avatarPath)) {
            return BASE_URL . '/profile-avatar.php?t=' . filemtime($this->avatarPath);
        }
        return null;
    }
    
    /**
     * Delete avatar
     */
    public function deleteAvatar()
    {
        if (file_exists($this->avatarPath)) {
            if (unlink($this->avatarPath)) {
                $data = $this->getData();
                $data['has_avatar'] = false;
                $this->save($data);
                return true;
            }
        }
        return false;
    }
    
    /**
     * Increment upload count
     */
    public function incrementUploads()
    {
        $data = $this->getData();
        $data['total_uploads'] = ($data['total_uploads'] ?? 0) + 1;
        $this->save($data);
    }
    
    /**
     * Update last login
     */
    public function updateLastLogin()
    {
        $data = $this->getData();
        $data['last_login'] = time();
        $this->save($data);
    }
}
