<?php
/**
 * PNG Handler Class
 * 
 * Kelas ini FOKUS pada mempertahankan transparansi PNG (alpha channel)
 * di seluruh proses: upload, optimize, thumbnail, output.
 * 
 * CRITICAL: Setiap operasi GD HARUS menggunakan:
 * - imagealphablending($img, false)
 * - imagesavealpha($img, true)
 */

class PngHandler
{
    private $errors = [];

    /**
     * Validate uploaded PNG file
     */
    public function validate($file)
    {
        $this->errors = [];

        // Check if file exists
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            $this->errors[] = 'File tidak valid';
            return false;
        }

        // Check file size
        if ($file['size'] > MAX_FILE_SIZE) {
            $this->errors[] = 'Ukuran file terlalu besar (max ' . (MAX_FILE_SIZE / 1024 / 1024) . 'MB)';
            return false;
        }

        // Check MIME type
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);

        if ($mimeType !== ALLOWED_MIME) {
            $this->errors[] = 'Hanya file PNG yang diizinkan (detected: ' . $mimeType . ')';
            return false;
        }

        // Check extension
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if ($ext !== 'png') {
            $this->errors[] = 'Ekstensi file harus .png';
            return false;
        }

        // Try to create image resource (validate if it's really a PNG)
        $img = @imagecreatefrompng($file['tmp_name']);
        if (!$img) {
            $this->errors[] = 'File PNG corrupt atau tidak valid';
            return false;
        }
        imagedestroy($img);

        return true;
    }

    /**
     * Process uploaded PNG
     * 
     * @param array $file - $_FILES array
     * @return array|false - ['original' => path, 'optimized' => path, 'thumb' => path, 'filename' => name]
     */
    public function process($file)
    {
        if (!$this->validate($file)) {
            return false;
        }

        // Generate unique filename
        $filename = $this->generateFilename($file['name']);
        
        $paths = [
            'original' => ORIGINALS_PATH . '/' . $filename,
            'optimized' => OPTIMIZED_PATH . '/' . $filename,
            'thumb' => THUMBS_PATH . '/' . $filename
        ];

        try {
            // 1. Save original
            if (!move_uploaded_file($file['tmp_name'], $paths['original'])) {
                throw new Exception('Gagal menyimpan file original');
            }

            // 2. Create optimized version (compressed but transparent)
            $this->optimize($paths['original'], $paths['optimized']);

            // 3. Create thumbnail (transparent)
            $this->createThumbnail($paths['original'], $paths['thumb']);

            return [
                'original' => $paths['original'],
                'optimized' => $paths['optimized'],
                'thumb' => $paths['thumb'],
                'filename' => $filename
            ];

        } catch (Exception $e) {
            // Cleanup on error
            foreach ($paths as $path) {
                if (file_exists($path)) {
                    unlink($path);
                }
            }
            $this->errors[] = $e->getMessage();
            return false;
        }
    }

    /**
     * Optimize PNG dengan compression, MEMPERTAHANKAN TRANSPARANSI
     * 
     * CRITICAL: Alpha channel HARUS dijaga!
     */
    public function optimize($sourcePath, $destPath)
    {
        // Load PNG
        $img = imagecreatefrompng($sourcePath);
        if (!$img) {
            throw new Exception('Gagal membaca PNG');
        }

        // === CRITICAL: PRESERVE ALPHA CHANNEL ===
        // Disable blending to preserve exact alpha values
        imagealphablending($img, false);
        
        // Enable saving alpha channel
        imagesavealpha($img, true);

        // Save with compression (0-9, higher = smaller file)
        $success = imagepng($img, $destPath, PNG_COMPRESSION);
        
        imagedestroy($img);

        if (!$success) {
            throw new Exception('Gagal menyimpan PNG optimized');
        }

        return true;
    }

    /**
     * Create thumbnail PNG dengan transparansi tetap terjaga
     */
    public function createThumbnail($sourcePath, $destPath, $size = THUMBNAIL_SIZE)
    {
        // Load original
        $source = imagecreatefrompng($sourcePath);
        if (!$source) {
            throw new Exception('Gagal membaca PNG untuk thumbnail');
        }

        $srcWidth = imagesx($source);
        $srcHeight = imagesy($source);

        // Calculate thumbnail dimensions (maintain aspect ratio)
        if ($srcWidth > $srcHeight) {
            $thumbWidth = $size;
            $thumbHeight = intval(($srcHeight / $srcWidth) * $size);
        } else {
            $thumbHeight = $size;
            $thumbWidth = intval(($srcWidth / $srcHeight) * $size);
        }

        // Create thumbnail canvas
        $thumb = imagecreatetruecolor($thumbWidth, $thumbHeight);
        
        // === CRITICAL: PRESERVE TRANSPARENCY IN THUMBNAIL ===
        // Disable blending
        imagealphablending($thumb, false);
        
        // Allocate transparent color
        $transparent = imagecolorallocatealpha($thumb, 0, 0, 0, 127);
        imagefill($thumb, 0, 0, $transparent);
        
        // Enable saving alpha
        imagesavealpha($thumb, true);

        // Resize (with transparency preserved)
        imagecopyresampled(
            $thumb, $source,
            0, 0, 0, 0,
            $thumbWidth, $thumbHeight,
            $srcWidth, $srcHeight
        );

        // Save thumbnail with compression
        $success = imagepng($thumb, $destPath, PNG_COMPRESSION);

        imagedestroy($source);
        imagedestroy($thumb);

        if (!$success) {
            throw new Exception('Gagal menyimpan thumbnail');
        }

        return true;
    }

    /**
     * Output PNG ke browser dengan transparansi utuh
     * 
     * @param string $path - Path to PNG file
     * @param bool $download - Force download or display
     */
    public function output($path, $download = false)
    {
        if (!file_exists($path)) {
            http_response_code(404);
            die('File not found');
        }

        // Clear any previous output
        if (ob_get_level()) {
            ob_clean();
        }

        // Set headers untuk PNG
        header('Content-Type: image/png');
        header('Content-Length: ' . filesize($path));
        
        if ($download) {
            $filename = basename($path);
            header('Content-Disposition: attachment; filename="' . $filename . '"');
        } else {
            header('Content-Disposition: inline');
        }

        // Cache headers (optional)
        header('Cache-Control: public, max-age=86400');
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 86400) . ' GMT');

        // Output file (binary safe)
        readfile($path);
        exit;
    }

    /**
     * Delete PNG and all its versions
     */
    public function delete($filename)
    {
        $paths = [
            ORIGINALS_PATH . '/' . $filename,
            OPTIMIZED_PATH . '/' . $filename,
            THUMBS_PATH . '/' . $filename
        ];

        $deleted = 0;
        foreach ($paths as $path) {
            if (file_exists($path)) {
                if (unlink($path)) {
                    $deleted++;
                }
            }
        }

        return $deleted > 0;
    }

    /**
     * Get all images
     */
    public function getAllImages()
    {
        $images = [];
        $files = glob(ORIGINALS_PATH . '/*.png');
        
        foreach ($files as $file) {
            $filename = basename($file);
            $images[] = [
                'filename' => $filename,
                'original' => $file,
                'optimized' => OPTIMIZED_PATH . '/' . $filename,
                'thumb' => THUMBS_PATH . '/' . $filename,
                'size' => filesize($file),
                'uploaded' => filemtime($file),
                'has_alpha' => $this->hasAlpha($file)
            ];
        }

        // Sort by upload time (newest first)
        usort($images, function($a, $b) {
            return $b['uploaded'] - $a['uploaded'];
        });

        return $images;
    }

    /**
     * Check if PNG has transparency (alpha channel)
     */
    public function hasAlpha($path)
    {
        $img = imagecreatefrompng($path);
        if (!$img) {
            return false;
        }

        // Check if image has alpha channel
        $hasAlpha = (imagecolorstotal($img) == 0) || 
                    (imagecolorsforindex($img, imagecolorat($img, 0, 0))['alpha'] < 127);

        imagedestroy($img);
        return $hasAlpha;
    }

    /**
     * Generate unique filename
     */
    private function generateFilename($originalName)
    {
        $ext = pathinfo($originalName, PATHINFO_EXTENSION);
        $name = pathinfo($originalName, PATHINFO_FILENAME);
        
        // Sanitize filename
        $name = preg_replace('/[^a-zA-Z0-9_-]/', '_', $name);
        $name = substr($name, 0, 50); // Limit length
        
        // Add timestamp untuk uniqueness
        return $name . '_' . time() . '.' . $ext;
    }

    /**
     * Get errors
     */
    public function getErrors()
    {
        return $this->errors;
    }

    /**
     * Get last error
     */
    public function getLastError()
    {
        return end($this->errors) ?: 'Unknown error';
    }

    /**
     * Rename PNG and all its versions
     * 
     * @param string $oldFilename - Current filename
     * @param string $newFilename - New filename (without extension)
     * @return bool|array - True on success, array with error on failure
     */
    public function rename($oldFilename, $newFilename)
    {
        // Validate old filename
        $oldFilename = basename($oldFilename);
        if (empty($oldFilename) || !preg_match('/^[a-zA-Z0-9_-]+\.png$/', $oldFilename)) {
            return ['error' => 'Invalid old filename'];
        }

        // Sanitize new filename
        $newFilename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $newFilename);
        $newFilename = substr($newFilename, 0, 50);
        $newFilename = trim($newFilename, '_-');
        
        if (empty($newFilename)) {
            return ['error' => 'New filename cannot be empty'];
        }

        // Add .png extension
        $newFilename = $newFilename . '.png';

        // Check if new filename already exists
        if (file_exists(ORIGINALS_PATH . '/' . $newFilename)) {
            return ['error' => 'File with this name already exists'];
        }

        $oldPaths = [
            'original' => ORIGINALS_PATH . '/' . $oldFilename,
            'optimized' => OPTIMIZED_PATH . '/' . $oldFilename,
            'thumb' => THUMBS_PATH . '/' . $oldFilename
        ];

        $newPaths = [
            'original' => ORIGINALS_PATH . '/' . $newFilename,
            'optimized' => OPTIMIZED_PATH . '/' . $newFilename,
            'thumb' => THUMBS_PATH . '/' . $newFilename
        ];

        // Check if all old files exist
        foreach ($oldPaths as $key => $path) {
            if (!file_exists($path)) {
                return ['error' => ucfirst($key) . ' file not found'];
            }
        }

        // Rename all files
        $renamed = 0;
        $errors = [];
        
        foreach ($oldPaths as $key => $oldPath) {
            if (rename($oldPath, $newPaths[$key])) {
                $renamed++;
            } else {
                $errors[] = "Failed to rename $key";
            }
        }

        // Rollback if not all files renamed
        if ($renamed < count($oldPaths)) {
            // Rollback successfully renamed files
            foreach ($newPaths as $key => $newPath) {
                if (file_exists($newPath)) {
                    rename($newPath, $oldPaths[$key]);
                }
            }
            return ['error' => 'Rename failed: ' . implode(', ', $errors)];
        }

        return true;
    }

    /**
     * Get paginated images
     * 
     * @param int $page - Current page (1-based)
     * @param int $perPage - Items per page
     * @return array - ['images' => [], 'total' => int, 'pages' => int, 'current_page' => int]
     */
    public function getPaginatedImages($page = 1, $perPage = 10)
    {
        $allImages = $this->getAllImages();
        $total = count($allImages);
        $pages = ceil($total / $perPage);
        
        // Validate page
        $page = max(1, min($page, $pages ?: 1));
        
        // Get slice
        $offset = ($page - 1) * $perPage;
        $images = array_slice($allImages, $offset, $perPage);
        
        return [
            'images' => $images,
            'total' => $total,
            'pages' => $pages,
            'current_page' => $page,
            'per_page' => $perPage,
            'has_prev' => $page > 1,
            'has_next' => $page < $pages
        ];
    }
}
