# 🆕 Update Guide - New Features v2.0

## New Features Added

✅ **Rename File** - Rename PNG files without re-uploading
✅ **Pagination** - 10 items per page with navigation
✅ **SweetAlert2** - Beautiful pop-up notifications and confirmations
✅ **Dark Mode** - Toggle between light and dark theme

---

## 📝 Files to Update/Add

### 1. **Update: includes/PngHandler.php**

Add these two methods to your existing `PngHandler.php` file:

**Location:** After the `delete()` method (around line 250)

**Add these methods:**
```php
/**
 * Rename PNG and all its versions
 */
public function rename($oldFilename, $newFilename)
{
    // Validate old filename
    $oldFilename = basename($oldFilename);
    if (empty($oldFilename) || !preg_match('/^[a-zA-Z0-9_-]+\.png$/', $oldFilename)) {
        return ['error' => 'Invalid old filename'];
    }

    // Sanitize new filename
    $newFilename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $newFilename);
    $newFilename = substr($newFilename, 0, 50);
    $newFilename = trim($newFilename, '_-');
    
    if (empty($newFilename)) {
        return ['error' => 'New filename cannot be empty'];
    }

    // Add .png extension
    $newFilename = $newFilename . '.png';

    // Check if new filename already exists
    if (file_exists(ORIGINALS_PATH . '/' . $newFilename)) {
        return ['error' => 'File with this name already exists'];
    }

    $oldPaths = [
        'original' => ORIGINALS_PATH . '/' . $oldFilename,
        'optimized' => OPTIMIZED_PATH . '/' . $oldFilename,
        'thumb' => THUMBS_PATH . '/' . $oldFilename
    ];

    $newPaths = [
        'original' => ORIGINALS_PATH . '/' . $newFilename,
        'optimized' => OPTIMIZED_PATH . '/' . $newFilename,
        'thumb' => THUMBS_PATH . '/' . $newFilename
    ];

    // Check if all old files exist
    foreach ($oldPaths as $key => $path) {
        if (!file_exists($path)) {
            return ['error' => ucfirst($key) . ' file not found'];
        }
    }

    // Rename all files
    $renamed = 0;
    $errors = [];
    
    foreach ($oldPaths as $key => $oldPath) {
        if (rename($oldPath, $newPaths[$key])) {
            $renamed++;
        } else {
            $errors[] = "Failed to rename $key";
        }
    }

    // Rollback if not all files renamed
    if ($renamed < count($oldPaths)) {
        // Rollback successfully renamed files
        foreach ($newPaths as $key => $newPath) {
            if (file_exists($newPath)) {
                rename($newPath, $oldPaths[$key]);
            }
        }
        return ['error' => 'Rename failed: ' . implode(', ', $errors)];
    }

    return true;
}

/**
 * Get paginated images
 */
public function getPaginatedImages($page = 1, $perPage = 10)
{
    $allImages = $this->getAllImages();
    $total = count($allImages);
    $pages = ceil($total / $perPage);
    
    // Validate page
    $page = max(1, min($page, $pages ?: 1));
    
    // Get slice
    $offset = ($page - 1) * $perPage;
    $images = array_slice($allImages, $offset, $perPage);
    
    return [
        'images' => $images,
        'total' => $total,
        'pages' => $pages,
        'current_page' => $page,
        'per_page' => $perPage,
        'has_prev' => $page > 1,
        'has_next' => $page < $pages
    ];
}
```

**OR:** Copy the complete method from `UPDATE_PngHandler.php` file

---

### 2. **REPLACE: public/index.php**

**Action:** Replace entire file with the new version

**New features:**
- Pagination system (10 items per page)
- Dark mode toggle (saved in localStorage)
- SweetAlert2 integration
- Rename button
- Delete confirmation with SweetAlert
- CSS variables for theming

**File location:** `public/index.php`

---

### 3. **REPLACE: public/admin/upload.php**

**Action:** Replace entire file with the new version

**New features:**
- SweetAlert success notification
- SweetAlert error notification
- Loading indicator during upload
- Redirect to gallery after success

**File location:** `public/admin/upload.php`

---

### 4. **REPLACE: public/admin/delete.php**

**Action:** Replace entire file with the new version

**New features:**
- Handle delete confirmation from SweetAlert
- Success/error notification
- Fallback confirmation for non-JS users

**File location:** `public/admin/delete.php`

---

### 5. **NEW FILE: public/admin/rename.php**

**Action:** Create new file

**Purpose:** Handle AJAX rename requests

**File location:** `public/admin/rename.php`

**Content:** Copy from the provided `rename.php` file

---

## 🚀 Installation Steps

### Option 1: Manual Update (Recommended for existing installations)

1. **Backup your current installation:**
   ```bash
   cp -r png-gallery png-gallery-backup
   ```

2. **Update PngHandler.php:**
   - Open `includes/PngHandler.php`
   - Scroll to end of class (before closing `}`)
   - Add the two new methods: `rename()` and `getPaginatedImages()`
   - Save file

3. **Replace index.php:**
   ```bash
   cp new-files/public/index.php png-gallery/public/index.php
   ```

4. **Replace upload.php:**
   ```bash
   cp new-files/public/admin/upload.php png-gallery/public/admin/upload.php
   ```

5. **Replace delete.php:**
   ```bash
   cp new-files/public/admin/delete.php png-gallery/public/admin/delete.php
   ```

6. **Add rename.php:**
   ```bash
   cp new-files/public/admin/rename.php png-gallery/public/admin/rename.php
   ```

7. **Test installation:**
   - Visit your gallery
   - Try dark mode toggle
   - Test pagination
   - Try renaming a file
   - Try deleting a file

---

### Option 2: Fresh Installation

If you prefer a fresh install:

1. **Backup your images:**
   ```bash
   cp -r png-gallery/storage/uploads png-gallery-backup/
   ```

2. **Remove old installation:**
   ```bash
   rm -rf png-gallery
   ```

3. **Extract new version:**
   ```bash
   unzip png-gallery-v2-system.zip
   ```

4. **Restore images:**
   ```bash
   cp -r png-gallery-backup/uploads/* png-gallery/storage/uploads/
   ```

5. **Update config:**
   - Edit `includes/config.php`
   - Set your password hash
   - Adjust other settings if needed

---

## 🎨 Feature Details

### 1. Dark Mode

**How it works:**
- Click moon/sun icon in header
- Theme preference saved in localStorage
- Persists across sessions
- CSS variables for easy customization

**Customization:**
Edit CSS variables in `public/index.php`:
```css
:root {
    --bg-primary: #f5f5f5;
    --text-primary: #333;
    /* ... more variables */
}

[data-theme="dark"] {
    --bg-primary: #1a1a1a;
    --text-primary: #e0e0e0;
    /* ... more variables */
}
```

### 2. Pagination

**Configuration:**
Change items per page in `public/index.php`:
```php
$perPage = 10; // Change this number
```

**Features:**
- First/Last page buttons
- Previous/Next buttons
- Current page indicator
- Page number display
- Responsive design

### 3. SweetAlert2

**Included notifications:**
- Upload success
- Upload error
- Delete confirmation
- Delete success
- Rename dialog
- Rename success
- Rename error
- Loading indicators

**Customization:**
```javascript
Swal.fire({
    icon: 'success',
    title: 'Your Title',
    text: 'Your message',
    confirmButtonColor: '#667eea'
});
```

### 4. Rename Function

**Features:**
- Real-time filename validation
- Duplicate name check
- AJAX request (no page reload)
- Renames all versions (original, optimized, thumb)
- Automatic rollback on failure

**Usage:**
Click "Rename" button → Enter new name → Confirm

---

## 🔧 Troubleshooting

### Issue: Rename button doesn't work

**Solution:**
1. Check if `public/admin/rename.php` exists
2. Check browser console for errors
3. Verify CSRF token is correct
4. Check file permissions on storage folders

### Issue: Dark mode not saving

**Solution:**
- Check browser localStorage
- Try different browser
- Clear cache and cookies

### Issue: Pagination shows wrong numbers

**Solution:**
- Clear browser cache
- Check `$perPage` value
- Verify `getPaginatedImages()` method is added

### Issue: SweetAlert not loading

**Solution:**
- Check internet connection (CDN required)
- Or download SweetAlert locally:
```html
<!-- Download from: https://github.com/sweetalert2/sweetalert2/releases -->
<link rel="stylesheet" href="assets/sweetalert2.min.css">
<script src="assets/sweetalert2.min.js"></script>
```

---

## 📊 Comparison: Old vs New

| Feature | v1.0 | v2.0 |
|---------|------|------|
| File Rename | ❌ No | ✅ Yes |
| Pagination | ❌ No | ✅ Yes (10/page) |
| Notifications | Basic HTML | ✅ SweetAlert2 |
| Dark Mode | ❌ No | ✅ Yes |
| Delete Confirm | Separate page | ✅ Modal popup |
| Theme Persistence | ❌ No | ✅ localStorage |
| Upload Feedback | Basic | ✅ Animated |

---

## 🎯 Testing Checklist

After installation, test:

- [ ] Dark mode toggle works
- [ ] Dark mode persists after refresh
- [ ] Pagination shows correct numbers
- [ ] Can navigate between pages
- [ ] Upload shows success notification
- [ ] Upload shows error for invalid files
- [ ] Rename dialog appears
- [ ] Can rename file successfully
- [ ] Rename validates duplicate names
- [ ] Delete asks for confirmation
- [ ] Delete shows success message
- [ ] All notifications are properly styled

---

## 📝 Notes

- **No database changes** - All changes are file-based
- **Backward compatible** - Existing images work without changes
- **CDN dependency** - SweetAlert2 loaded from CDN (requires internet)
- **Performance** - Pagination improves load time for large galleries
- **Mobile friendly** - All features work on mobile devices

---

## 🆘 Need Help?

1. **Check browser console** for JavaScript errors
2. **Check PHP error log** for server errors
3. **Verify file permissions** (755 for folders, 644 for files)
4. **Test each feature individually** to isolate issues
5. **Compare with backup** if something breaks

---

## 🎉 Enjoy Your Updated Gallery!

Your PNG Gallery now has:
- ✅ Professional notifications
- ✅ Better user experience
- ✅ Modern dark mode
- ✅ Efficient pagination
- ✅ File management (rename/delete)

All while keeping **PNG transparency 100% preserved**! 🖼️✨
