# Installation Guide - PNG Gallery Private

## 🎯 Overview

Sistem galeri private untuk menyimpan PNG transparan dengan fokus mempertahankan alpha channel (transparansi) di seluruh proses upload, kompresi, dan download.

## 📋 Requirements

### Server Requirements
- **PHP:** 7.4 atau lebih tinggi
- **Web Server:** Apache (dengan mod_rewrite) atau Nginx
- **PHP Extensions:**
  - GD Library (WAJIB - untuk image processing)
  - fileinfo (untuk MIME detection)
  - session (untuk authentication)

### Check Requirements

```bash
php -v  # Check PHP version
php -m | grep -i gd  # Check GD extension
php -m | grep -i fileinfo  # Check fileinfo extension
```

## 🚀 Installation Steps

### Step 1: Download/Clone Project

```bash
# Download project ke server Anda
cd /path/to/your/projects
git clone [repository-url] png-gallery
# atau extract zip file

cd png-gallery
```

### Step 2: Run Setup Script

```bash
php setup.php
```

Script ini akan:
- ✅ Membuat folder storage yang diperlukan
- ✅ Set permissions
- ✅ Check PHP extensions
- ✅ Membuat .gitkeep files

### Step 3: Configure Web Server

#### Apache Configuration

**Option A: VirtualHost (Recommended)**

Edit `/etc/apache2/sites-available/png-gallery.conf`:

```apache
<VirtualHost *:80>
    ServerName png-gallery.local
    DocumentRoot /path/to/png-gallery/public
    
    <Directory /path/to/png-gallery/public>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    # Blok akses ke storage folder
    <Directory /path/to/png-gallery/storage>
        Require all denied
    </Directory>
    
    ErrorLog ${APACHE_LOG_DIR}/png-gallery-error.log
    CustomLog ${APACHE_LOG_DIR}/png-gallery-access.log combined
</VirtualHost>
```

Enable site:
```bash
sudo a2ensite png-gallery
sudo systemctl reload apache2
```

**Option B: Subdirectory**

Jika install di subdirectory (e.g., `http://localhost/png-gallery/`):

1. Copy project ke `/var/www/html/png-gallery/`
2. Edit `includes/config.php`:
   ```php
   define('BASE_URL', '/png-gallery/public');
   ```
3. Pastikan `.htaccess` di `public/` aktif

#### Nginx Configuration

Edit `/etc/nginx/sites-available/png-gallery`:

```nginx
server {
    listen 80;
    server_name png-gallery.local;
    root /path/to/png-gallery/public;
    index index.php;

    # Block storage access
    location ^~ /storage {
        deny all;
        return 404;
    }

    # PHP processing
    location ~ \.php$ {
        include snippets/fastcgi-php.conf;
        fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }

    # Pretty URLs
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    # Security headers
    add_header X-Content-Type-Options "nosniff";
    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-XSS-Protection "1; mode=block";
}
```

Enable site:
```bash
sudo ln -s /etc/nginx/sites-available/png-gallery /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

### Step 4: Set Permissions

```bash
# Owner dan group (sesuaikan dengan web server user Anda)
sudo chown -R www-data:www-data /path/to/png-gallery

# Permissions
chmod 755 /path/to/png-gallery
chmod 755 /path/to/png-gallery/public
chmod 700 /path/to/png-gallery/storage
chmod 755 /path/to/png-gallery/storage/uploads
chmod 755 /path/to/png-gallery/storage/uploads/originals
chmod 755 /path/to/png-gallery/storage/uploads/optimized
chmod 755 /path/to/png-gallery/storage/thumbs
chmod 644 /path/to/png-gallery/public/.htaccess
chmod 644 /path/to/png-gallery/storage/.htaccess
```

**Important:** Folder `storage/` HARUS:
- Berada di LUAR document root, ATAU
- Dilindungi dengan `.htaccess` (sudah included)

### Step 5: Change Default Password

**CRITICAL:** Ganti password default sebelum production!

1. Generate password hash baru:
```bash
php -r "echo password_hash('password_baru_anda', PASSWORD_DEFAULT);"
```

2. Copy output (starts with `$2y$10$...`)

3. Edit `includes/config.php`:
```php
define('ADMIN_PASSWORD_HASH', '$2y$10$..your_hash_here..');
```

### Step 6: Test Installation

1. Buka browser: `http://png-gallery.local/login.php` (atau sesuai domain Anda)
2. Login dengan:
   - Username: `admin`
   - Password: `password` (atau password baru jika sudah diganti)
3. Jika berhasil, Anda akan redirect ke halaman galeri

## 🔧 Configuration

### Main Configuration File

Edit `includes/config.php` untuk customize:

```php
// Compression level (0-9, higher = smaller file)
define('PNG_COMPRESSION', 7);

// Thumbnail size in pixels
define('THUMBNAIL_SIZE', 200);

// Max file size (10MB default)
define('MAX_FILE_SIZE', 10 * 1024 * 1024);

// Session configuration
define('SESSION_NAME', 'PNG_GALLERY_SESSION');

// Base URL (jika di subdirectory)
define('BASE_URL', '/png-gallery/public');
```

### PHP Configuration

Edit `php.ini` atau `.htaccess`:

```ini
upload_max_filesize = 10M
post_max_size = 10M
max_execution_time = 60
memory_limit = 128M
```

## 🧪 Testing

### Test 1: Upload PNG Transparan

1. Buat atau download PNG dengan transparansi
2. Upload via halaman upload
3. Check hasil di galeri - harus terlihat checkerboard background

### Test 2: Verify Transparency

```bash
# Check jika PNG memiliki alpha channel
cd /path/to/png-gallery
php -r "
\$img = imagecreatefrompng('storage/uploads/originals/yourfile.png');
\$rgba = imagecolorat(\$img, 0, 0);
\$alpha = (\$rgba & 0x7F000000) >> 24;
echo \$alpha > 0 ? 'Has Alpha' : 'No Alpha';
"
```

### Test 3: Run Automated Tests

```bash
php test_all.php
```

Lihat `TESTING.md` untuk test cases lengkap.

## 🔒 Security Checklist

- [ ] Password default sudah diganti
- [ ] Folder `storage/` tidak bisa diakses via browser
- [ ] `.htaccess` sudah aktif (test: akses langsung `http://domain/storage/` harus 403)
- [ ] Session cookie httponly enabled
- [ ] HTTPS enabled (production)
- [ ] File permissions correct (755 untuk folder, 644 untuk file)

## 📁 Directory Structure

```
png-gallery/
├── public/              # WEB ROOT (point domain here)
│   ├── index.php       # Gallery view
│   ├── login.php       # Login page
│   ├── logout.php      # Logout
│   ├── image.php       # Secure image server
│   ├── .htaccess       # Apache config
│   ├── admin/
│   │   ├── upload.php  # Upload handler
│   │   └── delete.php  # Delete handler
│   └── assets/
│       └── style.css   # (future)
├── storage/            # PRIVATE (outside web root)
│   ├── .htaccess       # Deny all access
│   ├── uploads/
│   │   ├── originals/  # Original PNG files
│   │   └── optimized/  # Compressed versions
│   └── thumbs/         # Thumbnails
├── includes/           # PHP classes
│   ├── config.php      # Configuration
│   ├── auth.php        # Authentication
│   ├── csrf.php        # CSRF protection
│   └── PngHandler.php  # PNG processor (CRITICAL!)
├── setup.php           # Setup script
├── README.md           # Documentation
├── TESTING.md          # Test guide
└── .gitignore
```

## 🐛 Troubleshooting

### Issue: "GD library not found"

```bash
# Ubuntu/Debian
sudo apt-get install php-gd
sudo systemctl restart apache2

# CentOS/RHEL
sudo yum install php-gd
sudo systemctl restart httpd
```

### Issue: "Permission denied" saat upload

```bash
# Fix permissions
sudo chown -R www-data:www-data storage/
chmod -R 755 storage/
```

### Issue: Image tidak transparan setelah upload

1. Check GD version: `php -i | grep -i gd`
2. Verify code di `PngHandler.php`:
   - `imagealphablending($img, false)` harus ada
   - `imagesavealpha($img, true)` harus ada
3. Test dengan PNG sample yang proven transparan

### Issue: Cannot access storage folder

✅ **INI BENAR!** Storage folder TIDAK boleh diakses langsung.
Semua akses gambar HARUS via `image.php` dengan session check.

### Issue: 404 on image.php

Check `.htaccess`:
```bash
# Test if mod_rewrite enabled
apache2ctl -M | grep rewrite

# Enable if not
sudo a2enmod rewrite
sudo systemctl restart apache2
```

## 🔄 Updates

### Update to Latest Version

```bash
cd /path/to/png-gallery
git pull origin main
# Review CHANGELOG for breaking changes
```

### Backup Before Update

```bash
# Backup storage folder
tar -czf storage-backup-$(date +%Y%m%d).tar.gz storage/

# Backup config
cp includes/config.php includes/config.php.backup
```

## 📞 Support

Jika mengalami issues:

1. Check logs:
   - PHP error log: `/var/log/apache2/error.log` (Apache) atau `/var/log/nginx/error.log` (Nginx)
   - Application log: Check `error_log()` calls in code

2. Enable debugging:
   Edit `includes/config.php`:
   ```php
   error_reporting(E_ALL);
   ini_set('display_errors', 1);
   ```

3. Check requirements:
   ```bash
   php -v
   php -m | grep -E '(gd|fileinfo|session)'
   ```

## 🎉 Done!

Sistem siap digunakan. Upload PNG transparan Anda dan pastikan transparansi tetap terjaga!

**Default Login:**
- Username: `admin`
- Password: `password` (GANTI SECEPATNYA!)

**Important URLs:**
- Login: `http://your-domain/login.php`
- Gallery: `http://your-domain/index.php`
- Upload: `http://your-domain/admin/upload.php`
